# Self-Hosting Guide

This document explains how to deploy and run this Health App on your own server, completely independent of Lovable.

## Architecture Overview

```
┌─────────────────┐     ┌─────────────────┐     ┌─────────────────┐
│   Frontend      │────▶│   Express API   │────▶│   PostgreSQL    │
│   (React/Vite)  │     │   (Node.js)     │     │   Database      │
└─────────────────┘     └─────────────────┘     └─────────────────┘
                               │
                               ▼
                    ┌─────────────────┐
                    │   External APIs │
                    │  (Stripe, OpenAI)│
                    └─────────────────┘
```

## Deployment Options

### Option 1: Docker (Recommended)

1. **Clone and configure:**
   ```bash
   git clone <your-repo>
   cd server
   cp .env.example .env
   # Edit .env with your values
   ```

2. **Start with Docker Compose:**
   ```bash
   docker-compose up -d
   ```

3. **Build and deploy frontend:**
   ```bash
   cd ..
   npm run build
   # Deploy dist/ folder to your web server (nginx, Cloudflare Pages, etc.)
   ```

### Option 2: Manual Deployment

#### Backend Setup

1. **Install Node.js 18+** on your server

2. **Install and configure PostgreSQL:**
   ```bash
   sudo apt install postgresql postgresql-contrib
   sudo -u postgres createuser healthapp
   sudo -u postgres createdb healthapp -O healthapp
   sudo -u postgres psql -c "ALTER USER healthapp PASSWORD 'your-secure-password';"
   ```

3. **Run schema:**
   ```bash
   psql -U healthapp -d healthapp -f server/db/schema.sql
   ```

4. **Install and start server:**
   ```bash
   cd server
   npm install
   npm run build
   
   # Use PM2 or systemd for production
   npm install -g pm2
   pm2 start dist/index.js --name health-api
   ```

#### Frontend Setup

1. **Update API URL in `.env`:**
   ```
   VITE_API_URL=https://api.yourdomain.com/api
   ```

2. **Build:**
   ```bash
   npm run build
   ```

3. **Deploy `dist/` folder** to:
   - Nginx/Apache
   - Cloudflare Pages
   - Vercel
   - Netlify
   - Any static host

### Option 3: VPS/Cloud Provider

Popular options:
- **DigitalOcean Droplet** ($6/month)
- **Linode** ($5/month)
- **Hetzner** ($4/month)
- **AWS EC2** (free tier available)

## Environment Variables

### Backend (server/.env)

| Variable | Description | Required |
|----------|-------------|----------|
| `PORT` | Server port (default: 3001) | No |
| `NODE_ENV` | Environment (development/production) | Yes |
| `DATABASE_URL` | PostgreSQL connection string | Yes |
| `JWT_SECRET` | Secret for JWT signing (32+ chars) | Yes |
| `JWT_REFRESH_SECRET` | Secret for refresh tokens | Yes |
| `STRIPE_SECRET_KEY` | Stripe secret key | Yes |
| `OPENAI_API_KEY` | OpenAI API key (for AI features) | Yes |
| `FRONTEND_URL` | Frontend URL for CORS | Yes |

### Frontend (.env)

| Variable | Description | Required |
|----------|-------------|----------|
| `VITE_API_URL` | Backend API URL | Yes |

## SSL/HTTPS Setup

Use **Caddy** for automatic HTTPS:

```caddyfile
# Caddyfile
api.yourdomain.com {
    reverse_proxy localhost:3001
}

yourdomain.com {
    root * /var/www/health-app/dist
    file_server
    try_files {path} /index.html
}
```

Or use **nginx** with Let's Encrypt:

```nginx
server {
    listen 443 ssl;
    server_name api.yourdomain.com;
    
    ssl_certificate /etc/letsencrypt/live/api.yourdomain.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/api.yourdomain.com/privkey.pem;
    
    location / {
        proxy_pass http://localhost:3001;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
    }
}
```

## Stripe Configuration

1. Get your API keys from [Stripe Dashboard](https://dashboard.stripe.com/apikeys)
2. Create a product and price in Stripe
3. Update `SUBSCRIPTION_TIERS.premium.priceId` in `src/hooks/useSubscriptionApi.ts`
4. Set up Customer Portal: [Stripe Portal Settings](https://dashboard.stripe.com/settings/billing/portal)

## Database Backups

Automated backup script:

```bash
#!/bin/bash
# backup.sh
DATE=$(date +%Y%m%d_%H%M%S)
pg_dump -U healthapp healthapp > /backups/healthapp_$DATE.sql
# Keep last 7 days
find /backups -name "healthapp_*.sql" -mtime +7 -delete
```

Add to crontab:
```
0 2 * * * /path/to/backup.sh
```

## Monitoring

Recommended tools:
- **PM2** - Process management and monitoring
- **UptimeRobot** - Uptime monitoring (free)
- **Sentry** - Error tracking

## Security Checklist

- [ ] Use strong, unique JWT secrets (32+ characters)
- [ ] Enable HTTPS everywhere
- [ ] Set proper CORS origins
- [ ] Use environment variables for all secrets
- [ ] Enable rate limiting (already configured)
- [ ] Regular database backups
- [ ] Keep dependencies updated
- [ ] Use a firewall (ufw, iptables)

## Migrating from Lovable Cloud

If you have existing data in Lovable Cloud:

1. Export data using Cloud View
2. Transform data to match the new schema
3. Import into your PostgreSQL database

## Support

For issues with self-hosting:
- Check server logs: `pm2 logs health-api`
- Check database: `psql -U healthapp -d healthapp`
- Verify environment variables
- Test API endpoints with curl/Postman
