# Health App - Self-Hosted Backend

A complete self-hosted backend for the Health App, built with Node.js, Express, PostgreSQL, and Stripe.

## Prerequisites

- Node.js 18+ 
- PostgreSQL 14+
- Stripe account (for payments)
- OpenAI API key (for AI features)

## Quick Start

### 1. Install Dependencies

```bash
cd server
npm install
```

### 2. Set Up Database

Create a PostgreSQL database and run the schema:

```bash
# Create database
createdb healthapp

# Run schema
psql healthapp < db/schema.sql
```

### 3. Configure Environment

Copy `.env.example` to `.env` and fill in your values:

```bash
cp .env.example .env
```

Required environment variables:
- `DATABASE_URL` - PostgreSQL connection string
- `JWT_SECRET` - Random string for JWT signing (min 32 chars)
- `JWT_REFRESH_SECRET` - Random string for refresh tokens
- `STRIPE_SECRET_KEY` - Your Stripe secret key
- `OPENAI_API_KEY` - Your OpenAI API key
- `FRONTEND_URL` - URL where your frontend is hosted

### 4. Run the Server

Development:
```bash
npm run dev
```

Production:
```bash
npm run build
npm start
```

## API Endpoints

### Authentication
- `POST /api/auth/signup` - Create account
- `POST /api/auth/login` - Sign in
- `POST /api/auth/refresh` - Refresh tokens
- `GET /api/auth/me` - Get current user
- `POST /api/auth/logout` - Sign out

### Users
- `GET /api/users/profile` - Get profile
- `PATCH /api/users/profile` - Update profile
- `GET /api/users/health-profile` - Get health profile
- `PUT /api/users/health-profile` - Update health profile
- `GET /api/users/dietary-preferences` - Get dietary prefs
- `PUT /api/users/dietary-preferences` - Update dietary prefs

### Recipes
- `GET /api/recipes` - List recipes
- `POST /api/recipes` - Create recipe
- `GET /api/recipes/:id` - Get recipe
- `PATCH /api/recipes/:id` - Update recipe
- `DELETE /api/recipes/:id` - Delete recipe
- `GET /api/recipes/shared/:token` - Get shared recipe

### Meal Plans
- `GET /api/meal-plans` - List meal plans
- `POST /api/meal-plans` - Create meal plan
- `PATCH /api/meal-plans/:id` - Update meal plan
- `DELETE /api/meal-plans/:id` - Delete meal plan

### Scan History
- `GET /api/scan-history` - List scans
- `POST /api/scan-history` - Add scan
- `DELETE /api/scan-history/:id` - Delete scan
- `DELETE /api/scan-history` - Clear all

### Favorites
- `GET /api/favorites` - List favorites
- `POST /api/favorites` - Add favorite
- `DELETE /api/favorites/:itemType/:itemId` - Remove favorite

### Shopping List
- `GET /api/shopping-list` - List items
- `POST /api/shopping-list` - Add item
- `PATCH /api/shopping-list/:id` - Update item
- `DELETE /api/shopping-list/:id` - Remove item

### Community
- `GET /api/community/posts` - List posts
- `POST /api/community/posts` - Create post
- `POST /api/community/posts/:id/like` - Like/unlike
- `GET /api/community/posts/:id/replies` - Get replies
- `POST /api/community/posts/:id/replies` - Add reply

### Subscription
- `GET /api/subscription/status` - Check status
- `POST /api/subscription/create-checkout` - Create checkout
- `POST /api/subscription/customer-portal` - Open portal

### AI
- `POST /api/ai/analyze-ingredients` - Analyze ingredients
- `POST /api/ai/generate-recipes` - Generate recipes
- `POST /api/ai/analyze-phytoestrogens` - Phytoestrogen analysis
- `POST /api/ai/health-chat` - Chat with AI
- `POST /api/ai/analyze-nutrition` - Nutrition analysis
- `POST /api/ai/lookup-barcode` - Barcode lookup

## Docker Deployment

See `docker-compose.yml` for containerized deployment.

```bash
docker-compose up -d
```

## Security Notes

1. Change all secrets in production
2. Use HTTPS in production
3. Set up proper CORS origins
4. Enable rate limiting (already configured)
5. Use strong JWT secrets (32+ chars)

## Frontend Configuration

Update your frontend to point to your self-hosted backend:

```typescript
// src/lib/api.ts
const API_URL = 'https://your-server.com/api';
```
