-- Health App Database Schema
-- Run this on your PostgreSQL server to set up the database

-- Enable UUID extension
CREATE EXTENSION IF NOT EXISTS "pgcrypto";

-- Users table (replaces Supabase auth.users)
CREATE TABLE IF NOT EXISTS users (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    email VARCHAR(255) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    email_verified BOOLEAN DEFAULT false,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        updated_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_users_email ON users (email);

-- Profiles table
CREATE TABLE IF NOT EXISTS profiles (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID UNIQUE NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    display_name VARCHAR(255),
    avatar_url TEXT,
    bio TEXT,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        updated_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_profiles_user_id ON profiles (user_id);

-- User health profiles
CREATE TABLE IF NOT EXISTS user_health_profiles (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID UNIQUE NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    has_allergies BOOLEAN DEFAULT false,
    has_autoimmune BOOLEAN DEFAULT false,
    has_cardiovascular_issues BOOLEAN DEFAULT false,
    has_digestive_issues BOOLEAN DEFAULT false,
    has_hormonal_imbalance BOOLEAN DEFAULT false,
    has_skin_sensitivity BOOLEAN DEFAULT false,
    show_alerts BOOLEAN DEFAULT true,
    total_threshold INTEGER DEFAULT 100,
    toxin_threshold INTEGER DEFAULT 50,
    biological_threshold INTEGER DEFAULT 50,
    inflammatory_threshold INTEGER DEFAULT 50,
    oxidative_threshold INTEGER DEFAULT 50,
    nutrient_threshold INTEGER DEFAULT 50,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        updated_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

-- User dietary preferences
CREATE TABLE IF NOT EXISTS user_dietary_preferences (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID UNIQUE NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    halal_preference VARCHAR(50),
    kosher_preference VARCHAR(50),
    organic_preference VARCHAR(50),
    avoid_xenoestrogens BOOLEAN DEFAULT false,
    max_risk_level INTEGER DEFAULT 5,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        updated_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

-- Saved recipes
CREATE TABLE IF NOT EXISTS saved_recipes (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    title VARCHAR(255) NOT NULL,
    content TEXT NOT NULL,
    meal_type VARCHAR(50) DEFAULT 'any',
    servings INTEGER DEFAULT 4,
    nutritional_info JSONB,
    is_shared BOOLEAN DEFAULT false,
    share_token UUID,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_saved_recipes_user_id ON saved_recipes (user_id);

CREATE INDEX idx_saved_recipes_share_token ON saved_recipes (share_token);

-- Meal plans
CREATE TABLE IF NOT EXISTS meal_plans (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    recipe_id UUID NOT NULL REFERENCES saved_recipes (id) ON DELETE CASCADE,
    planned_date DATE NOT NULL,
    meal_slot VARCHAR(50) DEFAULT 'dinner',
    notes TEXT,
    reminder_time TIME,
    reminder_type VARCHAR(50), -- 'morning_before' | 'evening_before' | 'morning_of' for reminder job
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_meal_plans_user_id ON meal_plans (user_id);

CREATE INDEX idx_meal_plans_planned_date ON meal_plans (planned_date);

-- Optional: add reminder_type for meal reminder job (if column missing on existing DB)
-- ALTER TABLE meal_plans ADD COLUMN IF NOT EXISTS reminder_type VARCHAR(50);

-- Scan history
CREATE TABLE IF NOT EXISTS scan_history (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    barcode VARCHAR(100) NOT NULL,
    product_id VARCHAR(255) NOT NULL,
    product_name VARCHAR(255) NOT NULL,
    product_type VARCHAR(50) NOT NULL,
    image_url TEXT,
    product_details JSONB,
    scanned_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_scan_history_user_id ON scan_history (user_id);

CREATE INDEX idx_scan_history_scanned_at ON scan_history (scanned_at DESC);

-- User favorites
CREATE TABLE IF NOT EXISTS user_favorites (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    item_id VARCHAR(255) NOT NULL,
    item_type VARCHAR(50) NOT NULL,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        UNIQUE (user_id, item_id, item_type)
);

CREATE INDEX idx_user_favorites_user_id ON user_favorites (user_id);

-- Shopping list
CREATE TABLE IF NOT EXISTS shopping_list (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    item_id VARCHAR(255) NOT NULL,
    item_type VARCHAR(50) NOT NULL,
    quantity INTEGER DEFAULT 1,
    notes TEXT,
    is_purchased BOOLEAN DEFAULT false,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_shopping_list_user_id ON shopping_list (user_id);

-- Community posts
CREATE TABLE IF NOT EXISTS community_posts (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    author_name VARCHAR(255) NOT NULL,
    category VARCHAR(100) NOT NULL,
    content TEXT NOT NULL,
    likes_count INTEGER DEFAULT 0,
    replies_count INTEGER DEFAULT 0,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        updated_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_community_posts_category ON community_posts (category);

CREATE INDEX idx_community_posts_created_at ON community_posts (created_at DESC);

-- Community post likes
CREATE TABLE IF NOT EXISTS community_post_likes (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    post_id UUID NOT NULL REFERENCES community_posts (id) ON DELETE CASCADE,
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        UNIQUE (post_id, user_id)
);

-- Community post replies
CREATE TABLE IF NOT EXISTS community_post_replies (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    post_id UUID NOT NULL REFERENCES community_posts (id) ON DELETE CASCADE,
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    author_name VARCHAR(255) NOT NULL,
    content TEXT NOT NULL,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_community_post_replies_post_id ON community_post_replies (post_id);

-- User follows
CREATE TABLE IF NOT EXISTS user_follows (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    follower_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    following_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        UNIQUE (follower_id, following_id)
);

CREATE INDEX idx_user_follows_follower_id ON user_follows (follower_id);

CREATE INDEX idx_user_follows_following_id ON user_follows (following_id);

-- User achievements
CREATE TABLE IF NOT EXISTS user_achievements (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    achievement_id VARCHAR(100) NOT NULL,
    progress INTEGER DEFAULT 0,
    earned_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        UNIQUE (user_id, achievement_id)
);

-- Phytoestrogen exposure tracking
CREATE TABLE IF NOT EXISTS phytoestrogen_exposure (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    phytoestrogen_id VARCHAR(100) NOT NULL,
    phytoestrogen_name VARCHAR(255) NOT NULL,
    food_source VARCHAR(255) NOT NULL,
    potency VARCHAR(50) NOT NULL,
    servings INTEGER DEFAULT 1,
    notes TEXT,
    logged_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_phytoestrogen_exposure_user_id ON phytoestrogen_exposure (user_id);

CREATE INDEX idx_phytoestrogen_exposure_logged_at ON phytoestrogen_exposure (logged_at);

-- Phytoestrogen goals
CREATE TABLE IF NOT EXISTS phytoestrogen_goals (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID UNIQUE NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    daily_limit INTEGER DEFAULT 100,
    weekly_limit INTEGER DEFAULT 500,
    show_alerts BOOLEAN DEFAULT true,
    alert_at_percentage INTEGER DEFAULT 80,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        updated_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

-- Notifications
CREATE TABLE IF NOT EXISTS notifications (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    type VARCHAR(50) NOT NULL,
    title VARCHAR(255) NOT NULL,
    message TEXT NOT NULL,
    link TEXT,
    metadata JSONB,
    is_read BOOLEAN DEFAULT false,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_notifications_user_id ON notifications (user_id);

CREATE INDEX idx_notifications_is_read ON notifications (is_read);

-- Community products (user-contributed)
CREATE TABLE IF NOT EXISTS community_products (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    barcode VARCHAR(100) UNIQUE NOT NULL,
    product_name VARCHAR(255) NOT NULL,
    brand VARCHAR(255),
    product_type VARCHAR(50) DEFAULT 'food',
    ingredients TEXT,
    categories TEXT,
    image_url TEXT,
    contributed_by UUID REFERENCES users (id),
    verified_count INTEGER DEFAULT 0,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        updated_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_community_products_barcode ON community_products (barcode);

-- Conversations (messaging)
CREATE TABLE IF NOT EXISTS conversations (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    participant_1 UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    participant_2 UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    name VARCHAR(255),
    is_group BOOLEAN DEFAULT false,
    created_by UUID REFERENCES users (id),
    last_message_at TIMESTAMP
    WITH
        TIME ZONE,
        archived_by_1 BOOLEAN DEFAULT false,
        archived_by_2 BOOLEAN DEFAULT false,
        created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_conversations_participants ON conversations (participant_1, participant_2);

-- Messages
CREATE TABLE IF NOT EXISTS messages (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    conversation_id UUID NOT NULL REFERENCES conversations (id) ON DELETE CASCADE,
    sender_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    content TEXT NOT NULL,
    is_read BOOLEAN DEFAULT false,
    attachment_url TEXT,
    attachment_type VARCHAR(50),
    attachment_name VARCHAR(255),
    voice_url TEXT,
    voice_duration INTEGER,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW()
);

CREATE INDEX idx_messages_conversation_id ON messages (conversation_id);

CREATE INDEX idx_messages_created_at ON messages (created_at DESC);

-- Message reactions
CREATE TABLE IF NOT EXISTS message_reactions (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    message_id UUID NOT NULL REFERENCES messages (id) ON DELETE CASCADE,
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    emoji VARCHAR(10) NOT NULL,
    created_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        UNIQUE (message_id, user_id, emoji)
);

-- Conversation participants (for group chats)
CREATE TABLE IF NOT EXISTS conversation_participants (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid (),
    conversation_id UUID NOT NULL REFERENCES conversations (id) ON DELETE CASCADE,
    user_id UUID NOT NULL REFERENCES users (id) ON DELETE CASCADE,
    joined_at TIMESTAMP
    WITH
        TIME ZONE DEFAULT NOW(),
        left_at TIMESTAMP
    WITH
        TIME ZONE
);

-- Function to update updated_at timestamp
CREATE OR REPLACE FUNCTION update_updated_at_column()
RETURNS TRIGGER AS $$
BEGIN
  NEW.updated_at = NOW();
  RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Triggers for updated_at
CREATE TRIGGER update_users_updated_at BEFORE UPDATE ON users FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_profiles_updated_at BEFORE UPDATE ON profiles FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_user_health_profiles_updated_at BEFORE UPDATE ON user_health_profiles FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_user_dietary_preferences_updated_at BEFORE UPDATE ON user_dietary_preferences FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_community_posts_updated_at BEFORE UPDATE ON community_posts FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_phytoestrogen_goals_updated_at BEFORE UPDATE ON phytoestrogen_goals FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_community_products_updated_at BEFORE UPDATE ON community_products FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();