import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import rateLimit from 'express-rate-limit';
import dotenv from 'dotenv';
import path from 'path';

import authRoutes from './routes/auth';
import userRoutes from './routes/users';
import recipesRoutes from './routes/recipes';
import mealPlanRoutes from './routes/mealPlans';
import scanHistoryRoutes from './routes/scanHistory';
import favoritesRoutes from './routes/favorites';
import shoppingListRoutes from './routes/shoppingList';
import communityRoutes from './routes/community';
import subscriptionRoutes from './routes/subscription';
import aiRoutes from './routes/ai';
import phytoestrogenRoutes from './routes/phytoestrogen';
import achievementsRoutes from './routes/achievements';
import notificationsRoutes from './routes/notifications';
import messagingRoutes from './routes/messaging';
import { errorHandler } from './middleware/errorHandler';

// Load .env from server directory so OPENAI_API_KEY etc. are found when run from repo root
dotenv.config({ path: path.resolve(__dirname, '../.env') });

const app = express();
const PORT = process.env.PORT || 3001;

// Security middleware
app.use(helmet());
app.use(cors({
  origin: (origin, cb) => {
    const allowed = [
      process.env.FRONTEND_URL,
      'http://localhost:5173',
      'http://localhost:8080',
      'http://127.0.0.1:5173',
      'http://127.0.0.1:8080',
    ].filter(Boolean);
    if (!origin || allowed.some(u => origin === u)) {
      cb(null, true);
    } else {
      cb(null, allowed[0] || true);
    }
  },
  credentials: true,
}));

// Rate limiting (skip for auth so sign-in/sign-up always work)
const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 500, // limit each IP to 500 requests per 15 min
  message: { error: 'Too many requests, please try again later.' },
  skip: (req) => req.path.startsWith('/api/auth'),
});
app.use(limiter);

// Body parsing
app.use(express.json({ limit: '10mb' }));

// Static uploads (avatars)
app.use('/uploads', express.static('uploads'));

// Health check
app.get('/health', (req, res) => {
  res.json({ status: 'ok', timestamp: new Date().toISOString() });
});

// Routes
app.use('/api/auth', authRoutes);
app.use('/api/users', userRoutes);
app.use('/api/recipes', recipesRoutes);
app.use('/api/meal-plans', mealPlanRoutes);
app.use('/api/scan-history', scanHistoryRoutes);
app.use('/api/favorites', favoritesRoutes);
app.use('/api/shopping-list', shoppingListRoutes);
app.use('/api/community', communityRoutes);
app.use('/api/subscription', subscriptionRoutes);
app.use('/api/ai', aiRoutes);
app.use('/api/phytoestrogen', phytoestrogenRoutes);
app.use('/api/achievements', achievementsRoutes);
app.use('/api/notifications', notificationsRoutes);
app.use('/api/messaging', messagingRoutes);

// Error handling
app.use(errorHandler);

app.listen(PORT, () => {
  console.log(`🚀 Server running on http://localhost:${PORT}`);
});

export default app;
