import { Router, Response } from 'express';
import { query } from '../db';
import { authenticate, optionalAuth, AuthRequest } from '../middleware/auth';
import { v4 as uuidv4 } from 'uuid';

const router = Router();

// Get achievements for a user (query user_id for profile page, or own when authenticated)
router.get('/', optionalAuth, async (req: AuthRequest, res: Response) => {
  try {
    const userId = (req.query.user_id as string) || req.user?.id;
    if (!userId) {
      return res.status(400).json({ error: 'user_id required' });
    }
    const result = await query(
      'SELECT * FROM user_achievements WHERE user_id = $1 ORDER BY earned_at DESC',
      [userId]
    );
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Unlock / update achievement (authenticated user only)
router.post('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { achievement_id, progress = 0 } = req.body;
    const result = await query(
      `INSERT INTO user_achievements (id, user_id, achievement_id, progress, earned_at)
       VALUES ($1, $2, $3, $4, NOW())
       ON CONFLICT (user_id, achievement_id) DO UPDATE SET progress = GREATEST(user_achievements.progress, EXCLUDED.progress), earned_at = NOW()
       RETURNING *`,
      [uuidv4(), req.user!.id, achievement_id, progress]
    );
    res.status(201).json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

export default router;
