import { Router, Response } from 'express';
import { authenticate, optionalAuth, AuthRequest } from '../middleware/auth';

const router = Router();

const OPENAI_API_KEY = process.env.OPENAI_API_KEY;
const GEMINI_API_KEY = process.env.GEMINI_API_KEY;
const FIRECRAWL_API_KEY = process.env.FIRECRAWL_API_KEY;
const BARCODE_LOOKUP_API_KEY = process.env.BARCODE_LOOKUP_API_KEY;
const UPC_DATABASE_API_KEY = process.env.UPC_DATABASE_API_KEY;

const GEMINI_MODEL = process.env.GEMINI_MODEL || 'gemini-1.5-flash';

// Barcode/product identification model (default: gpt-4o for reliability). Set OPENAI_BARCODE_MODEL=gpt-5.2 if your account has access.
const BARCODE_AI_MODEL = process.env.OPENAI_BARCODE_MODEL || 'gpt-4o';
const BARCODE_AI_FALLBACK_MODEL = 'gpt-4o'; // used when primary model fails (e.g. not available on account)

function isReasoningModel(model: string): boolean {
  const m = (model || '').toLowerCase();
  return m.startsWith('o1') || m.startsWith('o3') || m.startsWith('o4') || m.startsWith('gpt-5') || m.includes('reasoning');
}

function isModelError(err: unknown): boolean {
  const msg = (err instanceof Error ? err.message : String(err)).toLowerCase();
  return (
    msg.includes('model') && (msg.includes('not found') || msg.includes('does not exist') || msg.includes('invalid') || msg.includes('404'))
  ) || msg.includes('404');
}

/** Call OpenAI for barcode flows; on model-unavailable errors, retry with gpt-4o so lookups keep working. */
async function callBarcodeOpenAI(
  messages: { role: string; content: string }[],
  options: { temperature?: number; max_tokens?: number; reasoning_effort?: 'low' | 'medium' | 'high' } = {}
): Promise<string> {
  try {
    return await callOpenAI(messages, BARCODE_AI_MODEL, options);
  } catch (e) {
    if (isModelError(e) && BARCODE_AI_MODEL !== BARCODE_AI_FALLBACK_MODEL) {
      console.warn('[barcode AI] Primary model failed, retrying with', BARCODE_AI_FALLBACK_MODEL, e);
      return await callOpenAI(messages, BARCODE_AI_FALLBACK_MODEL, {
        ...options,
        reasoning_effort: undefined, // gpt-4o doesn't use reasoning_effort
      });
    }
    throw e;
  }
}

/** Call Google Gemini API (REST). Returns response text or throws. */
async function callGemini(prompt: string, systemInstruction?: string): Promise<string> {
  if (!GEMINI_API_KEY || !GEMINI_API_KEY.trim()) {
    throw new Error('GEMINI_API_KEY is not set');
  }
  const url = `https://generativelanguage.googleapis.com/v1beta/models/${GEMINI_MODEL}:generateContent?key=${encodeURIComponent(GEMINI_API_KEY)}`;
  const body: Record<string, unknown> = {
    contents: [{ role: 'user', parts: [{ text: prompt }] }],
    generationConfig: {
      temperature: 0.2,
      maxOutputTokens: 8192,
    },
  };
  if (systemInstruction) {
    body.systemInstruction = { parts: [{ text: systemInstruction }] };
  }
  const res = await fetch(url, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(body),
  });
  if (!res.ok) {
    const errText = await res.text();
    throw new Error(`Gemini API error: ${res.status} ${errText}`);
  }
  const data = await res.json();
  const text = data?.candidates?.[0]?.content?.parts?.[0]?.text;
  if (text == null) {
    throw new Error('Gemini API returned no text');
  }
  return text;
}

// Product result shape for barcode lookup
interface BarcodeProduct {
  name: string;
  brand: string | null;
  categories: string | null;
  image_url: string | null;
  ingredients: string | null;
  nutrition?: {
    calories: number | null;
    fat: number | null;
    carbs: number | null;
    protein: number | null;
    fiber: number | null;
    sugar: number | null;
    salt: number | null;
  } | null;
  nutriscore_grade?: string | null;
  nova_group?: number | null;
}

// Dietary classification values (match food database / dashboard filters)
const HALAL_VALUES = ['halal', 'haram', 'mashbooh', 'not_applicable'] as const;
const KOSHER_VALUES = ['kosher', 'not_kosher', 'pareve', 'dairy', 'meat', 'not_applicable'] as const;
const ORGANIC_VALUES = ['organic', 'non_organic', 'varies', 'not_applicable'] as const;

interface AIAnalysis {
  summary: string;
  safetyRating: 'Safe' | 'Moderate Concern' | 'High Concern';
  keyIngredients: string[];
  concerns: string[];
  benefits: string[];
  suitableFor: string[];
  avoidIf: string[];
  alternatives: string[];
  sources?: string[];
  tobinScore?: number | null;
  glycogenicIndex?: number | null;
  lipogenicIndex?: number | null;
  nutriscoreGrade?: string | null;
  novaGroup?: number | null;
  halal?: string | null;
  kosher?: string | null;
  organic?: string | null;
  certificationNotes?: string | null;
}

interface LookupResult {
  found: boolean;
  product?: BarcodeProduct;
  source?: string;
  productType?: 'food' | 'cosmetic';
  aiGenerated?: boolean;
  aiAnalysis?: AIAnalysis | null;
  /** When AI was attempted but failed (e.g. API error) */
  aiError?: string;
  /** Raw model response text (for feeding into unified analysis) */
  rawResponse?: string | null;
}

// Generic AI completion endpoint (supports reasoning/thinking models: o1, o1-mini use reasoning_effort and max_completion_tokens)
async function callOpenAI(messages: any[], model = 'gpt-4o-mini', options: { temperature?: number; max_tokens?: number; reasoning_effort?: 'low' | 'medium' | 'high' } = {}) {
  if (!OPENAI_API_KEY || OPENAI_API_KEY === 'sk-your-openai-api-key') {
    throw new Error('OpenAI API key is not configured. Add OPENAI_API_KEY to server/.env');
  }
  const useReasoning = isReasoningModel(model);
  const maxTokens = options.max_tokens ?? 2000;
  const body: Record<string, unknown> = {
    model,
    messages,
  };
  if (useReasoning) {
    body.reasoning_effort = options.reasoning_effort ?? 'medium';
    body.max_completion_tokens = maxTokens;
  } else {
    body.temperature = options.temperature ?? 0.7;
    body.max_tokens = maxTokens;
  }
  const response = await fetch('https://api.openai.com/v1/chat/completions', {
    method: 'POST',
    headers: {
      'Authorization': `Bearer ${OPENAI_API_KEY}`,
      'Content-Type': 'application/json',
    },
    body: JSON.stringify(body),
  });

  if (!response.ok) {
    const errText = await response.text();
    throw new Error(`OpenAI API error: ${response.statusText}${errText ? ` - ${errText}` : ''}`);
  }

  const data = await response.json();
  return data.choices[0].message.content;
}

// Stream OpenAI response to Express res (for SSE)
function streamOpenAIToRes(
  res: Response,
  messages: any[],
  model = 'gpt-4o-mini',
  options: { temperature?: number; max_tokens?: number } = {}
) {
  if (!OPENAI_API_KEY || OPENAI_API_KEY === 'sk-your-openai-api-key') {
    res.status(500).json({ error: 'OpenAI API key is not configured' });
    return;
  }
  res.setHeader('Content-Type', 'text/event-stream');
  res.setHeader('Cache-Control', 'no-cache');
  res.setHeader('Connection', 'keep-alive');
  res.flushHeaders();

  fetch('https://api.openai.com/v1/chat/completions', {
    method: 'POST',
    headers: {
      'Authorization': `Bearer ${OPENAI_API_KEY}`,
      'Content-Type': 'application/json',
    },
    body: JSON.stringify({
      model,
      messages,
      stream: true,
      temperature: options.temperature ?? 0.7,
      max_tokens: options.max_tokens ?? 2000,
    }),
  })
    .then(async (streamRes) => {
      if (!streamRes.ok || !streamRes.body) {
        res.status(streamRes.status).end();
        return;
      }
      const reader = streamRes.body.getReader();
      const decoder = new TextDecoder();
      try {
        while (true) {
          const { done, value } = await reader.read();
          if (done) break;
          res.write(decoder.decode(value, { stream: true }));
          if (typeof (res as any).flush === 'function') (res as any).flush();
        }
      } finally {
        reader.releaseLock();
      }
      res.end();
    })
    .catch((err) => {
      console.error('Stream error:', err);
      if (!res.writableEnded) res.end();
    });
}

const USER_AGENT = 'MetabolicFoodScanner/1.0';

// Normalize categories to a string (API may return string or array)
function categoriesToString(categories: string | string[] | null | undefined): string {
  if (categories == null) return '';
  if (typeof categories === 'string') return categories;
  if (Array.isArray(categories)) return categories.join(', ');
  return '';
}

// Shared: detect if product name/categories indicate a cosmetic (so it goes to Cosmetics/Hazard dashboard)
// OFF may put lotion/cosmetic in name or _keywords in local script (e.g. Arabic لوشن = lotion); we merge _keywords into categories in OFF lookup
function looksLikeCosmetic(name: string, categories: string | string[] | null | undefined): boolean {
  const catStr = categoriesToString(categories);
  const combined = ((name || '') + ' ' + catStr).toLowerCase().replace(/-/g, ' ');
  return (
    /lotion|lotions|cream|shampoo|sunscreen|makeup|mascara|lipstick|perfume|deodorant|soap|skincare|haircare|cosmetic|beauty|nail polish|serum|moisturizer|moisturizers|cleanser|toner|exfoliant|face mask|body wash|hand cream|lip balm|hair dye|conditioner|styling|foundation|concealer|blush|eyeliner|eyeshadow|brow|spf|anti-aging|wrinkle|retinol|hyaluronic|paraben|fragrance|essential oil|body lotion|hair care|skin care|personal care|emulsion|dermatolog|topical|body care/.test(combined) ||
    /لوشن|كريم|شامبو|غسول|مرطب|مستحضر|عناية بالبشرة|عناية بالشعر|تجميل/.test(combined)  // Arabic: lotion, cream, shampoo, cleanser, moisturizer, cosmetic, skincare, haircare, beauty
  );
}

function normNutrition( nut: Record<string, unknown> | undefined ): BarcodeProduct['nutrition'] {
  if (!nut) return null;
  return {
    calories: nut['energy-kcal_100g'] != null ? Number(nut['energy-kcal_100g']) : (nut['energy_100g'] != null ? Math.round(Number(nut['energy_100g']) / 4.184) : null),
    protein: nut['proteins_100g'] != null ? Number(nut['proteins_100g']) : null,
    carbs: nut['carbohydrates_100g'] != null ? Number(nut['carbohydrates_100g']) : null,
    fat: nut['fat_100g'] != null ? Number(nut['fat_100g']) : null,
    fiber: nut['fiber_100g'] != null ? Number(nut['fiber_100g']) : null,
    sugar: nut['sugars_100g'] != null ? Number(nut['sugars_100g']) : null,
    salt: nut['salt_100g'] != null ? Number(nut['salt_100g']) : null,
  };
}

const FETCH_TIMEOUT_MS = 12000;
async function fetchWithTimeout(url: string, options: RequestInit = {}, timeoutMs = FETCH_TIMEOUT_MS): Promise<Response> {
  const ctrl = new AbortController();
  const t = setTimeout(() => ctrl.abort(), timeoutMs);
  try {
    const res = await fetch(url, { ...options, signal: ctrl.signal });
    clearTimeout(t);
    return res;
  } catch (e) {
    clearTimeout(t);
    throw e;
  }
}

async function lookupOpenFoodFacts(barcode: string): Promise<LookupResult> {
  const url = `https://world.openfoodfacts.org/api/v0/product/${barcode}.json`;
  for (let attempt = 0; attempt < 2; attempt++) {
    try {
      const res = await fetchWithTimeout(url, { headers: { 'User-Agent': USER_AGENT } });
      const data = await res.json();
    if (data.status !== 1 || !data.product) return { found: false };
    const p = data.product;
    // Include categories_tags and _keywords (e.g. en:body-lotions, or Arabic لوشن) for cosmetic classification
    const categoriesTags = Array.isArray(p.categories_tags) ? p.categories_tags.join(' ') : (p.categories_tags || '');
    const keywords = Array.isArray(p._keywords) ? p._keywords.join(' ') : (p._keywords || '');
    const categoriesForClassification = [p.categories, categoriesTags, keywords].filter(Boolean).join(' ') || null;
    return {
      found: true,
      source: 'openfoodfacts',
      productType: 'food',
      product: {
        name: p.product_name || p.brands || 'Unknown',
        brand: p.brands || null,
        categories: categoriesForClassification || p.categories || null,
        image_url: p.image_url || null,
        ingredients: p.ingredients_text || null,
        nutrition: normNutrition(p.nutriments),
        nutriscore_grade: p.nutriscore_grade ?? null,
        nova_group: p.nova_group ?? null,
      },
    };
    } catch (e) {
      if (attempt === 1) {
        console.error('Open Food Facts error:', e);
        return { found: false };
      }
    }
  }
  return { found: false };
}

async function lookupOpenBeautyFacts(barcode: string): Promise<LookupResult> {
  const url = `https://world.openbeautyfacts.org/api/v0/product/${barcode}.json`;
  for (let attempt = 0; attempt < 2; attempt++) {
    try {
      const res = await fetchWithTimeout(url, { headers: { 'User-Agent': USER_AGENT } });
      const data = await res.json();
    if (data.status !== 1 || !data.product) return { found: false };
    const p = data.product;
    return {
      found: true,
      source: 'openbeautyfacts',
      productType: 'cosmetic',
      product: {
        name: p.product_name || p.brands || 'Unknown',
        brand: p.brands || null,
        categories: p.categories || null,
        image_url: p.image_url || null,
        ingredients: p.ingredients_text || null,
      },
    };
    } catch (e) {
      if (attempt === 1) {
        console.error('Open Beauty Facts error:', e);
        return { found: false };
      }
    }
  }
  return { found: false };
}

async function lookupOpenProductsFacts(barcode: string): Promise<LookupResult> {
  const url = `https://world.openproductsfacts.org/api/v0/product/${barcode}.json`;
  for (let attempt = 0; attempt < 2; attempt++) {
    try {
      const res = await fetchWithTimeout(url, { headers: { 'User-Agent': USER_AGENT } });
      const data = await res.json();
    if (data.status !== 1 || !data.product) return { found: false };
    const p = data.product;
    const name = p.product_name || p.brands || '';
    const cat = p.categories || '';
    const isCosmetic = looksLikeCosmetic(name, cat);
    return {
      found: true,
      source: 'openproductsfacts',
      productType: isCosmetic ? 'cosmetic' : 'food',
      product: {
        name: p.product_name || p.brands || 'Unknown',
        brand: p.brands || null,
        categories: p.categories || null,
        image_url: p.image_url || null,
        ingredients: p.ingredients_text || null,
      },
    };
    } catch (e) {
      if (attempt === 1) {
        console.error('Open Products Facts error:', e);
        return { found: false };
      }
    }
  }
  return { found: false };
}

// Open Food Facts API v2 (alternative endpoint; sometimes has different coverage)
async function lookupOpenFoodFactsV2(barcode: string): Promise<LookupResult> {
  try {
    const res = await fetch(`https://world.openfoodfacts.org/api/v2/product/${barcode}.json?fields=product_name,brands,categories,image_url,ingredients_text,nutriments,nutriscore_grade,nova_group`, { headers: { 'User-Agent': USER_AGENT } });
    const data = await res.json();
    if (data.status !== 1 || !data.product) return { found: false };
    const p = data.product;
    return {
      found: true,
      source: 'openfoodfacts',
      productType: 'food',
      product: {
        name: p.product_name || p.brands || 'Unknown',
        brand: p.brands || null,
        categories: p.categories || null,
        image_url: p.image_url || null,
        ingredients: p.ingredients_text || null,
        nutrition: normNutrition(p.nutriments),
        nutriscore_grade: p.nutriscore_grade ?? null,
        nova_group: p.nova_group ?? null,
      },
    };
  } catch (e) {
    console.error('Open Food Facts v2 error:', e);
    return { found: false };
  }
}

// Barcode Lookup (barcodelookup.com) - optional; set BARCODE_LOOKUP_API_KEY
async function lookupBarcodeLookup(barcode: string): Promise<LookupResult> {
  if (!BARCODE_LOOKUP_API_KEY) return { found: false };
  try {
    const url = `https://api.barcodelookup.com/v3/products?barcode=${encodeURIComponent(barcode)}&formatted=y&key=${encodeURIComponent(BARCODE_LOOKUP_API_KEY)}`;
    const res = await fetch(url, { headers: { 'User-Agent': USER_AGENT } });
    const data = await res.json();
    const products = data.products;
    if (!Array.isArray(products) || products.length === 0) return { found: false };
    const p = products[0];
    const name = p.product_name || p.title || p.brand || 'Unknown';
    const cat = p.category || p.description || '';
    const isCosmetic = looksLikeCosmetic(name, cat);
    let nutrition: BarcodeProduct['nutrition'] = null;
    if (p.nutrition_facts || p.nutrition) {
      const n = p.nutrition_facts || p.nutrition;
      const num = (x: unknown) => (x != null && Number.isFinite(Number(x)) ? Number(x) : null);
      nutrition = {
        calories: num(n.calories ?? n.energy),
        protein: num(n.protein),
        carbs: num(n.carbohydrates ?? n.carbs),
        fat: num(n.fat),
        fiber: num(n.fiber),
        sugar: num(n.sugar),
        salt: num(n.salt ?? n.sodium),
      };
    }
    return {
      found: true,
      source: 'barcodelookup',
      productType: isCosmetic ? 'cosmetic' : 'food',
      product: {
        name: p.product_name || p.title || p.brand || 'Unknown',
        brand: p.brand || null,
        categories: p.category || null,
        image_url: Array.isArray(p.images) && p.images[0] ? p.images[0] : null,
        ingredients: p.ingredients || null,
        nutrition,
      },
    };
  } catch (e) {
    console.error('Barcode Lookup API error:', e);
    return { found: false };
  }
}

// UPC Database (upcdatabase.org) - optional; set UPC_DATABASE_API_KEY
async function lookupUPCDatabase(barcode: string): Promise<LookupResult> {
  if (!UPC_DATABASE_API_KEY) return { found: false };
  try {
    const res = await fetch(`https://api.upcdatabase.org/product/${barcode}`, {
      headers: { 'Authorization': `Bearer ${UPC_DATABASE_API_KEY}` },
    });
    const data = await res.json();
    if (data.success !== true || !data.title) return { found: false };
    const name = data.title || data.description || 'Unknown';
    const cat = data.category || '';
    const isCosmetic = looksLikeCosmetic(name, cat);
    return {
      found: true,
      source: 'upcdatabase',
      productType: isCosmetic ? 'cosmetic' : 'food',
      product: {
        name: data.title || data.description || 'Unknown',
        brand: data.brand || null,
        categories: data.category || null,
        image_url: data.image_url || null,
        ingredients: data.ingredients || null,
      },
    };
  } catch (e) {
    console.error('UPC Database error:', e);
    return { found: false };
  }
}

async function searchWebForProduct(barcode: string): Promise<string | null> {
  if (!FIRECRAWL_API_KEY) return null;
  try {
    const queries = [
      `"${barcode}" barcode product`,
      `${barcode} barcode product name ingredients`,
      `EAN ${barcode} product`,
      `"${barcode}" product nutrition`,
    ];
    let combined = '';
    for (const query of queries) {
      const res = await fetch('https://api.firecrawl.dev/v1/search', {
        method: 'POST',
        headers: { 'Authorization': `Bearer ${FIRECRAWL_API_KEY}`, 'Content-Type': 'application/json' },
        body: JSON.stringify({
          query,
          limit: 6,
          scrapeOptions: { formats: ['markdown'], onlyMainContent: true },
        }),
      });
      if (!res.ok) continue;
      const data = await res.json();
      if (!data.success || !data.data?.length) continue;
      const block = data.data.slice(0, 5).map((r: { url?: string; title?: string; markdown?: string; description?: string }) =>
        `Source: ${r.url}\nTitle: ${r.title || 'N/A'}\nContent: ${(r.markdown || r.description || 'N/A').substring(0, 4000)}`
      ).join('\n\n---\n\n');
      if (block) combined = combined ? `${combined}\n\n---\n\n${block}` : block;
    }
    if (combined) return combined;
    // Fallback: scrape multiple barcode lookup pages (different sites have different coverage, e.g. regional products)
    const scrapeUrls = [
      `https://www.barcodelookup.com/${barcode}`,
      `https://www.upcitemdb.com/upc/${barcode}`,
      `https://www.buycott.com/upc/${barcode}`,
    ];
    const scrapeResults = await Promise.allSettled(
      scrapeUrls.map((url) =>
        fetch('https://api.firecrawl.dev/v1/scrape', {
          method: 'POST',
          headers: { 'Authorization': `Bearer ${FIRECRAWL_API_KEY}`, 'Content-Type': 'application/json' },
          body: JSON.stringify({ url, formats: ['markdown'], onlyMainContent: true }),
        }).then((r) => r.ok ? r.json() : null)
      )
    );
    const parts: string[] = [];
    for (let i = 0; i < scrapeUrls.length; i++) {
      const s = scrapeResults[i];
      if (s.status === 'fulfilled' && s.value?.success && s.value?.data?.markdown) {
        const text = (s.value.data.markdown || '').substring(0, 5000);
        if (text.length > 50) parts.push(`Source: ${scrapeUrls[i]}\nContent: ${text}`);
      }
    }
    if (parts.length) return parts.join('\n\n---\n\n');
    return null;
  } catch (e) {
    console.error('Web search error:', e);
    return null;
  }
}

// Ask AI to classify product as food or cosmetic (any language, simple one-shot call)
async function classifyProductTypeWithAI(product: BarcodeProduct): Promise<'food' | 'cosmetic'> {
  const name = product.name || '';
  const categories = categoriesToString(product.categories);
  const text = `Product: ${name}\nCategories/keywords: ${categories || '(none)'}`.slice(0, 800);
  const content = await callOpenAI(
    [
      {
        role: 'system',
        content: 'You classify products into exactly one category. Reply with only one word: "food" or "cosmetic". Food = edible items, drinks, snacks. Cosmetic = lotions, creams, shampoo, makeup, skincare, haircare, perfume, soap, deodorant, sunscreen, personal care products applied to skin/hair/body. Use product name and categories (any language) to decide.',
      },
      { role: 'user', content: text },
    ],
    'gpt-4o-mini',
    { temperature: 0.1, max_tokens: 10 }
  );
  const answer = (content || '').toLowerCase().trim();
  if (answer.includes('cosmetic')) return 'cosmetic';
  return 'food';
}

function parseAIAnalysis(content: string): AIAnalysis | null {
  const jsonMatch = content.match(/\{[\s\S]*\}/);
  if (!jsonMatch) return null;
  try {
    const o = JSON.parse(jsonMatch[0]);
    const tobin = o.tobinScore;
    const gi = o.glycogenicIndex;
    const li = o.lipogenicIndex;
    const clamp = (n: number, min: number, max: number) => Math.max(min, Math.min(max, n));
    const nutriscore = o.nutriscoreGrade ?? o.nutriscore_grade;
    const nova = o.novaGroup ?? o.nova_group;
    const nutriscoreGrade = typeof nutriscore === 'string' && /^[a-eA-E]$/.test(nutriscore) ? nutriscore.toLowerCase() : null;
    const novaGroup = typeof nova === 'number' && Number.isFinite(nova) && nova >= 1 && nova <= 4 ? Math.round(nova) : null;
    const halalStr = o.halal != null ? String(o.halal).toLowerCase().replace(/-/g, '_') : '';
    const kosherStr = o.kosher != null ? String(o.kosher).toLowerCase().replace(/-/g, '_') : '';
    const organicStr = o.organic != null ? String(o.organic).toLowerCase().replace(/-/g, '_') : '';
    const halal = halalStr && HALAL_VALUES.includes(halalStr as typeof HALAL_VALUES[number]) ? halalStr : null;
    const kosher = kosherStr && KOSHER_VALUES.includes(kosherStr as typeof KOSHER_VALUES[number]) ? kosherStr : null;
    const organic = organicStr && ORGANIC_VALUES.includes(organicStr as typeof ORGANIC_VALUES[number]) ? organicStr : null;
    const certificationNotes = typeof o.certificationNotes === 'string' ? o.certificationNotes : null;
    return {
      summary: o.summary || '',
      safetyRating: ['Safe', 'Moderate Concern', 'High Concern'].includes(o.safetyRating) ? o.safetyRating : 'Moderate Concern',
      keyIngredients: Array.isArray(o.keyIngredients) ? o.keyIngredients : [],
      concerns: Array.isArray(o.concerns) ? o.concerns : [],
      benefits: Array.isArray(o.benefits) ? o.benefits : [],
      suitableFor: Array.isArray(o.suitableFor) ? o.suitableFor : [],
      avoidIf: Array.isArray(o.avoidIf) ? o.avoidIf : [],
      alternatives: Array.isArray(o.alternatives) ? o.alternatives : [],
      sources: Array.isArray(o.sources) ? o.sources : [],
      tobinScore: typeof tobin === 'number' && Number.isFinite(tobin) ? clamp(Math.round(tobin), 0, 50) : null,
      glycogenicIndex: typeof gi === 'number' && Number.isFinite(gi) ? clamp(gi, 0, 3) : null,
      lipogenicIndex: typeof li === 'number' && Number.isFinite(li) ? clamp(li, 0, 3) : null,
      nutriscoreGrade: nutriscoreGrade,
      novaGroup: novaGroup,
      halal: halal ?? null,
      kosher: kosher ?? null,
      organic: organic ?? null,
      certificationNotes: certificationNotes ?? null,
    };
  } catch {
    return null;
  }
}

async function analyzeProductWithAI(
  barcode: string,
  product: BarcodeProduct,
  productType: 'food' | 'cosmetic',
  webContext?: string | null
): Promise<AIAnalysis | null> {
  try {
    const web = webContext ?? (await searchWebForProduct(barcode));
    const webInfo = web
      ? `\n\n--- WEB SEARCH RESULTS (use this together with the database data above for a detailed analysis) ---\n${web}\n\nCombine the database product data and the web search results above to provide an accurate, up-to-date, detailed analysis.`
      : '';

    const isFood = productType === 'food';
    const systemPrompt = isFood
      ? `You are an expert nutritionist and food safety expert. You will receive: (1) product data from our database, and (2) optional web search results. Use BOTH sources to produce a detailed analysis.

When nutrition (per 100g) is provided, the app will calculate GI* and LI from formulas; when nutrition is missing you MUST provide numeric estimates from product name, category, and typical composition—do not leave tobinScore, glycogenicIndex, or lipogenicIndex null for food.
- **Glycogenic Index (GI*)** 0–3: typical 0.3–1.5. Low < 0.6, High > 1. E.g. potato chips/crisps → ~1.0–1.3; soda → ~1.2–1.5; plain meat → ~0.3–0.5.
- **Lipogenic Index (LI)** 0–3: typical 0.2–2. LI < 0.5 = fat-neutral; LI > 1 = fat storage. E.g. chips → ~1.0–1.5; nuts → ~0.6–1.0.
- **TOBIN Score** 0–50 (five categories 0–10 each). Lower is better. E.g. ultra-processed snacks → 25–40.

Provide a thorough, detailed analysis. Respond with valid JSON only. For food always include: nutriscoreGrade, novaGroup, tobinScore, glycogenicIndex, lipogenicIndex, and dietary: halal, kosher, organic (use values: halal/haram/mashbooh/not_applicable, kosher/not_kosher/pareve/dairy/meat/not_applicable, organic/non_organic/varies/not_applicable). Include certificationNotes when relevant (e.g. certification symbols, cross-contact).
{
  "summary": "3-5 sentence overview. Mention TOBIN, GI*, LI, Nutri-Score, NOVA, and Halal/Kosher when relevant.",
  "safetyRating": "Safe" | "Moderate Concern" | "High Concern",
  "keyIngredients": ["list 5-10 main ingredients with brief note if concerning or beneficial"],
  "concerns": ["list specific concerns with brief explanation for each"],
  "benefits": ["list benefits or positive aspects"],
  "suitableFor": ["who this product is good for"],
  "avoidIf": ["who should avoid or use with caution"],
  "alternatives": ["2-4 healthier or safer alternatives if concerns exist"],
  "sources": ["mention database (Open Food Facts) and any web sources used"],
  "nutriscoreGrade": "a"|"b"|"c"|"d"|"e",
  "novaGroup": 1|2|3|4,
  "tobinScore": number 0-50,
  "glycogenicIndex": number 0-3,
  "lipogenicIndex": number 0-3,
  "halal": "halal"|"haram"|"mashbooh"|"not_applicable",
  "kosher": "kosher"|"not_kosher"|"pareve"|"dairy"|"meat"|"not_applicable",
  "organic": "organic"|"non_organic"|"varies"|"not_applicable",
  "certificationNotes": "optional brief note on certifications or cross-contact"
}`
      : `You are a cosmetic and skincare safety expert. You will receive: (1) product data from our database, and (2) optional web search results. Use BOTH sources to produce a detailed analysis. For cosmetics you MUST provide TOBIN score (0-50, ingredient safety). GI* and LI are metabolic indices; for topical cosmetics use 0 or null (N/A). Provide numeric values:

Provide a thorough, detailed analysis. Respond with valid JSON only:
{
  "summary": "3-5 sentence safety profile covering ingredients, skin compatibility, and any regulatory or research notes. Be specific.",
  "safetyRating": "Safe" | "Moderate Concern" | "High Concern",
  "keyIngredients": ["list 5-10 main ingredients with brief safety or benefit note"],
  "concerns": ["list specific concerns with brief explanation"],
  "benefits": ["list benefits or positive aspects"],
  "suitableFor": ["skin types or demographics"],
  "avoidIf": ["conditions or sensitivities that should avoid"],
  "alternatives": ["2-4 safer or gentler alternatives if concerns exist"],
  "sources": ["mention database (Open Beauty Facts) and any web sources used"],
  "tobinScore": number 0-50 (TOBIN ingredient safety; always provide for cosmetics),
  "glycogenicIndex": 0 or null (N/A for topical; use 0),
  "lipogenicIndex": 0 or null (N/A for topical; use 0)
}`;

    const userContent = `PRODUCT DATA FROM DATABASE (barcode ${barcode}):\nNAME: ${product.name}\nBRAND: ${product.brand || 'Unknown'}\nCATEGORIES: ${product.categories || 'Unknown'}\nINGREDIENTS: ${product.ingredients || 'Not available'}${product.nutrition ? `\nNUTRITION (per 100g): ${JSON.stringify(product.nutrition)}` : ''}${webInfo}`;

    const content = await callBarcodeOpenAI(
      [{ role: 'system', content: systemPrompt }, { role: 'user', content: userContent }],
      { temperature: 0.2, max_tokens: 2200, reasoning_effort: 'high' }
    );
    return parseAIAnalysis(content);
  } catch (e) {
    console.error('AI analysis error:', e);
    return null;
  }
}

async function lookupWithAI(barcode: string, webContext?: string | null): Promise<LookupResult> {
  if (!OPENAI_API_KEY || OPENAI_API_KEY === 'sk-your-openai-api-key') {
    console.log('Barcode not in DBs; AI lookup skipped (OPENAI_API_KEY not set or placeholder).');
    return { found: false };
  }
  try {
    const web = webContext ?? (await searchWebForProduct(barcode));
    const hasWeb = !!(web && web.trim());
    const webSaysNotFound = hasWeb && /no product|not found|couldn't find|not in (our )?database|no (results|listings?)/i.test(web) && /\d{8,14}/.test(web);
    const lastResort = !hasWeb || webSaysNotFound;
    if (lastResort) {
      console.log('[lookupWithAI]', hasWeb && webSaysNotFound ? 'Web says no product; using AI knowledge as fallback.' : 'No web/database results; using AI as last resort.');
    }
    const webInfo = hasWeb
      ? webSaysNotFound
        ? `\n\n--- WEB SEARCH (no product listing found for this barcode) ---\n${web.substring(0, 3000)}\n\nWeb search did not find a listing for barcode ${barcode}. You may still use your knowledge to identify the product if you have information (e.g. regional/Vietnamese products for 893 prefix).`
        : `\n\n--- WEB SEARCH RESULTS (use this to identify the product and to provide a detailed analysis) ---\n${web}\n\nUse the web search results above together with the barcode to identify the product and provide a detailed safety/ingredient analysis.`
      : '';

    const barcodeHint = barcode.startsWith('893') ? ' (Barcode prefix 893 = Vietnam/GS1; consider Vietnamese products if you have knowledge.)' : '';
    const lastResortRules = lastResort
      ? `LAST RESORT MODE (no database or web results): Set "found": true when you have information from your training knowledge linking barcode ${barcode} to a product (name/brand). Include regional products where relevant.${barcodeHint} Do NOT invent a product or use a generic placeholder like "Product (barcode from Vietnam)" without a real name/brand. When you return found: true, include the full "analysis" object with nutriscoreGrade, novaGroup, tobinScore, glycogenicIndex, lipogenicIndex for food (estimate from typical composition). Set "found": false when you cannot identify a product for this barcode.`
      : `When web search results are provided: set "found": true when at least one result clearly mentions this barcode (${barcode}) or a product explicitly linked to it. Use any plausible product name/brand from the results; do not return a generic placeholder when you have real product info.`;

    const systemPrompt = `You are a product research expert. Your task is to IDENTIFY a product by its barcode (GTIN) and provide a detailed safety/ingredient analysis.

BARCODE: ${barcode} (length: ${String(barcode).length} digits). You must never invent or use a different barcode number in your response; always refer to this exact barcode ${barcode}.

${lastResortRules}

Always respond with valid JSON.
- Set "found": true when you can identify a product for barcode ${barcode}. Include "name", "brand", "productType" ("food" or "cosmetic"), "categories", "ingredients", and a detailed "analysis" object.
- Set "found": false when you cannot identify a product for this barcode.

The "analysis" object must be detailed.
For FOOD: always provide nutriscoreGrade, novaGroup, tobinScore, glycogenicIndex, lipogenicIndex, halal, kosher, organic (use halal: halal/haram/mashbooh/not_applicable; kosher: kosher/not_kosher/pareve/dairy/meat/not_applicable; organic: organic/non_organic/varies/not_applicable). Optional certificationNotes.
For COSMETIC: always provide tobinScore (0-50); glycogenicIndex, lipogenicIndex can be 0 or null.
"summary", "safetyRating", "keyIngredients", "concerns", "benefits", "suitableFor", "avoidIf", "alternatives", "sources"; for food also "halal", "kosher", "organic", "certificationNotes".

Example: {"found": true, "name": "...", "brand": "...", "productType": "food"|"cosmetic", "categories": "...", "ingredients": "...", "analysis": {...}}
When you cannot identify a product: {"found": false}`;

    const userPrompt = hasWeb
      ? `Identify this product by barcode ${barcode} and provide a detailed analysis.${webInfo}\n\nReturn JSON with found, name, brand, productType, categories, ingredients, and a detailed analysis object (including GI*, LI, TOBIN, Nutri-Score, NOVA for food). Use only the barcode ${barcode} in your response.`
      : `Identify this product by barcode ${barcode} (last resort: no database or web results). Return found: true when you can identify a product for this barcode. Provide full "analysis" with nutriscoreGrade, novaGroup, tobinScore, glycogenicIndex, lipogenicIndex, and for food also halal, kosher, organic (values: halal/haram/mashbooh/not_applicable, kosher/not_kosher/pareve/dairy/meat/not_applicable, organic/non_organic/varies/not_applicable), certificationNotes. Return JSON with found, name, brand, productType, categories, ingredients, and analysis.`;

    const content = await callBarcodeOpenAI(
      [{ role: 'system', content: systemPrompt }, { role: 'user', content: userPrompt }],
      { temperature: 0.2, max_tokens: 2000, reasoning_effort: 'high' }
    );
    const jsonMatch = content.match(/\{[\s\S]*\}/);
    if (!jsonMatch) {
      console.error('AI barcode lookup: no JSON in response');
      return { found: false, rawResponse: content };
    }
    const ai = JSON.parse(jsonMatch[0]);
    if (!ai.found || !ai.name) {
      console.log('AI barcode lookup: model returned found=false or no name');
      return { found: false, rawResponse: content };
    }
    const analysis = ai.analysis ? parseAIAnalysis(JSON.stringify(ai.analysis)) : null;
    const isFood = ai.productType === 'cosmetic' ? false : true;
    return {
      found: true,
      source: webContext ? 'AI + Web Research' : 'AI',
      productType: ai.productType === 'cosmetic' ? 'cosmetic' : 'food',
      aiGenerated: true,
      rawResponse: content,
      product: {
        name: ai.name,
        brand: ai.brand || null,
        categories: ai.categories || null,
        image_url: null,
        ingredients: ai.ingredients || null,
        nutriscore_grade: (isFood && analysis?.nutriscoreGrade) ? analysis.nutriscoreGrade : null,
        nova_group: (isFood && analysis?.novaGroup != null) ? analysis.novaGroup : null,
      },
      aiAnalysis: analysis,
    };
  } catch (e) {
    const msg = e instanceof Error ? e.message : String(e);
    console.error('AI lookup error:', msg);
    return { found: false, aiError: msg };
  }
}

/** Gemini barcode identification (same contract as lookupWithAI). Optional when GEMINI_API_KEY is set. */
async function lookupWithGemini(barcode: string, webContext?: string | null): Promise<LookupResult> {
  if (!GEMINI_API_KEY || !GEMINI_API_KEY.trim()) {
    return { found: false };
  }
  try {
    const web = webContext ?? (await searchWebForProduct(barcode));
    const hasWeb = !!(web && web.trim());
    const webSaysNotFound = hasWeb && /no product|not found|couldn't find|not in (our )?database|no (results|listings?)/i.test(web) && /\d{8,14}/.test(web);
    const webBlock = hasWeb
      ? webSaysNotFound
        ? `\n\n--- WEB (no listing found for this barcode) ---\n${web.substring(0, 2500)}\n\nYou may still use your knowledge to identify the product (e.g. 893 = Vietnam).`
        : `\n\n--- WEB SEARCH RESULTS ---\n${web}\n\nUse the above and the barcode to identify the product.`
      : '\n\nNo web results. Use your knowledge to identify the product if you have information (e.g. regional/Vietnamese for 893).';
    const systemInstruction = `You are a product research expert. Identify the product for barcode ${barcode}. Never invent or use a different barcode. Respond with valid JSON only. Set "found": true when you can identify a real product (from web or knowledge); include "name", "brand", "productType" ("food" or "cosmetic"), "categories", "ingredients", and "analysis" (object with summary, safetyRating, keyIngredients, concerns, benefits, suitableFor, avoidIf, alternatives, sources; for food also nutriscoreGrade, novaGroup, tobinScore, glycogenicIndex, lipogenicIndex, halal, kosher, organic, certificationNotes; for cosmetic also tobinScore 0-50, glycogenicIndex and lipogenicIndex can be 0 or null). Set "found": false when you cannot identify a product.`;
    const prompt = `Barcode: ${barcode}. Identify this product and provide a detailed analysis.${webBlock}\n\nReturn JSON: {"found": true|false, "name": "...", "brand": "...", "productType": "food"|"cosmetic", "categories": "...", "ingredients": "...", "analysis": {...}}`;
    const raw = await callGemini(prompt, systemInstruction);
    const jsonMatch = raw.match(/\{[\s\S]*\}/);
    if (!jsonMatch) return { found: false, rawResponse: raw };
    const ai = JSON.parse(jsonMatch[0]);
    if (!ai.found || !ai.name) return { found: false, rawResponse: raw };
    const analysis = ai.analysis ? parseAIAnalysis(JSON.stringify(ai.analysis)) : null;
    const isFood = ai.productType !== 'cosmetic';
    return {
      found: true,
      source: hasWeb ? 'Gemini + Web' : 'Gemini',
      productType: ai.productType === 'cosmetic' ? 'cosmetic' : 'food',
      aiGenerated: true,
      rawResponse: raw,
      product: {
        name: ai.name,
        brand: ai.brand || null,
        categories: ai.categories || null,
        image_url: null,
        ingredients: ai.ingredients || null,
        nutriscore_grade: (isFood && analysis?.nutriscoreGrade) ? analysis.nutriscoreGrade : null,
        nova_group: (isFood && analysis?.novaGroup != null) ? analysis.novaGroup : null,
      },
      aiAnalysis: analysis,
    };
  } catch (e) {
    console.error('Gemini lookup error:', e);
    return { found: false };
  }
}

// Gathered info from all sources for unified analysis
interface GatheredBarcodeInfo {
  barcode: string;
  databaseProduct: BarcodeProduct | null;
  databaseSource: string | null;
  webContext: string | null;
  openAIText: string | null;
  geminiText: string | null;
}

/** Single AI call that receives ALL gathered info and returns authoritative stats + detailed analysis. */
async function computeAnalysisFromGathered(
  gathered: GatheredBarcodeInfo,
  product: BarcodeProduct,
  productType: 'food' | 'cosmetic'
): Promise<AIAnalysis | null> {
  const parts: string[] = [];
  parts.push(`BARCODE: ${gathered.barcode}`);
  parts.push(`\n--- PRODUCT (canonical) ---\nName: ${product.name}\nBrand: ${product.brand ?? 'N/A'}\nCategories: ${product.categories ?? 'N/A'}\nIngredients: ${product.ingredients ?? 'N/A'}`);
  if (product.nutrition) parts.push(`\nNutrition (per 100g): ${JSON.stringify(product.nutrition)}`);
  if (gathered.databaseSource) parts.push(`\n(Database source: ${gathered.databaseSource})`);
  if (gathered.webContext && gathered.webContext.trim()) {
    parts.push(`\n--- WEB SEARCH RESULTS ---\n${gathered.webContext.substring(0, 12000)}`);
  }
  if (gathered.openAIText && gathered.openAIText.trim()) {
    parts.push(`\n--- OPENAI IDENTIFICATION / ANALYSIS ---\n${gathered.openAIText.substring(0, 4000)}`);
  }
  if (gathered.geminiText && gathered.geminiText.trim()) {
    parts.push(`\n--- GEMINI IDENTIFICATION / ANALYSIS ---\n${gathered.geminiText.substring(0, 4000)}`);
  }
  const systemPrompt = productType === 'food'
    ? `You are an expert nutritionist. You will receive: (1) the canonical product, (2) optional database source, (3) web search results, (4) OpenAI's identification/analysis, (5) Gemini's identification/analysis. Synthesize ALL sources into ONE authoritative analysis. Compute stats from all available data; when numbers conflict, use your best judgment. For food you MUST provide: nutriscoreGrade, novaGroup, tobinScore, glycogenicIndex, lipogenicIndex, halal, kosher, organic (halal: halal|haram|mashbooh|not_applicable; kosher: kosher|not_kosher|pareve|dairy|meat|not_applicable; organic: organic|non_organic|varies|not_applicable). Optional certificationNotes. Respond with valid JSON only:
{"summary":"...","safetyRating":"...","keyIngredients":[],"concerns":[],"benefits":[],"suitableFor":[],"avoidIf":[],"alternatives":[],"sources":[],"nutriscoreGrade":"a"|"b"|"c"|"d"|"e","novaGroup":1|2|3|4,"tobinScore":0-50,"glycogenicIndex":0-3,"lipogenicIndex":0-3,"halal":"...","kosher":"...","organic":"...","certificationNotes":"..."}`
    : `You are a cosmetic safety expert. Synthesize all provided sources (database, web, OpenAI, Gemini) into ONE authoritative analysis. For cosmetics you MUST provide tobinScore (0-50, ingredient safety). GI* and LI are N/A for topical products—use 0 or null. Respond with valid JSON only: summary, safetyRating, keyIngredients, concerns, benefits, suitableFor, avoidIf, alternatives, sources, tobinScore (number 0-50), glycogenicIndex (0 or null), lipogenicIndex (0 or null).`;
  const userContent = `Synthesize the following information and provide the single authoritative analysis for this product.\n\n${parts.join('\n')}\n\nReturn only valid JSON with the analysis object.`;
  try {
    const content = await callBarcodeOpenAI(
      [{ role: 'system', content: systemPrompt }, { role: 'user', content: userContent }],
      { temperature: 0.2, max_tokens: 2200, reasoning_effort: 'high' }
    );
    return parseAIAnalysis(content);
  } catch (e) {
    console.error('Unified analysis error:', e);
    return null;
  }
}

// TOBIN framework system prompt (from Supabase)
const TOBIN_SYSTEM_PROMPT = `You are an expert health and safety analyst specializing in cosmetic ingredients, food additives, and environmental toxins. Your role is to analyze ingredient lists using the TOBIN framework and provide comprehensive safety assessments.

## TOBIN SCORING FRAMEWORK
TOBIN stands for five categories. Score each from 0-10 (0 = no concern, 10 = severe concern):
**T - TOXINS** (heavy metals, PFAS, pesticides, phthalates, BPA, formaldehyde, coal tar)
**O - OXIDATIVE STRESS** (pro-oxidants, UV-reactive chemicals, rancidity-prone oils, inflammatory triggers)
**B - BIOLOGICAL/HORMONAL** (endocrine disruptors, xenoestrogens, thyroid disruptors, phytoestrogens)
**I - INFECTIONS/INFLAMMATORY** (allergens, barrier disruptors, microbiome disruptors, immunotoxins)
**N - NUTRIENT DEFICIENCIES** (mineral chelators, vitamin antagonists, absorption blockers)

## RESPONSE FORMAT
Provide in this structure:
### TOBIN SCORE SUMMARY
| Category | Score | Key Concerns |
| T - Toxins | X/10 | ... |
| O - Oxidative Stress | X/10 | ... |
| B - Biological/Hormonal | X/10 | ... |
| I - Inflammatory | X/10 | ... |
| N - Nutrient Depleting | X/10 | ... |
| **OVERALL TOBIN** | **X/50** | ... |

### DETAILED INGREDIENT ANALYSIS
For each concerning ingredient: name, TOBIN category, risk level, health concern, who should avoid.

### SAFE INGREDIENTS
List ingredients with no significant concerns.

### RECOMMENDATIONS
Immediate actions, safer alternatives, mitigation strategies, long-term considerations.

### SPECIAL WARNINGS
Banned in EU/other countries, FDA scrutiny, vulnerable populations.

Be scientific and evidence-based.`;

// Analyze ingredients (TOBIN framework; optional streaming)
router.post('/analyze-ingredients', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { ingredients, analysisType, productType, stream } = req.body;
    if (!ingredients || typeof ingredients !== 'string') {
      return res.status(400).json({ error: 'Ingredients string is required' });
    }
    const userPrompt = `Analyze the following ingredient list from a ${productType || 'cosmetic/personal care'} product using the TOBIN framework:

INGREDIENTS:
${ingredients}

Provide: 1) Complete TOBIN scores (T,O,B,I,N) with the summary table 2) Detailed analysis of each concerning ingredient 3) List of safe ingredients 4) Recommendations 5) Any special warnings for vulnerable populations. Be thorough and scientific.`;

    if (stream) {
      streamOpenAIToRes(
        res,
        [
          { role: 'system', content: TOBIN_SYSTEM_PROMPT },
          { role: 'user', content: userPrompt },
        ],
        'gpt-4o-mini',
        { temperature: 0.3, max_tokens: 2500 }
      );
      return;
    }

    const result = await callOpenAI(
      [
        { role: 'system', content: TOBIN_SYSTEM_PROMPT },
        { role: 'user', content: userPrompt },
      ],
      'gpt-4o-mini',
      { temperature: 0.3, max_tokens: 2500 }
    );
    res.json({ analysis: result });
  } catch (error) {
    console.error('Error analyzing ingredients:', error);
    throw error;
  }
});

// Analyze recipe nutrition (per serving) - Supabase analyze-nutrition shape
router.post('/analyze-recipe-nutrition', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { recipeContent, servings } = req.body;
    if (!recipeContent) {
      return res.status(400).json({ error: 'Recipe content is required' });
    }
    const systemPrompt = `You are a nutritional analysis expert. Analyze recipes and estimate nutritional information per serving. Use common nutritional databases. Return ONLY valid JSON in this exact format:
{
  "calories": 450,
  "protein": 35,
  "carbs": 25,
  "fat": 24,
  "fiber": 6,
  "sugar": 4,
  "sodium": 580,
  "highlights": ["High in omega-3", "Low glycemic", "Rich in antioxidants"],
  "warnings": []
}
All values per serving; calories in kcal; protein, carbs, fat, fiber, sugar in grams; sodium in mg. Be realistic.`;
    const content = await callOpenAI(
      [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: `Analyze this recipe for ${servings || 2} servings and provide nutritional information per serving:\n\n${recipeContent}` },
      ],
      'gpt-4o-mini',
      { temperature: 0.3, max_tokens: 800 }
    );
    const jsonMatch = content.match(/\{[\s\S]*\}/);
    if (!jsonMatch) {
      return res.status(500).json({ error: 'Failed to parse nutritional info' });
    }
    const nutritionalInfo = JSON.parse(jsonMatch[0]);
    res.json(nutritionalInfo);
  } catch (error) {
    console.error('Error analyzing recipe nutrition:', error);
    throw error;
  }
});

// Generate recipes (streaming; supports phytoestrogenFree and LI/GI prompts from Supabase)
const THERAPEUTIC_FOODS = `THERAPEUTIC FOODS (LI < 0.5, GI* < 0.6): Wild salmon, sardines, pasture-raised eggs, grass-fed beef liver, avocado, olive oil, coconut oil, broccoli, spinach, kale, cauliflower, Brussels sprouts, zucchini, bell peppers, mushrooms, garlic, ginger, turmeric, bone broth, grass-fed butter, almonds, walnuts, macadamia nuts, chia seeds, lemon, lime, berries, green tea.`;
const SAFE_FOODS = `SAFE FOODS: Chicken breast, turkey, grass-fed beef, lamb, shrimp, cod, tuna, Greek yogurt, cheese, cottage cheese, sweet potato, quinoa, lentils, black beans, chickpeas. Avoid: white bread, pasta, sugar, seed oils, processed meats.`;

router.post('/generate-recipes', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { ingredients, preferences, mealType, servings, dietaryRestrictions, phytoestrogenFree } = req.body;
    const stream = true;

    if (phytoestrogenFree) {
      const systemPrompt = `You are a hormone-health chef. Create recipes 100% FREE of phytoestrogens and xenoestrogens. NEVER use: soy, tofu, tempeh, edamame, soy milk, legumes/beans, chickpeas, lentils, flaxseeds, sesame/tahini, oats, beer/hops. Use ONLY: wild salmon, sardines, eggs, grass-fed beef/liver, lamb, chicken, turkey, shrimp, cod, tuna, avocado, olive oil, coconut oil, grass-fed butter, broccoli, cauliflower, Brussels sprouts, cabbage, kale, spinach, asparagus, zucchini, bell peppers, mushrooms, garlic, ginger, turmeric, bone broth, macadamia nuts, pumpkin seeds, lemons, berries (small amounts), green tea, apple cider vinegar. Emphasize cruciferous vegetables. Create flavorful meals that don't feel restrictive.`;
      const userPrompt = `Create a complete ${mealType || 'daily'} PHYTOESTROGEN-FREE meal plan with ${servings || 2} servings per recipe. ${preferences ? `Preferences: ${preferences}` : ''} For each meal provide: recipe name, "Hormone-Safe" badge, description, prep/cook time, ingredient list (ONLY approved ingredients), step-by-step instructions, Anti-Estrogenic Highlights, chef's tips. Include breakfast, lunch, dinner, and 1-2 snacks.`;
      if (stream) {
        streamOpenAIToRes(res, [{ role: 'system', content: systemPrompt }, { role: 'user', content: userPrompt }], 'gpt-4o-mini', { temperature: 0.6, max_tokens: 2500 });
        return;
      }
    }

    const systemPrompt = `You are a metabolic health chef. Create delicious recipes using low Lipogenic Index (LI) and low Glycemic Index (GI*) foods. Focus on THERAPEUTIC and SAFE foods with LI < 0.5 and GI* < 0.6. Avoid refined sugars, white flour, processed foods. ${THERAPEUTIC_FOODS} ${SAFE_FOODS}`;
    const userPrompt = `Create a complete ${mealType || 'daily'} meal plan with ${servings || 2} servings per recipe. ${dietaryRestrictions ? `Dietary restrictions: ${dietaryRestrictions}` : ''} ${preferences ? `Preferences: ${preferences}` : ''} For each meal provide: recipe name, description, prep/cook time, ingredient list, step-by-step instructions, nutritional highlights, chef's tips. Include breakfast, lunch, dinner, and 1-2 snacks.`;
    if (stream) {
      streamOpenAIToRes(res, [{ role: 'system', content: systemPrompt }, { role: 'user', content: userPrompt }], 'gpt-4o-mini', { temperature: 0.6, max_tokens: 2500 });
      return;
    }

    const result = await callOpenAI([{ role: 'system', content: systemPrompt }, { role: 'user', content: userPrompt }], 'gpt-4o-mini', { temperature: 0.6, max_tokens: 2500 });
    res.json({ recipe: result });
  } catch (error) {
    console.error('Error generating recipes:', error);
    throw error;
  }
});

// Phytoestrogen keyword lists (from Supabase)
const PHYTO_STRONG = ['soy', 'soya', 'soja', 'tofu', 'tempeh', 'edamame', 'miso', 'natto', 'soy protein', 'soy lecithin', 'soybean', 'soy flour', 'soy milk', 'flaxseed', 'flax seed', 'linseed', 'red clover', 'kudzu', 'pueraria'];
const PHYTO_MODERATE = ['chickpea', 'garbanzo', 'hummus', 'lentil', 'sesame', 'tahini', 'oat', 'avoine', 'barley', 'rye', 'wheat bran', 'licorice', 'hops', 'beer', 'fennel', 'fenouil', 'anise', 'alfalfa', 'clover', 'fenugreek'];
const PHYTO_WEAK = ['sunflower', 'pumpkin seed', 'walnut', 'pecan', 'pistachio', 'olive oil', 'broccoli', 'cabbage', 'kale', 'brussels', 'berries', 'strawberry', 'raspberry', 'blueberry', 'cranberry', 'apple', 'pear', 'cherry', 'plum', 'coffee', 'tea', 'wine', 'grape', 'raisin'];

function quickPhytoAnalyze(ingredients: string): { detected: boolean; overallRisk: 'none' | 'low' | 'moderate' | 'high'; compounds: { name: string; source: string; potency: string }[]; summary: string; recommendation: string } {
  const lower = ingredients.toLowerCase();
  const compounds: { name: string; source: string; potency: string }[] = [];
  for (const k of PHYTO_STRONG) {
    if (lower.includes(k)) {
      compounds.push({ name: k.includes('soy') ? 'Isoflavones (Genistein, Daidzein)' : k.includes('flax') ? 'Lignans' : 'Isoflavones', source: k.charAt(0).toUpperCase() + k.slice(1), potency: 'strong' });
    }
  }
  for (const k of PHYTO_MODERATE) {
    if (lower.includes(k)) {
      compounds.push({ name: 'Isoflavones/Lignans', source: k.charAt(0).toUpperCase() + k.slice(1), potency: 'moderate' });
    }
  }
  for (const k of PHYTO_WEAK) {
    if (lower.includes(k)) {
      compounds.push({ name: 'Trace Lignans/Flavonoids', source: k.charAt(0).toUpperCase() + k.slice(1), potency: 'weak' });
    }
  }
  const unique = compounds.filter((c, i, a) => a.findIndex(x => x.source.toLowerCase() === c.source.toLowerCase()) === i);
  const hasStrong = unique.some(c => c.potency === 'strong');
  const hasModerate = unique.some(c => c.potency === 'moderate');
  const overallRisk: 'none' | 'low' | 'moderate' | 'high' = hasStrong ? 'high' : hasModerate ? 'moderate' : unique.length > 0 ? 'low' : 'none';
  return {
    detected: unique.length > 0,
    overallRisk,
    compounds: unique.slice(0, 5),
    summary: unique.length === 0 ? 'No significant phytoestrogen sources detected.' : `Found ${unique.length} potential phytoestrogen source(s).`,
    recommendation: overallRisk === 'high' ? 'Consider limiting intake if avoiding estrogenic compounds.' : overallRisk === 'moderate' ? 'Monitor consumption if sensitive.' : overallRisk === 'low' ? 'Generally low concern.' : 'No phytoestrogen concerns identified.',
  };
}

// Analyze phytoestrogens (keyword quick analysis + optional AI enhancement - Supabase behavior)
router.post('/analyze-phytoestrogens', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { ingredients, productName, useAI } = req.body;
    if (!ingredients) {
      return res.status(400).json({ error: 'Ingredients list is required' });
    }
    const quickResult = quickPhytoAnalyze(ingredients);

    if (useAI && quickResult.detected && OPENAI_API_KEY && OPENAI_API_KEY !== 'sk-your-openai-api-key') {
      try {
        const aiContent = await callOpenAI(
          [
            { role: 'system', content: 'You are an expert in phytoestrogens and endocrine-affecting compounds in food. Be concise and factual. Focus on isoflavones (soy), lignans (flax, sesame), and other estrogenic compounds.' },
            { role: 'user', content: `Analyze these ingredients for phytoestrogen content: "${ingredients}". Product: ${productName || 'Unknown'}. Provide a brief (2-3 sentence) health-focused summary about phytoestrogen exposure from this product.` },
          ],
          'gpt-4o-mini',
          { max_tokens: 200 }
        );
        if (aiContent) quickResult.summary = aiContent;
      } catch (e) {
        console.error('Phyto AI enhancement failed:', e);
      }
    }

    res.json(quickResult);
  } catch (error) {
    console.error('Error analyzing phytoestrogens:', error);
    throw error;
  }
});

// Health chat (optional auth; Supabase prompt + optional streaming)
const HEALTH_CHAT_SYSTEM = `You are a knowledgeable, evidence-based health and nutrition advisor specializing in metabolic health, weight management, and chronic disease prevention.

Your expertise includes: Lipogenic Index (LI) and Glycogenic Index (GI*) for food analysis, insulin biology, AMPK/mTOR, mitochondrial health, de novo lipogenesis, practical dietary strategies.

CRITICAL: 1) Provide evidence-based information; if uncertain say so. 2) NEVER diagnose; recommend consulting healthcare professionals. 3) Reference LI and GI* when relevant. 4) Use measured, scientific language.

KEY PRINCIPLES: Weight gain is driven by insulin-driven fat storage; high LI foods promote fat storage; high GI* foods spike insulin; protein + fiber + healthy fat = optimal meal structure; avoid "naked carbohydrates." Format with bullet points, **bold** for key terms, clear sections.`;

router.post('/health-chat', optionalAuth, async (req: AuthRequest, res: Response) => {
  try {
    const { message, conversationHistory, stream } = req.body;
    const messages = [
      { role: 'system', content: HEALTH_CHAT_SYSTEM },
      ...(Array.isArray(conversationHistory) ? conversationHistory : []),
      { role: 'user', content: message || '' },
    ];

    if (stream) {
      streamOpenAIToRes(res, messages, 'gpt-4o-mini', { temperature: 0.6, max_tokens: 1500 });
      return;
    }

    const result = await callOpenAI(messages, 'gpt-4o-mini', { temperature: 0.6, max_tokens: 1500 });
    res.json({ response: result });
  } catch (error) {
    console.error('Error in health chat:', error);
    const err = error instanceof Error ? error : new Error('Unknown error');
    res.status(500).json({ error: err.message });
  }
});

// Analyze nutrition
router.post('/analyze-nutrition', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { ingredients, servingSize } = req.body;

    const systemPrompt = `You are a certified nutritionist. Analyze the nutritional content of food items.
    Return JSON with: calories, protein_g, carbs_g, fat_g, fiber_g, sugar_g, sodium_mg, vitamins (object), minerals (object), health_score (1-100).`;

    const result = await callOpenAI([
      { role: 'system', content: systemPrompt },
      { role: 'user', content: `Analyze nutrition for ${servingSize || '1 serving'} of: ${ingredients}` }
    ]);

    res.json({ nutrition: result });
  } catch (error) {
    console.error('Error analyzing nutrition:', error);
    throw error;
  }
});

// Extract ingredients from recipes (for grocery list; optional useAI for AI extraction like Supabase)
router.post('/extract-ingredients', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { recipes, useAI } = req.body as { recipes: { title: string; content: string; multiplier?: number }[]; useAI?: boolean };
    if (!recipes || !Array.isArray(recipes)) {
      return res.status(400).json({ error: 'recipes array required' });
    }

    if (useAI && OPENAI_API_KEY && OPENAI_API_KEY !== 'sk-your-openai-api-key') {
      const combined = recipes.map((r: { title: string; content: string }) => `=== ${r.title} ===\n${r.content}`).join('\n\n');
      const systemPrompt = `You are a helpful assistant that extracts ingredients from recipes and creates organized shopping lists. Parse the recipes, combine duplicate ingredients and sum quantities, organize by category: Produce, Proteins, Dairy, Pantry, Spices & Seasonings, Oils & Fats, Other. Use consistent units. Return ONLY valid JSON: { "categories": [ { "name": "Produce", "items": [ { "name": "Avocados", "quantity": "4", "unit": "medium" } ] } ], "summary": { "totalItems": 15, "recipesProcessed": 2 } }`;
      const content = await callOpenAI(
        [{ role: 'system', content: systemPrompt }, { role: 'user', content: `Extract ingredients from these recipes:\n\n${combined}` }],
        'gpt-4o-mini',
        { temperature: 0.3, max_tokens: 2000 }
      );
      const jsonMatch = content.match(/\{[\s\S]*\}/);
      if (jsonMatch) {
        const parsed = JSON.parse(jsonMatch[0]);
        return res.json(parsed);
      }
    }

    const categories: { name: string; items: { name: string; quantity: string; unit: string }[] }[] = [];
    const seen = new Set<string>();
    for (const r of recipes) {
      const content = (r.content || '').replace(/\r\n/g, '\n');
      const multiplier = r.multiplier || 1;
      const lines = content.split('\n').filter(Boolean);
      for (const line of lines) {
        const trimmed = line.replace(/^[\s\-*•·]+/, '').trim();
        if (!trimmed || trimmed.length < 2) continue;
        const qtyMatch = trimmed.match(/^([\d¼½¾⅓⅔⅛⅜\/\.\-\s]+)\s+(\w+)\s+(.+)$/i)
          || trimmed.match(/^(.+?)\s*[-–—]\s*(.+)$/);
        let name = trimmed;
        let quantity = '1';
        let unit = '';
        if (qtyMatch) {
          quantity = (qtyMatch[1] || '1').trim();
          unit = (qtyMatch[2] || '').trim();
          name = (qtyMatch[3] || trimmed).trim();
        }
        const key = `${name.toLowerCase()}`;
        if (seen.has(key)) continue;
        seen.add(key);
        const num = parseFloat(quantity.replace(/[¼½¾]/g, (m: string) => ({ '¼': '0.25', '½': '0.5', '¾': '0.75' }[m] || m)));
        const scaled = Number.isNaN(num) ? quantity : (num * multiplier).toString();
        const cat = categories.find(c => c.name === 'Ingredients') || (() => {
          const c = { name: 'Ingredients', items: [] as { name: string; quantity: string; unit: string }[] };
          categories.push(c);
          return c;
        })();
        cat.items.push({ name, quantity: scaled, unit: unit || 'item' });
      }
    }
    if (categories.length === 0) categories.push({ name: 'Ingredients', items: [] });
    res.json({
      categories,
      summary: { totalItems: categories.reduce((s, c) => s + c.items.length, 0), recipesProcessed: recipes.length },
    });
  } catch (error) {
    console.error('Error extracting ingredients:', error);
    throw error;
  }
});

// Lookup barcode: DB (OFF/OBF/OPF) + Firecrawl (when key set) in parallel, then AI for detailed analysis
router.post('/lookup-barcode', authenticate, async (req: AuthRequest, res: Response) => {
  const barcode = req.body?.barcode;
  if (!barcode) {
    return res.status(400).json({ error: 'barcode is required', found: false });
  }
  const skipAI = req.body?.skipAI;

  try {
    const extraSources = [BARCODE_LOOKUP_API_KEY && 'BarcodeLookup', UPC_DATABASE_API_KEY && 'UPC Database'].filter(Boolean);
    console.log('[lookup-barcode]', barcode, '– OFF, OBF, OPF, OFFv2' + (extraSources.length ? `, ${extraSources.join(', ')}` : '') + (FIRECRAWL_API_KEY ? ', Firecrawl' : ''));

    // Use allSettled so one failing lookup (e.g. network) doesn't break the whole request
    const [foodSettled, beautySettled, productsSettled, foodV2Settled, barcodeLookupSettled, upcDbSettled, webSettled] = await Promise.allSettled([
      lookupOpenFoodFacts(barcode),
      lookupOpenBeautyFacts(barcode),
      lookupOpenProductsFacts(barcode),
      lookupOpenFoodFactsV2(barcode),
      BARCODE_LOOKUP_API_KEY ? lookupBarcodeLookup(barcode) : Promise.resolve({ found: false } as LookupResult),
      UPC_DATABASE_API_KEY ? lookupUPCDatabase(barcode) : Promise.resolve({ found: false } as LookupResult),
      FIRECRAWL_API_KEY ? searchWebForProduct(barcode) : Promise.resolve(null),
    ]);

    const foodResult: LookupResult = foodSettled.status === 'fulfilled' ? foodSettled.value : { found: false };
    const beautyResult: LookupResult = beautySettled.status === 'fulfilled' ? beautySettled.value : { found: false };
    const productsResult: LookupResult = productsSettled.status === 'fulfilled' ? productsSettled.value : { found: false };
    const foodV2Result: LookupResult = foodV2Settled.status === 'fulfilled' ? foodV2Settled.value : { found: false };
    const barcodeLookupResult: LookupResult = barcodeLookupSettled.status === 'fulfilled' ? barcodeLookupSettled.value : { found: false };
    const upcDbResult: LookupResult = upcDbSettled.status === 'fulfilled' ? upcDbSettled.value : { found: false };
    const webContext = webSettled.status === 'fulfilled' ? webSettled.value : null;
    if (foodSettled.status === 'rejected') console.error('[lookup-barcode] OFF rejected:', foodSettled.reason);
    if (beautySettled.status === 'rejected') console.error('[lookup-barcode] OBF rejected:', beautySettled.reason);
    if (productsSettled.status === 'rejected') console.error('[lookup-barcode] OPF rejected:', productsSettled.reason);
    if (foodV2Settled.status === 'rejected') console.error('[lookup-barcode] OFF v2 rejected:', foodV2Settled.reason);
    if (barcodeLookupSettled.status === 'rejected') console.error('[lookup-barcode] BarcodeLookup rejected:', barcodeLookupSettled.reason);
    if (upcDbSettled.status === 'rejected') console.error('[lookup-barcode] UPC Database rejected:', upcDbSettled.reason);
    if (webSettled.status === 'rejected') console.error('[lookup-barcode] Firecrawl rejected:', webSettled.reason);

    let result: LookupResult = { found: false };
    if (beautyResult.found) result = beautyResult;
    else if (foodResult.found) result = foodResult;
    else if (foodV2Result.found) result = foodV2Result;
    else if (productsResult.found) result = productsResult;
    else if (barcodeLookupResult.found) result = barcodeLookupResult;
    else if (upcDbResult.found) result = upcDbResult;

    let openAIResult: LookupResult = { found: false };
    let geminiResult: LookupResult = { found: false };
    if (!result.found && !skipAI) {
      const hasOpenAI = !!(OPENAI_API_KEY && OPENAI_API_KEY !== 'sk-your-openai-api-key');
      const hasGemini = !!(GEMINI_API_KEY && GEMINI_API_KEY.trim());
      if (hasOpenAI || hasGemini) {
        console.log('Barcode not in DBs, trying AI identification (OpenAI + Gemini in parallel)...');
        const [openAISettled, geminiSettled] = await Promise.allSettled([
          hasOpenAI ? lookupWithAI(barcode, webContext ?? undefined) : Promise.resolve({ found: false } as LookupResult),
          hasGemini ? lookupWithGemini(barcode, webContext ?? undefined) : Promise.resolve({ found: false } as LookupResult),
        ]);
        if (openAISettled.status === 'rejected') console.error('[lookup-barcode] OpenAI identification rejected:', openAISettled.reason);
        if (geminiSettled.status === 'rejected') console.error('[lookup-barcode] Gemini identification rejected:', geminiSettled.reason);
        openAIResult = openAISettled.status === 'fulfilled' ? openAISettled.value : { found: false };
        geminiResult = geminiSettled.status === 'fulfilled' ? geminiSettled.value : { found: false };
        if (openAIResult.found) result = openAIResult;
        else if (geminiResult.found) result = geminiResult;
        else if (openAIResult.aiError) result.aiError = openAIResult.aiError;
        else if (geminiResult.aiError && !result.aiError) result.aiError = geminiResult.aiError;
      } else {
        console.log('Barcode not in DBs; AI lookup skipped (add OPENAI_API_KEY or GEMINI_API_KEY).');
      }
    }

    if (!result.found) {
      const aiAvailable = !!(OPENAI_API_KEY && OPENAI_API_KEY !== 'sk-your-openai-api-key') || !!(GEMINI_API_KEY && GEMINI_API_KEY.trim());
      const errorMsg = result.aiError
        ? `AI identification failed: ${result.aiError}`
        : !aiAvailable
          ? 'Product not in databases. Add OPENAI_API_KEY or GEMINI_API_KEY in server/.env to enable AI identification.'
          : 'This barcode wasn\'t found in Open Food Facts, Open Beauty Facts, or our community database. AI (OpenAI/Gemini) couldn\'t identify it—you can add it below to help others.';
      console.log('[lookup-barcode]', barcode, '– not found' + (aiAvailable ? ' (OpenAI + Gemini tried)' : ' (no AI key)'));
      return res.json({
        found: false,
        error: errorMsg,
      });
    }

    // Use AI to categorize as food or cosmetic (works in any language). Fallback to keyword match if no key or AI fails.
    if (result.found && result.product) {
      const hasOpenAI = !!(OPENAI_API_KEY && OPENAI_API_KEY !== 'sk-your-openai-api-key');
      if (hasOpenAI && !skipAI) {
        try {
          result.productType = await classifyProductTypeWithAI(result.product);
        } catch (err) {
          console.warn('[lookup-barcode] AI classification failed, using keyword fallback:', err);
          result.productType = looksLikeCosmetic(result.product.name || '', result.product.categories) ? 'cosmetic' : 'food';
        }
      } else {
        result.productType = looksLikeCosmetic(result.product.name || '', result.product.categories) ? 'cosmetic' : 'food';
      }
    }

    // Build gathered info from all sources (DB, web, OpenAI, Gemini) and run unified analysis to compute stats + detailed analysis
    if (result.found && result.product && !skipAI && OPENAI_API_KEY && OPENAI_API_KEY !== 'sk-your-openai-api-key') {
      try {
        const fromDb = !result.aiGenerated;
        const gathered: GatheredBarcodeInfo = {
          barcode,
          databaseProduct: fromDb ? result.product : null,
          databaseSource: fromDb ? (result.source || null) : null,
          webContext: webContext ?? null,
          openAIText: openAIResult.rawResponse ?? null,
          geminiText: geminiResult.rawResponse ?? null,
        };
        const aiAnalysis = await computeAnalysisFromGathered(gathered, result.product, result.productType || 'food');
        if (aiAnalysis) {
          result.aiAnalysis = aiAnalysis;
          const sources: string[] = [];
          if (gathered.databaseSource) sources.push(gathered.databaseSource);
          if (gathered.webContext) sources.push('Web');
          if (gathered.openAIText) sources.push('OpenAI');
          if (gathered.geminiText) sources.push('Gemini');
          result.source = (result.source || '') + ' + Unified Analysis (' + (sources.length ? sources.join(', ') : 'AI') + ')';
        } else {
          const fallback = await analyzeProductWithAI(barcode, result.product, result.productType || 'food', webContext ?? undefined);
          if (fallback) {
            result.aiAnalysis = fallback;
            result.source = (result.source || '') + (webContext ? ' + AI & Web Analysis' : ' + AI Analysis');
          }
        }
      } catch (analysisErr) {
        console.error('Unified analysis failed (still returning product):', analysisErr);
        try {
          const fallback = await analyzeProductWithAI(barcode, result.product!, result.productType || 'food', webContext ?? undefined);
          if (fallback) {
            result.aiAnalysis = fallback;
            result.source = (result.source || '') + ' + AI Analysis';
          }
        } catch (_) {}
      }
    }

    // For AI-generated or AI-analysed products, fill Nutri-Score and NOVA from AI when product is missing them
    if (result.product && result.aiAnalysis) {
      if (result.product.nutriscore_grade == null && result.aiAnalysis.nutriscoreGrade != null) {
        result.product.nutriscore_grade = result.aiAnalysis.nutriscoreGrade;
      }
      if (result.product.nova_group == null && result.aiAnalysis.novaGroup != null) {
        result.product.nova_group = result.aiAnalysis.novaGroup;
      }
    }

    console.log('[lookup-barcode]', barcode, '– returning product:', result.source, result.productType);
    res.json({
      found: true,
      source: result.source,
      productType: result.productType,
      product: result.product,
      aiGenerated: result.aiGenerated || false,
      aiAnalysis: result.aiAnalysis || null,
    });
  } catch (error) {
    console.error('[lookup-barcode] Error:', error);
    res.status(500).json({ found: false, error: 'Failed to lookup barcode' });
  }
});

export default router;
