import { Router, Response } from 'express';
import { query } from '../db';
import { authenticate, optionalAuth, AuthRequest } from '../middleware/auth';
import { v4 as uuidv4 } from 'uuid';

const router = Router();

// Get community posts
router.get('/posts', optionalAuth, async (req: AuthRequest, res: Response) => {
  try {
    const { category, limit = 20, offset = 0, user_id, user_ids } = req.query;

    let sql = 'SELECT * FROM community_posts';
    const params: any[] = [];
    const conditions: string[] = [];

    if (category && category !== 'all') {
      conditions.push(`category = $${params.length + 1}`);
      params.push(category);
    }
    if (user_id) {
      conditions.push(`user_id = $${params.length + 1}`);
      params.push(user_id);
    }
    if (user_ids && typeof user_ids === 'string') {
      const ids = user_ids.split(',').map((s: string) => s.trim()).filter(Boolean);
      if (ids.length > 0) {
        conditions.push(`user_id = ANY($${params.length + 1}::uuid[])`);
        params.push(ids);
      }
    }
    if (conditions.length) sql += ' WHERE ' + conditions.join(' AND ');

    sql += ` ORDER BY created_at DESC LIMIT $${params.length + 1} OFFSET $${params.length + 2}`;
    params.push(limit, offset);

    const result = await query(sql, params);
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Create post
router.post('/posts', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { content, category, author_name } = req.body;

    const result = await query(
      `INSERT INTO community_posts (id, user_id, content, category, author_name, likes_count, replies_count, created_at, updated_at)
       VALUES ($1, $2, $3, $4, $5, 0, 0, NOW(), NOW())
       RETURNING *`,
      [uuidv4(), req.user!.id, content, category, author_name]
    );

    res.status(201).json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Like/unlike post
router.post('/posts/:id/like', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const postId = req.params.id;

    // Check if already liked
    const existing = await query(
      'SELECT id FROM community_post_likes WHERE post_id = $1 AND user_id = $2',
      [postId, req.user!.id]
    );

    if (existing.rows.length > 0) {
      // Unlike
      await query('DELETE FROM community_post_likes WHERE post_id = $1 AND user_id = $2', [postId, req.user!.id]);
      await query('UPDATE community_posts SET likes_count = GREATEST(0, likes_count - 1) WHERE id = $1', [postId]);
      res.json({ liked: false });
    } else {
      // Like
      await query(
        'INSERT INTO community_post_likes (id, post_id, user_id, created_at) VALUES ($1, $2, $3, NOW())',
        [uuidv4(), postId, req.user!.id]
      );
      await query('UPDATE community_posts SET likes_count = likes_count + 1 WHERE id = $1', [postId]);
      res.json({ liked: true });
    }
  } catch (error) {
    throw error;
  }
});

// Get replies for a post
router.get('/posts/:id/replies', async (req, res: Response) => {
  try {
    const result = await query(
      'SELECT * FROM community_post_replies WHERE post_id = $1 ORDER BY created_at ASC',
      [req.params.id]
    );
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Get replies (optionally by user_id for profile page; otherwise recent for leaderboard)
router.get('/replies', async (req, res: Response) => {
  try {
    const { user_id, limit = 2000 } = req.query;
    if (user_id) {
      const result = await query(
        'SELECT * FROM community_post_replies WHERE user_id = $1 ORDER BY created_at DESC',
        [user_id]
      );
      return res.json(result.rows);
    }
    const result = await query(
      'SELECT * FROM community_post_replies ORDER BY created_at DESC LIMIT $1',
      [Number(limit) || 2000]
    );
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Add reply to post
router.post('/posts/:id/replies', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { content, author_name } = req.body;
    const postId = req.params.id;

    const result = await query(
      `INSERT INTO community_post_replies (id, post_id, user_id, content, author_name, created_at)
       VALUES ($1, $2, $3, $4, $5, NOW())
       RETURNING *`,
      [uuidv4(), postId, req.user!.id, content, author_name]
    );

    // Update reply count
    await query('UPDATE community_posts SET replies_count = replies_count + 1 WHERE id = $1', [postId]);

    res.status(201).json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Get user follows
router.get('/follows', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      `SELECT uf.*, p.display_name, p.avatar_url 
       FROM user_follows uf 
       LEFT JOIN profiles p ON p.user_id = uf.following_id 
       WHERE uf.follower_id = $1`,
      [req.user!.id]
    );
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Follow user
router.post('/follow/:userId', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const followingId = req.params.userId;

    if (followingId === req.user!.id) {
      return res.status(400).json({ error: 'Cannot follow yourself' });
    }

    await query(
      `INSERT INTO user_follows (id, follower_id, following_id, created_at)
       VALUES ($1, $2, $3, NOW())
       ON CONFLICT (follower_id, following_id) DO NOTHING`,
      [uuidv4(), req.user!.id, followingId]
    );

    res.json({ following: true });
  } catch (error) {
    throw error;
  }
});

// Unfollow user
router.delete('/follow/:userId', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    await query(
      'DELETE FROM user_follows WHERE follower_id = $1 AND following_id = $2',
      [req.user!.id, req.params.userId]
    );
    res.json({ following: false });
  } catch (error) {
    throw error;
  }
});

// Get follow stats for a user
router.get('/stats/:userId', async (req, res: Response) => {
  try {
    const userId = req.params.userId;
    const followers = await query(
      'SELECT COUNT(*)::int AS count FROM user_follows WHERE following_id = $1',
      [userId]
    );
    const following = await query(
      'SELECT COUNT(*)::int AS count FROM user_follows WHERE follower_id = $1',
      [userId]
    );
    res.json({
      followersCount: followers.rows[0]?.count ?? 0,
      followingCount: following.rows[0]?.count ?? 0,
    });
  } catch (error) {
    throw error;
  }
});

export default router;
