import { Router, Response } from 'express';
import { query } from '../db';
import { authenticate, AuthRequest } from '../middleware/auth';
import { v4 as uuidv4 } from 'uuid';

const router = Router();

// Get all favorites
router.get('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'SELECT * FROM user_favorites WHERE user_id = $1 ORDER BY created_at DESC',
      [req.user!.id]
    );
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Check if item is favorited
router.get('/check/:itemType/:itemId', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { itemType, itemId } = req.params;

    const result = await query(
      'SELECT id FROM user_favorites WHERE user_id = $1 AND item_type = $2 AND item_id = $3',
      [req.user!.id, itemType, itemId]
    );

    res.json({ isFavorite: result.rows.length > 0 });
  } catch (error) {
    throw error;
  }
});

// Add favorite
router.post('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { item_id, item_type } = req.body;

    const result = await query(
      `INSERT INTO user_favorites (id, user_id, item_id, item_type, created_at)
       VALUES ($1, $2, $3, $4, NOW())
       ON CONFLICT (user_id, item_id, item_type) DO NOTHING
       RETURNING *`,
      [uuidv4(), req.user!.id, item_id, item_type]
    );

    res.status(201).json(result.rows[0] || { message: 'Already favorited' });
  } catch (error) {
    throw error;
  }
});

// Remove favorite
router.delete('/:itemType/:itemId', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { itemType, itemId } = req.params;

    const result = await query(
      'DELETE FROM user_favorites WHERE user_id = $1 AND item_type = $2 AND item_id = $3 RETURNING id',
      [req.user!.id, itemType, itemId]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Favorite not found' });
    }

    res.json({ message: 'Favorite removed' });
  } catch (error) {
    throw error;
  }
});

export default router;
