import { Router, Response } from 'express';
import { query } from '../db';
import { authenticate, AuthRequest } from '../middleware/auth';
import { v4 as uuidv4 } from 'uuid';

const router = Router();

// Get meal plans for date range
router.get('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { start_date, end_date } = req.query;

    let sql = 'SELECT mp.*, sr.title as recipe_title, sr.content as recipe_content FROM meal_plans mp LEFT JOIN saved_recipes sr ON sr.id = mp.recipe_id WHERE mp.user_id = $1';
    const params: any[] = [req.user!.id];

    if (start_date && end_date) {
      sql += ' AND mp.planned_date >= $2 AND mp.planned_date <= $3';
      params.push(start_date, end_date);
    }

    sql += ' ORDER BY mp.planned_date, mp.meal_slot';

    const result = await query(sql, params);
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Create meal plan
router.post('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { recipe_id, planned_date, meal_slot, notes, reminder_time, reminder_type } = req.body;

    const result = await query(
      `INSERT INTO meal_plans (id, user_id, recipe_id, planned_date, meal_slot, notes, reminder_time, reminder_type, created_at)
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8, NOW())
       RETURNING *`,
      [uuidv4(), req.user!.id, recipe_id, planned_date, meal_slot || 'dinner', notes, reminder_time, reminder_type || null]
    );

    res.status(201).json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Update meal plan
router.patch('/:id', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { planned_date, meal_slot, notes, reminder_time, reminder_type } = req.body;

    const result = await query(
      `UPDATE meal_plans 
       SET planned_date = COALESCE($1, planned_date),
           meal_slot = COALESCE($2, meal_slot),
           notes = COALESCE($3, notes),
           reminder_time = COALESCE($4, reminder_time),
           reminder_type = COALESCE($5, reminder_type)
       WHERE id = $6 AND user_id = $7
       RETURNING *`,
      [planned_date, meal_slot, notes, reminder_time, reminder_type, req.params.id, req.user!.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Meal plan not found' });
    }

    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Delete meal plan
router.delete('/:id', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'DELETE FROM meal_plans WHERE id = $1 AND user_id = $2 RETURNING id',
      [req.params.id, req.user!.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Meal plan not found' });
    }

    res.json({ message: 'Meal plan deleted' });
  } catch (error) {
    throw error;
  }
});

// Check meal reminders (cron or client-triggered; creates notifications for due reminders)
// Optional body: { user_id } to limit to one user. Call without auth for cron (or use cron secret).
router.post('/check-reminders', async (req: AuthRequest, res: Response) => {
  try {
    const { user_id: bodyUserId } = req.body || {};
    const now = new Date();
    const today = now.toISOString().split('T')[0];
    const tomorrow = new Date(now.getTime() + 24 * 60 * 60 * 1000).toISOString().split('T')[0];
    const currentHour = now.getHours();

    let sql = `SELECT mp.id, mp.user_id, mp.recipe_id, mp.planned_date, mp.meal_slot, mp.reminder_type, sr.title as recipe_title
      FROM meal_plans mp
      LEFT JOIN saved_recipes sr ON sr.id = mp.recipe_id
      WHERE mp.planned_date IN ($1, $2)`;
    const params: any[] = [today, tomorrow];

    if (bodyUserId) {
      sql += ' AND mp.user_id = $3';
      params.push(bodyUserId);
    }

    sql += ` AND (mp.reminder_type IS NOT NULL AND mp.reminder_type IN ('morning_before', 'evening_before', 'morning_of'))`;

    let plans: { id: string; user_id: string; recipe_id: string | null; planned_date: string; meal_slot: string; reminder_type: string | null; recipe_title: string | null }[] = [];
    try {
      const result = await query(sql, params);
      plans = result.rows;
    } catch (err: any) {
      if (err?.code === '42703') {
        console.warn('Check meal reminders: reminder_type column missing. Run migration: server/db/migrations/001_add_meal_plan_reminder_type.sql');
        return res.json({ success: true, checked: 0, notificationsSent: 0 });
      }
      throw err;
    }

    const notifications: { user_id: string; type: string; title: string; message: string; link: string; metadata: object }[] = [];

    for (const plan of plans) {
      const recipeTitle = plan.recipe_title || 'your meal';
      const mealDate = plan.planned_date;
      const reminderType = plan.reminder_type;
      let shouldSend = false;
      let message = '';

      if (mealDate === tomorrow) {
        if (reminderType === 'morning_before' && currentHour >= 8 && currentHour < 10) {
          shouldSend = true;
          message = `Tomorrow's ${plan.meal_slot}: ${recipeTitle}. Time to prep ingredients!`;
        } else if (reminderType === 'evening_before' && currentHour >= 17 && currentHour < 19) {
          shouldSend = true;
          message = `Tomorrow's ${plan.meal_slot}: ${recipeTitle}. Check if you have all ingredients!`;
        }
      } else if (mealDate === today) {
        if (reminderType === 'morning_of' && currentHour >= 6 && currentHour < 8) {
          shouldSend = true;
          message = `Today's ${plan.meal_slot}: ${recipeTitle}. Time to start cooking!`;
        }
      }

      if (shouldSend) {
        const since = new Date(now.getTime() - 6 * 60 * 60 * 1000).toISOString();
        const existing = await query(
          'SELECT id FROM notifications WHERE user_id = $1 AND type = $2 AND created_at >= $3 LIMIT 1',
          [plan.user_id, 'meal_reminder', since]
        );
        if (!existing.rows.length) {
          notifications.push({
            user_id: plan.user_id,
            type: 'meal_reminder',
            title: '🍳 Meal Prep Reminder',
            message,
            link: '/recipes',
            metadata: { meal_plan_id: plan.id, recipe_id: plan.recipe_id, planned_date: plan.planned_date, meal_slot: plan.meal_slot },
          });
        }
      }
    }

    if (notifications.length > 0) {
      for (const n of notifications) {
        await query(
          `INSERT INTO notifications (id, user_id, type, title, message, link, metadata) VALUES ($1, $2, $3, $4, $5, $6, $7)`,
          [uuidv4(), n.user_id, n.type, n.title, n.message, n.link, JSON.stringify(n.metadata)]
        );
      }
    }

    res.json({ success: true, checked: plans.length, notificationsSent: notifications.length });
  } catch (error) {
    console.error('Check meal reminders error:', error);
    throw error;
  }
});

export default router;
