import { Router, Response } from 'express';
import { query } from '../db';
import { authenticate, AuthRequest } from '../middleware/auth';
import { v4 as uuidv4 } from 'uuid';

const router = Router();

// List conversations for current user
router.get('/conversations', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const userId = req.user!.id;
    const result = await query(
      `SELECT c.*, 
        (CASE WHEN c.participant_1 = $1 THEN c.participant_2 ELSE c.participant_1 END) AS other_user_id,
        p.display_name, p.avatar_url,
        (SELECT content FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) AS last_message,
        (SELECT COUNT(*)::int FROM messages WHERE conversation_id = c.id AND sender_id != $1 AND is_read = false) AS unread_count
       FROM conversations c
       LEFT JOIN profiles p ON p.user_id = (CASE WHEN c.participant_1 = $1 THEN c.participant_2 ELSE c.participant_1 END)
       WHERE (c.participant_1 = $1 OR c.participant_2 = $1) AND c.is_group = false
       ORDER BY c.last_message_at DESC NULLS LAST`,
      [userId]
    );
    const rows = result.rows.map((r: any) => ({
      ...r,
      other_user: { user_id: r.other_user_id, display_name: r.display_name || null, avatar_url: r.avatar_url || null },
    }));
    res.json(rows);
  } catch (error) {
    throw error;
  }
});

// Get one conversation
router.get('/conversations/:id', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const userId = req.user!.id;
    const result = await query(
      `SELECT c.*, p.display_name, p.avatar_url FROM conversations c
       LEFT JOIN profiles p ON p.user_id = (CASE WHEN c.participant_1 = $1 THEN c.participant_2 ELSE c.participant_1 END)
       WHERE c.id = $2 AND (c.participant_1 = $1 OR c.participant_2 = $1)`,
      [userId, id]
    );
    if (result.rows.length === 0) return res.status(404).json({ error: 'Not found' });
    const r = result.rows[0];
    res.json({
      ...r,
      other_user: r.display_name != null ? { user_id: r.participant_1 === userId ? r.participant_2 : r.participant_1, display_name: r.display_name, avatar_url: r.avatar_url } : undefined,
    });
  } catch (error) {
    throw error;
  }
});

// Create DM conversation
router.post('/conversations', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { other_user_id } = req.body;
    if (!other_user_id) return res.status(400).json({ error: 'other_user_id required' });
    const userId = req.user!.id;
    const p1 = userId < other_user_id ? userId : other_user_id;
    const p2 = userId < other_user_id ? other_user_id : userId;
    const existing = await query(
      'SELECT id FROM conversations WHERE participant_1 = $1 AND participant_2 = $2 AND is_group = false',
      [p1, p2]
    );
    if (existing.rows.length > 0) {
      return res.json(existing.rows[0]);
    }
    const ins = await query(
      `INSERT INTO conversations (id, participant_1, participant_2, is_group) VALUES ($1, $2, $3, false) RETURNING *`,
      [uuidv4(), p1, p2]
    );
    res.status(201).json(ins.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Get messages
router.get('/conversations/:id/messages', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const userId = req.user!.id;
    const conv = await query(
      'SELECT id FROM conversations WHERE id = $1 AND (participant_1 = $2 OR participant_2 = $2)',
      [id, userId]
    );
    if (conv.rows.length === 0) return res.status(404).json({ error: 'Not found' });
    const result = await query(
      `SELECT m.*, p.display_name, p.avatar_url FROM messages m
       LEFT JOIN profiles p ON p.user_id = m.sender_id
       WHERE m.conversation_id = $1 ORDER BY m.created_at ASC`,
      [id]
    );
    const rows = result.rows.map((r: any) => ({
      ...r,
      sender_profile: r.display_name != null ? { display_name: r.display_name, avatar_url: r.avatar_url } : undefined,
    }));
    res.json(rows);
  } catch (error) {
    throw error;
  }
});

// Send message
router.post('/conversations/:id/messages', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const { content, attachment, voice } = req.body;
    const userId = req.user!.id;
    const conv = await query(
      'SELECT id FROM conversations WHERE id = $1 AND (participant_1 = $2 OR participant_2 = $2)',
      [id, userId]
    );
    if (conv.rows.length === 0) return res.status(404).json({ error: 'Not found' });
    const ins = await query(
      `INSERT INTO messages (id, conversation_id, sender_id, content, attachment_url, voice_url, voice_duration)
       VALUES ($1, $2, $3, $4, $5, $6, $7) RETURNING *`,
      [
        uuidv4(),
        id,
        userId,
        content || '',
        attachment?.url || null,
        voice?.url || null,
        voice?.duration || null,
      ]
    );
    await query(
      'UPDATE conversations SET last_message_at = NOW() WHERE id = $1',
      [id]
    );
    res.status(201).json(ins.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Mark as read
router.post('/conversations/:id/read', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const userId = req.user!.id;
    await query(
      'UPDATE messages SET is_read = true WHERE conversation_id = $1 AND sender_id != $2',
      [id, userId]
    );
    res.json({ ok: true });
  } catch (error) {
    throw error;
  }
});

// Stub for groups (return 501 or empty so UI doesn't break)
router.post('/groups', authenticate, (_req, res: Response) => {
  res.status(501).json({ error: 'Group chats not implemented' });
});

export default router;
