import { Router, Response } from 'express';
import { query } from '../db';
import { authenticate, AuthRequest } from '../middleware/auth';
import { v4 as uuidv4 } from 'uuid';

const router = Router();

// Get exposure logs for current user
router.get('/exposure', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'SELECT * FROM phytoestrogen_exposure WHERE user_id = $1 ORDER BY logged_at DESC',
      [req.user!.id]
    );
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Log exposure
router.post('/exposure', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { phytoestrogen_id, phytoestrogen_name, potency, food_source, servings = 1, notes, logged_at } = req.body;
    const result = await query(
      `INSERT INTO phytoestrogen_exposure (id, user_id, phytoestrogen_id, phytoestrogen_name, potency, food_source, servings, notes, logged_at)
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
       RETURNING *`,
      [
        uuidv4(),
        req.user!.id,
        phytoestrogen_id,
        phytoestrogen_name,
        potency,
        food_source,
        servings,
        notes || null,
        logged_at || new Date().toISOString(),
      ]
    );
    res.status(201).json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Delete exposure entry
router.delete('/exposure/:id', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    await query(
      'DELETE FROM phytoestrogen_exposure WHERE id = $1 AND user_id = $2',
      [req.params.id, req.user!.id]
    );
    res.status(204).send();
  } catch (error) {
    throw error;
  }
});

// Get goals for current user
router.get('/goals', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'SELECT * FROM phytoestrogen_goals WHERE user_id = $1',
      [req.user!.id]
    );
    res.json(result.rows[0] || null);
  } catch (error) {
    throw error;
  }
});

// Create or update goals
router.put('/goals', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { daily_limit, weekly_limit, show_alerts, alert_at_percentage } = req.body;
    const result = await query(
      `INSERT INTO phytoestrogen_goals (id, user_id, daily_limit, weekly_limit, show_alerts, alert_at_percentage, created_at, updated_at)
       VALUES (gen_random_uuid(), $1, $2, $3, $4, $5, NOW(), NOW())
       ON CONFLICT (user_id) DO UPDATE SET
         daily_limit = COALESCE(EXCLUDED.daily_limit, phytoestrogen_goals.daily_limit),
         weekly_limit = COALESCE(EXCLUDED.weekly_limit, phytoestrogen_goals.weekly_limit),
         show_alerts = COALESCE(EXCLUDED.show_alerts, phytoestrogen_goals.show_alerts),
         alert_at_percentage = COALESCE(EXCLUDED.alert_at_percentage, phytoestrogen_goals.alert_at_percentage),
         updated_at = NOW()
       RETURNING *`,
      [
        req.user!.id,
        daily_limit ?? 100,
        weekly_limit ?? 500,
        show_alerts ?? true,
        alert_at_percentage ?? 80,
      ]
    );
    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

export default router;
