import { Router, Response } from 'express';
import { query } from '../db';
import { authenticate, AuthRequest } from '../middleware/auth';
import { v4 as uuidv4 } from 'uuid';

const router = Router();

// Get all saved recipes
router.get('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'SELECT * FROM saved_recipes WHERE user_id = $1 ORDER BY created_at DESC',
      [req.user!.id]
    );
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Get single recipe
router.get('/:id', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'SELECT * FROM saved_recipes WHERE id = $1 AND user_id = $2',
      [req.params.id, req.user!.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Recipe not found' });
    }

    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Get shared recipe by token
router.get('/shared/:token', async (req, res: Response) => {
  try {
    const result = await query(
      'SELECT * FROM saved_recipes WHERE share_token = $1 AND is_shared = true',
      [req.params.token]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Recipe not found' });
    }

    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Create recipe
router.post('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { title, content, meal_type, servings, nutritional_info } = req.body;

    const result = await query(
      `INSERT INTO saved_recipes (id, user_id, title, content, meal_type, servings, nutritional_info, created_at)
       VALUES ($1, $2, $3, $4, $5, $6, $7, NOW())
       RETURNING *`,
      [uuidv4(), req.user!.id, title, content, meal_type || 'any', servings || 4, nutritional_info]
    );

    res.status(201).json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Update recipe
router.patch('/:id', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { title, content, meal_type, servings, nutritional_info, is_shared } = req.body;

    let shareToken = null;
    if (is_shared) {
      shareToken = uuidv4();
    }

    const result = await query(
      `UPDATE saved_recipes 
       SET title = COALESCE($1, title),
           content = COALESCE($2, content),
           meal_type = COALESCE($3, meal_type),
           servings = COALESCE($4, servings),
           nutritional_info = COALESCE($5, nutritional_info),
           is_shared = COALESCE($6, is_shared),
           share_token = CASE WHEN $6 = true THEN COALESCE(share_token, $7) ELSE share_token END
       WHERE id = $8 AND user_id = $9
       RETURNING *`,
      [title, content, meal_type, servings, nutritional_info, is_shared, shareToken, req.params.id, req.user!.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Recipe not found' });
    }

    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Delete recipe
router.delete('/:id', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'DELETE FROM saved_recipes WHERE id = $1 AND user_id = $2 RETURNING id',
      [req.params.id, req.user!.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Recipe not found' });
    }

    res.json({ message: 'Recipe deleted' });
  } catch (error) {
    throw error;
  }
});

export default router;
