import { Router, Response } from 'express';
import { query } from '../db';
import { authenticate, AuthRequest } from '../middleware/auth';
import { v4 as uuidv4 } from 'uuid';

const router = Router();

// Get scan history
router.get('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { limit = 50 } = req.query;

    const result = await query(
      'SELECT * FROM scan_history WHERE user_id = $1 ORDER BY scanned_at DESC LIMIT $2',
      [req.user!.id, limit]
    );

    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Add scan to history
router.post('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { barcode, product_id, product_name, product_type, image_url, product_details } = req.body;

    const result = await query(
      `INSERT INTO scan_history (id, user_id, barcode, product_id, product_name, product_type, image_url, product_details, scanned_at)
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8, NOW())
       RETURNING *`,
      [uuidv4(), req.user!.id, barcode, product_id, product_name, product_type, image_url, product_details]
    );

    res.status(201).json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Delete scan from history
router.delete('/:id', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'DELETE FROM scan_history WHERE id = $1 AND user_id = $2 RETURNING id',
      [req.params.id, req.user!.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Scan not found' });
    }

    res.json({ message: 'Scan deleted' });
  } catch (error) {
    throw error;
  }
});

// Clear all scan history
router.delete('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    await query('DELETE FROM scan_history WHERE user_id = $1', [req.user!.id]);
    res.json({ message: 'Scan history cleared' });
  } catch (error) {
    throw error;
  }
});

export default router;
