import { Router, Response } from 'express';
import { query } from '../db';
import { authenticate, AuthRequest } from '../middleware/auth';
import { v4 as uuidv4 } from 'uuid';

const router = Router();

// Get shopping list
router.get('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'SELECT * FROM shopping_list WHERE user_id = $1 ORDER BY created_at DESC',
      [req.user!.id]
    );
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Add item to shopping list
router.post('/', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { item_id, item_type, quantity, notes } = req.body;

    const result = await query(
      `INSERT INTO shopping_list (id, user_id, item_id, item_type, quantity, notes, is_purchased, created_at)
       VALUES ($1, $2, $3, $4, $5, $6, false, NOW())
       RETURNING *`,
      [uuidv4(), req.user!.id, item_id, item_type, quantity || 1, notes]
    );

    res.status(201).json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Update shopping list item
router.patch('/:id', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { quantity, notes, is_purchased } = req.body;

    const result = await query(
      `UPDATE shopping_list 
       SET quantity = COALESCE($1, quantity),
           notes = COALESCE($2, notes),
           is_purchased = COALESCE($3, is_purchased)
       WHERE id = $4 AND user_id = $5
       RETURNING *`,
      [quantity, notes, is_purchased, req.params.id, req.user!.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Item not found' });
    }

    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Toggle purchased status
router.patch('/:id/toggle', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      `UPDATE shopping_list 
       SET is_purchased = NOT is_purchased
       WHERE id = $1 AND user_id = $2
       RETURNING *`,
      [req.params.id, req.user!.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Item not found' });
    }

    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Remove item from shopping list
router.delete('/:id', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'DELETE FROM shopping_list WHERE id = $1 AND user_id = $2 RETURNING id',
      [req.params.id, req.user!.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Item not found' });
    }

    res.json({ message: 'Item removed' });
  } catch (error) {
    throw error;
  }
});

// Clear purchased items
router.delete('/clear/purchased', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    await query('DELETE FROM shopping_list WHERE user_id = $1 AND is_purchased = true', [req.user!.id]);
    res.json({ message: 'Purchased items cleared' });
  } catch (error) {
    throw error;
  }
});

export default router;
