import { Router, Response } from 'express';
import Stripe from 'stripe';
import { authenticate, AuthRequest } from '../middleware/auth';
import { query } from '../db';

const router = Router();

const stripe = new Stripe(process.env.STRIPE_SECRET_KEY || '', {
  apiVersion: '2024-11-20.acacia',
});

const STRIPE_PRICE_ID = process.env.STRIPE_PRICE_ID || process.env.STRIPE_PREMIUM_PRICE_ID;

// Check subscription status
router.get('/status', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const userResult = await query('SELECT email FROM users WHERE id = $1', [req.user!.id]);
    if (userResult.rows.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    const email = userResult.rows[0].email;
    const customers = await stripe.customers.list({ email, limit: 1 });

    if (customers.data.length === 0) {
      return res.json({ subscribed: false, is_trial: false, product_id: null, subscription_end: null, trial_end: null });
    }

    const customerId = customers.data[0].id;

    // Check for active subscriptions
    const subscriptions = await stripe.subscriptions.list({
      customer: customerId,
      status: 'active',
      limit: 1,
    });

    // Check for trialing subscriptions
    const trialingSubs = await stripe.subscriptions.list({
      customer: customerId,
      status: 'trialing',
      limit: 1,
    });

    const hasActiveSub = subscriptions.data.length > 0;
    const hasTrialingSub = trialingSubs.data.length > 0;
    const activeSubscription = subscriptions.data[0] || trialingSubs.data[0];

    if (!activeSubscription) {
      return res.json({
        subscribed: false,
        is_trial: false,
        product_id: null,
        subscription_end: null,
        trial_end: null,
      });
    }

    let subscriptionEnd: string | null = null;
    let trialEnd: string | null = null;

    if (activeSubscription.current_period_end) {
      subscriptionEnd = new Date(activeSubscription.current_period_end * 1000).toISOString();
    }
    if (activeSubscription.trial_end) {
      trialEnd = new Date(activeSubscription.trial_end * 1000).toISOString();
    }

    const productId = (activeSubscription.items?.data?.[0]?.price?.product as string) || null;

    res.json({
      subscribed: hasActiveSub || hasTrialingSub,
      is_trial: hasTrialingSub && !hasActiveSub,
      product_id: productId,
      subscription_end: subscriptionEnd,
      trial_end: trialEnd,
    });
  } catch (error) {
    console.error('Error checking subscription:', error);
    throw error;
  }
});

// Create checkout session
router.post('/create-checkout', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const userResult = await query('SELECT email FROM users WHERE id = $1', [req.user!.id]);
    if (userResult.rows.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    const email = userResult.rows[0].email;
    const customers = await stripe.customers.list({ email, limit: 1 });
    
    let customerId: string | undefined;
    if (customers.data.length > 0) {
      customerId = customers.data[0].id;
    }

    const priceId = req.body?.priceId || STRIPE_PRICE_ID;
    if (!priceId || priceId.startsWith('price_YOUR_')) {
      return res.status(400).json({ error: 'Price ID not configured. Set STRIPE_PRICE_ID in server .env.' });
    }
    const origin = req.headers.origin || process.env.FRONTEND_URL;

    const session = await stripe.checkout.sessions.create({
      customer: customerId,
      customer_email: customerId ? undefined : email,
      line_items: [
        {
          price: priceId,
          quantity: 1,
        },
      ],
      mode: 'subscription',
      subscription_data: {
        trial_period_days: 7,
      },
      success_url: `${origin}/subscription-success`,
      cancel_url: `${origin}/subscription-canceled`,
    });

    res.json({ url: session.url });
  } catch (error) {
    console.error('Error creating checkout:', error);
    throw error;
  }
});

// Customer portal
router.post('/customer-portal', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const userResult = await query('SELECT email FROM users WHERE id = $1', [req.user!.id]);
    if (userResult.rows.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    const email = userResult.rows[0].email;
    const customers = await stripe.customers.list({ email, limit: 1 });

    if (customers.data.length === 0) {
      return res.status(400).json({ error: 'No Stripe customer found' });
    }

    const origin = req.headers.origin || process.env.FRONTEND_URL;
    const portalSession = await stripe.billingPortal.sessions.create({
      customer: customers.data[0].id,
      return_url: `${origin}/subscription`,
    });

    res.json({ url: portalSession.url });
  } catch (error) {
    console.error('Error creating portal session:', error);
    throw error;
  }
});

export default router;
