import { Router, Response } from 'express';
import fs from 'fs';
import path from 'path';
import { query } from '../db';
import { authenticate, AuthRequest } from '../middleware/auth';

const router = Router();
const UPLOADS_DIR = path.join(process.cwd(), 'uploads', 'avatars');

// Get user profile (own)
router.get('/profile', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      `SELECT p.*, u.email FROM profiles p 
       JOIN users u ON u.id = p.user_id 
       WHERE p.user_id = $1`,
      [req.user!.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Profile not found' });
    }

    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Upload avatar (base64 image in body)
router.post('/avatar', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { data } = req.body as { data?: string };
    if (!data || typeof data !== 'string') {
      return res.status(400).json({ error: 'data (base64 image) required' });
    }
    const match = data.match(/^data:image\/(\w+);base64,(.+)$/);
    const ext = match ? match[1] : 'png';
    const base64 = match ? match[2] : data;
    const buf = Buffer.from(base64, 'base64');
    if (buf.length > 2 * 1024 * 1024) {
      return res.status(400).json({ error: 'Image too large (max 2MB)' });
    }
    if (!fs.existsSync(UPLOADS_DIR)) {
      fs.mkdirSync(UPLOADS_DIR, { recursive: true });
    }
    const filename = `${req.user!.id}.${ext}`;
    const filepath = path.join(UPLOADS_DIR, filename);
    fs.writeFileSync(filepath, buf);
    const avatarUrl = `/uploads/avatars/${filename}`;
    await query(
      'UPDATE profiles SET avatar_url = $1, updated_at = NOW() WHERE user_id = $2',
      [avatarUrl, req.user!.id]
    );
    res.json({ avatar_url: avatarUrl });
  } catch (error) {
    throw error;
  }
});

// Update profile
router.patch('/profile', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const { display_name, avatar_url, bio } = req.body;

    const result = await query(
      `UPDATE profiles 
       SET display_name = COALESCE($1, display_name),
           avatar_url = COALESCE($2, avatar_url),
           bio = COALESCE($3, bio),
           updated_at = NOW()
       WHERE user_id = $4
       RETURNING *`,
      [display_name, avatar_url, bio, req.user!.id]
    );

    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Get health profile
router.get('/health-profile', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'SELECT * FROM user_health_profiles WHERE user_id = $1',
      [req.user!.id]
    );

    res.json(result.rows[0] || null);
  } catch (error) {
    throw error;
  }
});

// Update health profile
router.put('/health-profile', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const {
      has_allergies, has_autoimmune, has_cardiovascular_issues,
      has_digestive_issues, has_hormonal_imbalance, has_skin_sensitivity,
      show_alerts, total_threshold, toxin_threshold, biological_threshold,
      inflammatory_threshold, oxidative_threshold, nutrient_threshold
    } = req.body;

    const result = await query(
      `INSERT INTO user_health_profiles (
        id, user_id, has_allergies, has_autoimmune, has_cardiovascular_issues,
        has_digestive_issues, has_hormonal_imbalance, has_skin_sensitivity,
        show_alerts, total_threshold, toxin_threshold, biological_threshold,
        inflammatory_threshold, oxidative_threshold, nutrient_threshold,
        created_at, updated_at
      ) VALUES (
        gen_random_uuid(), $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, NOW(), NOW()
      )
      ON CONFLICT (user_id) DO UPDATE SET
        has_allergies = EXCLUDED.has_allergies,
        has_autoimmune = EXCLUDED.has_autoimmune,
        has_cardiovascular_issues = EXCLUDED.has_cardiovascular_issues,
        has_digestive_issues = EXCLUDED.has_digestive_issues,
        has_hormonal_imbalance = EXCLUDED.has_hormonal_imbalance,
        has_skin_sensitivity = EXCLUDED.has_skin_sensitivity,
        show_alerts = EXCLUDED.show_alerts,
        total_threshold = EXCLUDED.total_threshold,
        toxin_threshold = EXCLUDED.toxin_threshold,
        biological_threshold = EXCLUDED.biological_threshold,
        inflammatory_threshold = EXCLUDED.inflammatory_threshold,
        oxidative_threshold = EXCLUDED.oxidative_threshold,
        nutrient_threshold = EXCLUDED.nutrient_threshold,
        updated_at = NOW()
      RETURNING *`,
      [
        req.user!.id, has_allergies, has_autoimmune, has_cardiovascular_issues,
        has_digestive_issues, has_hormonal_imbalance, has_skin_sensitivity,
        show_alerts, total_threshold, toxin_threshold, biological_threshold,
        inflammatory_threshold, oxidative_threshold, nutrient_threshold
      ]
    );

    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Get dietary preferences
router.get('/dietary-preferences', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const result = await query(
      'SELECT * FROM user_dietary_preferences WHERE user_id = $1',
      [req.user!.id]
    );

    res.json(result.rows[0] || null);
  } catch (error) {
    throw error;
  }
});

// Update dietary preferences
router.put('/dietary-preferences', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const {
      halal_preference, kosher_preference, organic_preference,
      avoid_xenoestrogens, max_risk_level
    } = req.body;

    const result = await query(
      `INSERT INTO user_dietary_preferences (
        id, user_id, halal_preference, kosher_preference, organic_preference,
        avoid_xenoestrogens, max_risk_level, created_at, updated_at
      ) VALUES (
        gen_random_uuid(), $1, $2, $3, $4, $5, $6, NOW(), NOW()
      )
      ON CONFLICT (user_id) DO UPDATE SET
        halal_preference = EXCLUDED.halal_preference,
        kosher_preference = EXCLUDED.kosher_preference,
        organic_preference = EXCLUDED.organic_preference,
        avoid_xenoestrogens = EXCLUDED.avoid_xenoestrogens,
        max_risk_level = EXCLUDED.max_risk_level,
        updated_at = NOW()
      RETURNING *`,
      [req.user!.id, halal_preference, kosher_preference, organic_preference, avoid_xenoestrogens, max_risk_level]
    );

    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

// Search profiles by display name (for messaging, etc.)
router.get('/search', authenticate, async (req: AuthRequest, res: Response) => {
  try {
    const q = (req.query.q as string)?.trim();
    if (!q || q.length < 2) {
      return res.json([]);
    }
    const result = await query(
      `SELECT user_id, display_name, avatar_url FROM profiles 
       WHERE display_name ILIKE $1 AND user_id != $2 
       ORDER BY display_name LIMIT 10`,
      [`%${q}%`, req.user!.id]
    );
    res.json(result.rows);
  } catch (error) {
    throw error;
  }
});

// Get public profile by user id (for profile pages) - must be last to not match /profile
router.get('/:id', async (req, res: Response) => {
  try {
    const result = await query(
      'SELECT display_name, bio, avatar_url, user_id FROM profiles WHERE user_id = $1',
      [req.params.id]
    );
    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Profile not found' });
    }
    res.json(result.rows[0]);
  } catch (error) {
    throw error;
  }
});

export default router;
