import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { 
  Shield, 
  AlertTriangle, 
  CheckCircle2, 
  XCircle,
  Sparkles,
  Users,
  Ban,
  RefreshCw,
  Globe,
  FileText,
  Star
} from "lucide-react";
import type { AIAnalysis } from "@/hooks/useBarcodeLookup";
import { getHalalStatusColor, getKosherStatusColor, getOrganicStatusColor } from "@/data/foodDatabase";

interface AIAnalysisCardProps {
  analysis: AIAnalysis;
  hasWebSearch?: boolean;
}

const getSafetyColor = (rating: string) => {
  switch (rating) {
    case 'Safe':
      return 'bg-emerald-500/10 text-emerald-600 border-emerald-500/20';
    case 'Moderate Concern':
      return 'bg-amber-500/10 text-amber-600 border-amber-500/20';
    case 'High Concern':
      return 'bg-rose-500/10 text-rose-600 border-rose-500/20';
    default:
      return 'bg-muted text-muted-foreground';
  }
};

const getSafetyIcon = (rating: string) => {
  switch (rating) {
    case 'Safe':
      return <CheckCircle2 className="h-5 w-5 text-emerald-600" />;
    case 'Moderate Concern':
      return <AlertTriangle className="h-5 w-5 text-amber-600" />;
    case 'High Concern':
      return <XCircle className="h-5 w-5 text-rose-600" />;
    default:
      return <Shield className="h-5 w-5" />;
  }
};

// API may return items as strings or objects (e.g. { concern, explanation }) – always render a string
function toDisplayText(v: unknown): string {
  if (v == null) return '';
  if (typeof v === 'string') return v;
  if (typeof v === 'object' && v !== null) {
    const o = v as Record<string, unknown>;
    const concern = o.concern != null ? String(o.concern) : '';
    const explanation = o.explanation != null ? String(o.explanation) : '';
    if (concern && explanation) return `${concern}: ${explanation}`;
    if (concern) return concern;
    if (explanation) return explanation;
    const name = o.name != null ? String(o.name) : '';
    const note = o.note != null ? String(o.note) : '';
    if (name && note) return `${name}. ${note}`;
    if (name) return name;
    if (note) return note;
  }
  return String(v);
}

const getTobinRating = (score: number) => {
  if (score <= 10) return { label: 'Low concern', color: 'text-emerald-600', bg: 'bg-emerald-500/10' };
  if (score <= 25) return { label: 'Moderate', color: 'text-amber-600', bg: 'bg-amber-500/10' };
  return { label: 'High concern', color: 'text-rose-600', bg: 'bg-rose-500/10' };
};

export const AIAnalysisCard = ({ analysis, hasWebSearch = true }: AIAnalysisCardProps) => {
  const hasStats = analysis.tobinScore != null || analysis.glycogenicIndex != null || analysis.lipogenicIndex != null
    || analysis.nutriscoreGrade != null || analysis.novaGroup != null;
  const tobinRating = typeof analysis.tobinScore === 'number' && Number.isFinite(analysis.tobinScore)
    ? getTobinRating(analysis.tobinScore)
    : null;
  const nutriStyle = (g: string) => {
    const x = (g || '').toUpperCase();
    if (x === 'A') return 'bg-green-600 text-white';
    if (x === 'B') return 'bg-lime-500 text-white';
    if (x === 'C') return 'bg-yellow-500 text-black';
    if (x === 'D') return 'bg-orange-500 text-white';
    if (x === 'E') return 'bg-red-600 text-white';
    return 'bg-muted';
  };

  return (
    <Card className="border-primary/20 bg-gradient-to-br from-primary/5 to-background">
      <CardHeader className="pb-3">
        <CardTitle className="flex flex-wrap items-center gap-2 text-lg font-semibold">
          <Sparkles className="h-5 w-5 text-primary" />
          AI Research Analysis
          {hasWebSearch && (
            <Badge variant="secondary" className="text-xs font-normal gap-1">
              <Globe className="h-3 w-3" />
              With web research
            </Badge>
          )}
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-5">
        {/* Safety Rating */}
        <div className="flex items-center gap-3">
          {getSafetyIcon(analysis.safetyRating)}
          <Badge 
            variant="outline" 
            className={`${getSafetyColor(analysis.safetyRating)} text-sm px-3 py-1`}
          >
            {analysis.safetyRating}
          </Badge>
        </div>

        {/* Key statistics (TOBIN, GI*, LI) when present */}
        {hasStats && (
          <div className="rounded-lg border border-border/60 bg-muted/30 p-3 space-y-2">
            <h4 className="text-sm font-medium flex items-center gap-2">
              <FileText className="h-4 w-4 text-primary" />
              Key statistics
            </h4>
            <div className="flex flex-wrap gap-3 text-sm">
              {analysis.nutriscoreGrade && (
                <span className="flex items-center gap-1.5">
                  <span className="text-muted-foreground">Nutri-Score:</span>
                  <Badge className={nutriStyle(analysis.nutriscoreGrade)}>{analysis.nutriscoreGrade.toUpperCase()}</Badge>
                </span>
              )}
              {analysis.novaGroup != null && Number.isFinite(Number(analysis.novaGroup)) && (
                <span className="flex items-center gap-1.5">
                  <span className="text-muted-foreground">NOVA:</span>
                  <Badge variant="outline">Group {analysis.novaGroup}</Badge>
                </span>
              )}
              {analysis.tobinScore != null && Number.isFinite(analysis.tobinScore) && tobinRating && (
                <span className="flex items-center gap-1.5">
                  <span className="text-muted-foreground">TOBIN:</span>
                  <Badge variant="outline" className={tobinRating.bg}>{analysis.tobinScore}/50</Badge>
                  <span className={tobinRating.color}>{tobinRating.label}</span>
                </span>
              )}
              {analysis.glycogenicIndex != null && Number.isFinite(analysis.glycogenicIndex) && (
                <span className="flex items-center gap-1.5">
                  <span className="text-muted-foreground">GI*:</span>
                  <Badge variant="outline">{Number(analysis.glycogenicIndex).toFixed(2)}</Badge>
                </span>
              )}
              {analysis.lipogenicIndex != null && Number.isFinite(analysis.lipogenicIndex) && (
                <span className="flex items-center gap-1.5">
                  <span className="text-muted-foreground">LI:</span>
                  <Badge variant="outline">{Number(analysis.lipogenicIndex).toFixed(2)}</Badge>
                </span>
              )}
            </div>
          </div>
        )}

        {/* Dietary: Halal, Kosher, Organic */}
        {(analysis.halal || analysis.kosher || analysis.organic) && (
          <div className="rounded-lg border border-border/60 bg-muted/30 p-3 space-y-2">
            <h4 className="text-sm font-medium flex items-center gap-2">
              <Star className="h-4 w-4 text-primary" />
              Dietary status
            </h4>
            <div className="flex flex-wrap gap-2 text-sm">
              {analysis.halal && (
                <span className="flex items-center gap-1.5">
                  <span className="text-muted-foreground">Halal:</span>
                  <Badge className={getHalalStatusColor(analysis.halal as 'halal' | 'haram' | 'mashbooh')}>
                    {analysis.halal === 'mashbooh' ? 'Doubtful' : analysis.halal.replace('_', ' ')}
                  </Badge>
                </span>
              )}
              {analysis.kosher && (
                <span className="flex items-center gap-1.5">
                  <span className="text-muted-foreground">Kosher:</span>
                  <Badge className={getKosherStatusColor(analysis.kosher as 'kosher' | 'not_kosher' | 'pareve' | 'dairy' | 'meat')}>
                    {analysis.kosher.replace('_', ' ')}
                  </Badge>
                </span>
              )}
              {analysis.organic && (
                <span className="flex items-center gap-1.5">
                  <span className="text-muted-foreground">Organic:</span>
                  <Badge className={getOrganicStatusColor(analysis.organic as 'organic' | 'non_organic' | 'varies')}>
                    {analysis.organic.replace('_', ' ')}
                  </Badge>
                </span>
              )}
            </div>
            {analysis.certificationNotes && (
              <p className="text-xs text-muted-foreground italic mt-1">{analysis.certificationNotes}</p>
            )}
          </div>
        )}

        {/* Detailed summary */}
        <div>
          <h4 className="text-sm font-medium mb-2 flex items-center gap-2">
            <FileText className="h-4 w-4 text-primary" />
            Detailed summary
          </h4>
          <p className="text-sm text-muted-foreground leading-relaxed whitespace-pre-line">
            {analysis.summary}
          </p>
        </div>

        {/* Key Ingredients */}
        {analysis.keyIngredients && analysis.keyIngredients.length > 0 && (
          <div>
            <h4 className="text-sm font-medium mb-2 flex items-center gap-2">
              <Shield className="h-4 w-4 text-primary" />
              Key Ingredients
            </h4>
            <div className="flex flex-wrap gap-1.5">
              {analysis.keyIngredients.map((ingredient, index) => (
                <Badge key={index} variant="secondary" className="text-xs">
                  {toDisplayText(ingredient)}
                </Badge>
              ))}
            </div>
          </div>
        )}

        {/* Benefits */}
        {analysis.benefits && analysis.benefits.length > 0 && (
          <div>
            <h4 className="text-sm font-medium mb-2 flex items-center gap-2">
              <CheckCircle2 className="h-4 w-4 text-emerald-600" />
              Benefits
            </h4>
            <ul className="text-sm text-muted-foreground space-y-1">
              {analysis.benefits.map((benefit, index) => (
                <li key={index} className="flex items-start gap-2">
                  <span className="text-emerald-600 mt-1">•</span>
                  {toDisplayText(benefit)}
                </li>
              ))}
            </ul>
          </div>
        )}

        {/* Concerns */}
        {analysis.concerns && analysis.concerns.length > 0 && (
          <div>
            <h4 className="text-sm font-medium mb-2 flex items-center gap-2">
              <AlertTriangle className="h-4 w-4 text-amber-600" />
              Concerns
            </h4>
            <ul className="text-sm text-muted-foreground space-y-1">
              {analysis.concerns.map((concern, index) => (
                <li key={index} className="flex items-start gap-2">
                  <span className="text-amber-600 mt-1">•</span>
                  {toDisplayText(concern)}
                </li>
              ))}
            </ul>
          </div>
        )}

        {/* Suitable For */}
        {analysis.suitableFor && analysis.suitableFor.length > 0 && (
          <div>
            <h4 className="text-sm font-medium mb-2 flex items-center gap-2">
              <Users className="h-4 w-4 text-sky-600" />
              Suitable For
            </h4>
            <div className="flex flex-wrap gap-1.5">
              {analysis.suitableFor.map((item, index) => (
                <Badge key={index} variant="outline" className="text-xs bg-sky-500/10 text-sky-600 border-sky-500/20">
                  {toDisplayText(item)}
                </Badge>
              ))}
            </div>
          </div>
        )}

        {/* Avoid If */}
        {analysis.avoidIf && analysis.avoidIf.length > 0 && (
          <div>
            <h4 className="text-sm font-medium mb-2 flex items-center gap-2">
              <Ban className="h-4 w-4 text-rose-600" />
              Avoid If
            </h4>
            <div className="flex flex-wrap gap-1.5">
              {analysis.avoidIf.map((item, index) => (
                <Badge key={index} variant="outline" className="text-xs bg-rose-500/10 text-rose-600 border-rose-500/20">
                  {toDisplayText(item)}
                </Badge>
              ))}
            </div>
          </div>
        )}

        {/* Alternatives */}
        {analysis.alternatives && analysis.alternatives.length > 0 && (
          <div>
            <h4 className="text-sm font-medium mb-2 flex items-center gap-2">
              <RefreshCw className="h-4 w-4 text-primary" />
              Safer Alternatives
            </h4>
            <ul className="text-sm text-muted-foreground space-y-1">
              {analysis.alternatives.map((alt, index) => (
                <li key={index} className="flex items-start gap-2">
                  <span className="text-primary mt-1">→</span>
                  {toDisplayText(alt)}
                </li>
              ))}
            </ul>
          </div>
        )}

        {/* Sources */}
        {analysis.sources && analysis.sources.length > 0 && (
          <div className="pt-2 border-t border-border/50">
            <h4 className="text-xs font-medium mb-1.5 flex items-center gap-1.5 text-muted-foreground">
              <FileText className="h-3 w-3" />
              Research Sources
            </h4>
            <ul className="text-xs text-muted-foreground/70 space-y-0.5">
              {analysis.sources.map((source, index) => (
                <li key={index}>• {toDisplayText(source)}</li>
              ))}
            </ul>
          </div>
        )}
      </CardContent>
    </Card>
  );
};
