import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { useCommunityProducts } from '@/hooks/useCommunityProducts';
import { Plus, Loader2, Users } from 'lucide-react';

interface AddProductFormProps {
  barcode: string;
  onSuccess?: () => void;
  onCancel?: () => void;
}

export const AddProductForm = ({ barcode, onSuccess, onCancel }: AddProductFormProps) => {
  const { addProduct, loading } = useCommunityProducts();
  const [productName, setProductName] = useState('');
  const [brand, setBrand] = useState('');
  const [productType, setProductType] = useState<'food' | 'cosmetic'>('cosmetic');
  const [categories, setCategories] = useState('');
  const [ingredients, setIngredients] = useState('');

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!productName.trim()) {
      return;
    }

    const success = await addProduct({
      barcode,
      product_name: productName.trim(),
      brand: brand.trim() || undefined,
      product_type: productType,
      categories: categories.trim() || undefined,
      ingredients: ingredients.trim() || undefined,
    });

    if (success) {
      setProductName('');
      setBrand('');
      setCategories('');
      setIngredients('');
      onSuccess?.();
    }
  };

  return (
    <Card className="border-primary/20 bg-primary/5">
      <CardHeader className="pb-3">
        <CardTitle className="flex items-center gap-2 text-lg">
          <Users className="h-5 w-5 text-primary" />
          Help Build Our Database
        </CardTitle>
        <CardDescription>
          This product wasn't found. Add it to help others!
        </CardDescription>
      </CardHeader>
      <CardContent>
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="barcode-display">Barcode</Label>
            <Input id="barcode-display" value={barcode} disabled className="bg-muted" />
          </div>

          <div className="space-y-2">
            <Label htmlFor="product-name">Product Name *</Label>
            <Input
              id="product-name"
              placeholder="e.g., L'Oréal Paris Infallible Eye Liner"
              value={productName}
              onChange={(e) => setProductName(e.target.value)}
              required
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="brand">Brand</Label>
            <Input
              id="brand"
              placeholder="e.g., L'Oréal Paris"
              value={brand}
              onChange={(e) => setBrand(e.target.value)}
            />
          </div>

          <div className="space-y-2">
            <Label>Product Type *</Label>
            <RadioGroup
              value={productType}
              onValueChange={(value) => setProductType(value as 'food' | 'cosmetic')}
              className="flex gap-4"
            >
              <div className="flex items-center space-x-2">
                <RadioGroupItem value="cosmetic" id="cosmetic" />
                <Label htmlFor="cosmetic" className="cursor-pointer">💄 Cosmetic</Label>
              </div>
              <div className="flex items-center space-x-2">
                <RadioGroupItem value="food" id="food" />
                <Label htmlFor="food" className="cursor-pointer">🍽️ Food</Label>
              </div>
            </RadioGroup>
          </div>

          <div className="space-y-2">
            <Label htmlFor="categories">Categories</Label>
            <Input
              id="categories"
              placeholder="e.g., Makeup, Eye Liner, Cosmetics"
              value={categories}
              onChange={(e) => setCategories(e.target.value)}
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="ingredients">Ingredients (from label)</Label>
            <Textarea
              id="ingredients"
              placeholder="Copy ingredients from the product label..."
              value={ingredients}
              onChange={(e) => setIngredients(e.target.value)}
              rows={3}
            />
          </div>

          <div className="flex gap-2">
            <Button type="submit" disabled={loading || !productName.trim()} className="flex-1">
              {loading ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Adding...
                </>
              ) : (
                <>
                  <Plus className="h-4 w-4 mr-2" />
                  Add Product
                </>
              )}
            </Button>
            {onCancel && (
              <Button type="button" variant="outline" onClick={onCancel}>
                Cancel
              </Button>
            )}
          </div>
        </form>
      </CardContent>
    </Card>
  );
};
