import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Avatar, AvatarFallback } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';
import { Trophy, MessageSquare, Heart, Award, Crown, Medal, Loader2 } from 'lucide-react';
import { communityApi } from '@/lib/apiServices';

interface LeaderboardEntry {
  author_name: string;
  user_id: string;
  count: number;
}

const rankIcons = [Crown, Medal, Award];
const rankColors = [
  'text-amber-500',
  'text-slate-400',
  'text-amber-700',
];

const LeaderboardItem = ({ 
  entry, 
  rank, 
  metric 
}: { 
  entry: LeaderboardEntry; 
  rank: number;
  metric: string;
}) => {
  const RankIcon = rankIcons[rank] || null;
  const rankColor = rankColors[rank] || 'text-muted-foreground';
  
  return (
    <Link to={`/user/${entry.user_id}`} className="flex items-center gap-3 py-2 hover:bg-muted/50 rounded-md px-1 -mx-1 transition-colors">
      <div className="w-6 text-center">
        {RankIcon ? (
          <RankIcon className={`h-5 w-5 ${rankColor}`} />
        ) : (
          <span className="text-sm font-medium text-muted-foreground">{rank + 1}</span>
        )}
      </div>
      
      <Avatar className="h-8 w-8">
        <AvatarFallback className="text-xs bg-primary/10 text-primary">
          {entry.author_name.substring(0, 2).toUpperCase()}
        </AvatarFallback>
      </Avatar>
      
      <div className="flex-1 min-w-0">
        <p className="font-medium truncate hover:text-primary transition-colors">{entry.author_name}</p>
      </div>
      
      <Badge variant="secondary" className="gap-1">
        {entry.count} {metric}
      </Badge>
    </Link>
  );
};

export const CommunityLeaderboard = () => {
  const [loading, setLoading] = useState(true);
  const [topPosters, setTopPosters] = useState<LeaderboardEntry[]>([]);
  const [topRepliers, setTopRepliers] = useState<LeaderboardEntry[]>([]);
  const [topLiked, setTopLiked] = useState<LeaderboardEntry[]>([]);

  useEffect(() => {
    fetchLeaderboardData();
  }, []);

  const fetchLeaderboardData = async () => {
    setLoading(true);
    try {
      const [postsData, repliesData, likedData] = await Promise.all([
        communityApi.getPosts(undefined, 500, 0),
        communityApi.getAllReplies(2000),
        communityApi.getPosts(undefined, 500, 0),
      ]);

      const postCounts = (postsData || []).reduce((acc: Record<string, LeaderboardEntry>, post: { author_name: string; user_id: string }) => {
        const key = post.user_id;
        if (!acc[key]) acc[key] = { author_name: post.author_name, user_id: post.user_id, count: 0 };
        acc[key].count++;
        return acc;
      }, {});
      setTopPosters(Object.values(postCounts).sort((a, b) => b.count - a.count).slice(0, 5));

      const replyCounts = (repliesData || []).reduce((acc: Record<string, LeaderboardEntry>, reply: { author_name: string; user_id: string }) => {
        const key = reply.user_id;
        if (!acc[key]) acc[key] = { author_name: reply.author_name, user_id: reply.user_id, count: 0 };
        acc[key].count++;
        return acc;
      }, {});
      setTopRepliers(Object.values(replyCounts).sort((a, b) => b.count - a.count).slice(0, 5));

      const likeCounts = (likedData || []).reduce((acc: Record<string, LeaderboardEntry>, post: { author_name: string; user_id: string; likes_count?: number }) => {
        const key = post.user_id;
        if (!acc[key]) acc[key] = { author_name: post.author_name, user_id: post.user_id, count: 0 };
        acc[key].count += post.likes_count || 0;
        return acc;
      }, {});
      setTopLiked(Object.values(likeCounts).filter(e => e.count > 0).sort((a, b) => b.count - a.count).slice(0, 5));
    } catch (e) {
      console.error('Error fetching leaderboard:', e);
    }

    setLoading(false);
  };

  if (loading) {
    return (
      <Card>
        <CardContent className="p-8 flex justify-center">
          <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
        </CardContent>
      </Card>
    );
  }

  const hasData = topPosters.length > 0 || topRepliers.length > 0 || topLiked.length > 0;

  if (!hasData) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg flex items-center gap-2">
            <Trophy className="h-5 w-5 text-amber-500" />
            Leaderboard
          </CardTitle>
        </CardHeader>
        <CardContent className="text-center py-6">
          <Trophy className="h-10 w-10 mx-auto text-muted-foreground mb-2" />
          <p className="text-sm text-muted-foreground">
            No activity yet. Be the first to contribute!
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader className="pb-2">
        <CardTitle className="text-lg flex items-center gap-2">
          <Trophy className="h-5 w-5 text-amber-500" />
          Leaderboard
        </CardTitle>
      </CardHeader>
      <CardContent>
        <Tabs defaultValue="posts" className="w-full">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="posts" className="text-xs gap-1">
              <MessageSquare className="h-3 w-3" />
              Posts
            </TabsTrigger>
            <TabsTrigger value="replies" className="text-xs gap-1">
              <MessageSquare className="h-3 w-3" />
              Replies
            </TabsTrigger>
            <TabsTrigger value="likes" className="text-xs gap-1">
              <Heart className="h-3 w-3" />
              Liked
            </TabsTrigger>
          </TabsList>
          
          <TabsContent value="posts" className="mt-3">
            {topPosters.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-4">No posts yet</p>
            ) : (
              <div className="divide-y">
                {topPosters.map((entry, index) => (
                  <LeaderboardItem 
                    key={entry.user_id} 
                    entry={entry} 
                    rank={index}
                    metric="posts"
                  />
                ))}
              </div>
            )}
          </TabsContent>
          
          <TabsContent value="replies" className="mt-3">
            {topRepliers.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-4">No replies yet</p>
            ) : (
              <div className="divide-y">
                {topRepliers.map((entry, index) => (
                  <LeaderboardItem 
                    key={entry.user_id} 
                    entry={entry} 
                    rank={index}
                    metric="replies"
                  />
                ))}
              </div>
            )}
          </TabsContent>
          
          <TabsContent value="likes" className="mt-3">
            {topLiked.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-4">No likes yet</p>
            ) : (
              <div className="divide-y">
                {topLiked.map((entry, index) => (
                  <LeaderboardItem 
                    key={entry.user_id} 
                    entry={entry} 
                    rank={index}
                    metric="likes"
                  />
                ))}
              </div>
            )}
          </TabsContent>
        </Tabs>
      </CardContent>
    </Card>
  );
};
