import { CosmeticItem, getRiskLevelColor, getRiskLevelName, getTobinRating } from "@/data/cosmeticsDatabase";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { AlertTriangle, Shield, ShieldCheck, ShieldX } from "lucide-react";
import { FavoriteButton } from "./FavoriteButton";
import { ShoppingListButton } from "./ShoppingListButton";
import { PurchasedButton } from "./PurchasedButton";

interface CosmeticCardProps {
  cosmetic: CosmeticItem;
  onClick: () => void;
}

const getRiskIcon = (riskLevel: number) => {
  switch (riskLevel) {
    case 1: return <ShieldX className="h-4 w-4" />;
    case 2: return <AlertTriangle className="h-4 w-4" />;
    case 3: return <Shield className="h-4 w-4" />;
    case 4: return <ShieldCheck className="h-4 w-4" />;
    default: return null;
  }
};

export const CosmeticCard = ({ cosmetic, onClick }: CosmeticCardProps) => {
  const riskColor = getRiskLevelColor(cosmetic.riskLevel);
  const tobinRating = cosmetic.tobinScore ? getTobinRating(cosmetic.tobinScore.total) : null;
  
  return (
    <Card 
      className="cursor-pointer hover:shadow-lg transition-all duration-200 hover:-translate-y-1 border-l-4"
      style={{ borderLeftColor: `hsl(var(--${riskColor}))` }}
      onClick={onClick}
    >
      <CardContent className="p-4">
        <div className="flex items-start justify-between gap-2 mb-3">
          <div className="flex items-center gap-2">
            <span className="text-2xl">{cosmetic.emoji}</span>
            <h3 className="font-semibold text-sm leading-tight">{cosmetic.name}</h3>
          </div>
          <div className="flex items-center gap-1">
            <PurchasedButton itemId={cosmetic.id} itemType="cosmetic" size="sm" />
            <ShoppingListButton itemId={cosmetic.id} itemType="cosmetic" size="sm" />
            <FavoriteButton itemId={cosmetic.id} itemType="cosmetic" size="sm" />
          </div>
        </div>
        
        <div className="flex flex-wrap items-center gap-2 mb-3">
          <Badge variant="outline" className={`text-${riskColor} border-${riskColor} flex items-center gap-1`}>
            {getRiskIcon(cosmetic.riskLevel)}
            {getRiskLevelName(cosmetic.riskLevel)}
          </Badge>
          <Badge variant="secondary" className="text-xs capitalize">
            {cosmetic.category.replace("_", " ")}
          </Badge>
          {tobinRating && (
            <Badge 
              variant="outline" 
              className={`${tobinRating.color} ${tobinRating.bgColor} border-0 text-xs`}
            >
              TOBIN: {cosmetic.tobinScore!.total}/50
            </Badge>
          )}
        </div>

        {/* TOBIN Mini Display */}
        {cosmetic.tobinScore && (
          <div className="flex justify-between text-xs mb-3 px-1">
            <span className="text-red-500" title="Toxins">T:{cosmetic.tobinScore.T}</span>
            <span className="text-orange-500" title="Oxidative">O:{cosmetic.tobinScore.O}</span>
            <span className="text-purple-500" title="Biological">B:{cosmetic.tobinScore.B}</span>
            <span className="text-yellow-600" title="Inflammatory">I:{cosmetic.tobinScore.I}</span>
            <span className="text-blue-500" title="Nutrient">N:{cosmetic.tobinScore.N}</span>
          </div>
        )}
        
        <div className="space-y-2">
          <div>
            <p className="text-xs text-muted-foreground mb-1">Key Chemicals:</p>
            <div className="flex flex-wrap gap-1">
              {cosmetic.commonChemicals.slice(0, 2).map((chem, idx) => (
                <Badge key={idx} variant="outline" className="text-xs">
                  {chem.split(" ")[0]}
                </Badge>
              ))}
              {cosmetic.commonChemicals.length > 2 && (
                <Badge variant="outline" className="text-xs text-muted-foreground">
                  +{cosmetic.commonChemicals.length - 2}
                </Badge>
              )}
            </div>
          </div>
          
          {cosmetic.estrogenicProfile.hasXenoestrogens && (
            <div className="flex items-center gap-1 text-xs text-category-danger">
              <AlertTriangle className="h-3 w-3" />
              Contains xenoestrogens
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
};
