import { Button } from "@/components/ui/button";
import { 
  Sparkles, 
  Droplets, 
  Sun, 
  Wind, 
  Smile,
  SprayCan,
  Palette,
  Scissors
} from "lucide-react";

type CosmeticCategory = "skincare" | "makeup" | "haircare" | "fragrance" | "sunscreen" | "oral_care" | "deodorant" | "nail_care" | null;

interface CosmeticCategoryFilterProps {
  selectedCategory: CosmeticCategory;
  onChange: (category: CosmeticCategory) => void;
  selectedRisk: number | null;
  onRiskChange: (risk: number | null) => void;
}

const categories = [
  { id: null, label: "All", icon: Sparkles },
  { id: "skincare" as const, label: "Skincare", icon: Droplets },
  { id: "makeup" as const, label: "Makeup", icon: Palette },
  { id: "haircare" as const, label: "Haircare", icon: Wind },
  { id: "nail_care" as const, label: "Nail Care", icon: Scissors },
  { id: "fragrance" as const, label: "Fragrance", icon: SprayCan },
  { id: "sunscreen" as const, label: "Sunscreen", icon: Sun },
  { id: "oral_care" as const, label: "Oral Care", icon: Smile },
  { id: "deodorant" as const, label: "Deodorant", icon: Smile },
];

const riskLevels = [
  { id: null, label: "All Risks", color: "bg-muted" },
  { id: 4, label: "Safe", color: "bg-category-safe" },
  { id: 3, label: "Low Risk", color: "bg-category-therapeutic" },
  { id: 2, label: "Moderate", color: "bg-category-warning" },
  { id: 1, label: "High Risk", color: "bg-category-danger" },
];

export const CosmeticCategoryFilter = ({ 
  selectedCategory, 
  onChange,
  selectedRisk,
  onRiskChange 
}: CosmeticCategoryFilterProps) => {
  return (
    <div className="space-y-4">
      {/* Category Filter */}
      <div>
        <p className="text-sm text-muted-foreground mb-2">Product Type</p>
        <div className="flex flex-wrap gap-2">
          {categories.map((cat) => {
            const Icon = cat.icon;
            const isActive = selectedCategory === cat.id;
            return (
              <Button
                key={cat.id || "all"}
                variant={isActive ? "default" : "outline"}
                size="sm"
                onClick={() => onChange(cat.id)}
                className="gap-1.5"
              >
                <Icon className="h-4 w-4" />
                {cat.label}
              </Button>
            );
          })}
        </div>
      </div>
      
      {/* Risk Level Filter */}
      <div>
        <p className="text-sm text-muted-foreground mb-2">Risk Level</p>
        <div className="flex flex-wrap gap-2">
          {riskLevels.map((risk) => {
            const isActive = selectedRisk === risk.id;
            return (
              <Button
                key={risk.id || "all-risks"}
                variant={isActive ? "default" : "outline"}
                size="sm"
                onClick={() => onRiskChange(risk.id)}
                className="gap-1.5"
              >
                <span className={`w-2 h-2 rounded-full ${risk.color}`} />
                {risk.label}
              </Button>
            );
          })}
        </div>
      </div>
    </div>
  );
};
