import { CosmeticItem, getRiskLevelColor, getRiskLevelName, TobinScore, getTobinRating } from "@/data/cosmeticsDatabase";
import { getEstrogenicPotencyColor } from "@/data/chemicalDatabase";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { TobinScoreDisplay } from "@/components/TobinScoreDisplay";
import { 
  AlertTriangle, 
  Shield, 
  ShieldCheck, 
  ShieldX, 
  FlaskConical,
  Heart,
  Leaf,
  Info,
  CircleAlert
} from "lucide-react";

interface CosmeticDetailModalProps {
  cosmetic: CosmeticItem | null;
  open: boolean;
  onClose: () => void;
}

const getRiskIcon = (riskLevel: number) => {
  switch (riskLevel) {
    case 1: return <ShieldX className="h-5 w-5" />;
    case 2: return <AlertTriangle className="h-5 w-5" />;
    case 3: return <Shield className="h-5 w-5" />;
    case 4: return <ShieldCheck className="h-5 w-5" />;
    default: return null;
  }
};

export const CosmeticDetailModal = ({ cosmetic, open, onClose }: CosmeticDetailModalProps) => {
  if (!cosmetic) return null;
  
  const riskColor = getRiskLevelColor(cosmetic.riskLevel);
  const potencyColor = getEstrogenicPotencyColor(cosmetic.estrogenicProfile.estrogenicPotency);
  
  return (
    <Dialog open={open} onOpenChange={onClose}>
      <DialogContent className="flex max-h-[90dvh] w-[calc(100vw-2rem)] max-w-2xl flex-col gap-0 overflow-hidden p-4 sm:w-full sm:p-6">
        <DialogHeader className="shrink-0">
          <div className="flex items-center gap-3">
            <span className="text-3xl sm:text-4xl">{cosmetic.emoji}</span>
            <div className="min-w-0 flex-1">
              <DialogTitle className="text-lg break-words sm:text-xl">{cosmetic.name}</DialogTitle>
              <div className="flex items-center gap-2 mt-1">
                <Badge className={`bg-${riskColor} text-white flex items-center gap-1`}>
                  {getRiskIcon(cosmetic.riskLevel)}
                  {getRiskLevelName(cosmetic.riskLevel)}
                </Badge>
                <Badge variant="secondary" className="capitalize">
                  {cosmetic.category.replace("_", " ")}
                </Badge>
              </div>
            </div>
          </div>
        </DialogHeader>
        
        <div className="min-h-0 flex-1 overflow-y-auto overscroll-contain max-h-[min(70vh,50rem)] pr-2 sm:pr-4">
          <div className="space-y-6 pb-2">
            {/* TOBIN Score Section */}
            {cosmetic.tobinScore && (
              <>
                <TobinScoreDisplay tobinScore={cosmetic.tobinScore} />
                <Separator />
              </>
            )}
            
            {/* Usage Warning */}
            {cosmetic.usageWarning && (
              <div className="bg-destructive/10 border border-destructive/30 rounded-lg p-3 flex items-start gap-2">
                <CircleAlert className="h-5 w-5 text-destructive flex-shrink-0 mt-0.5" />
                <p className="text-sm text-destructive break-words">{cosmetic.usageWarning}</p>
              </div>
            )}
            
            {/* Chemical Details */}
            <div>
              <h3 className="font-semibold flex items-center gap-2 mb-3">
                <FlaskConical className="h-4 w-4 text-primary" />
                Chemical Breakdown
              </h3>
              <div className="space-y-3">
                {cosmetic.chemicalDetails.map((detail, idx) => (
                  <div key={idx} className="bg-muted/50 rounded-lg p-3">
                    <div className="flex items-center justify-between mb-1">
                      <span className="font-medium text-sm">{detail.chemical}</span>
                      <Badge variant="outline" className="text-xs">{detail.purpose}</Badge>
                    </div>
                    <p className="text-xs text-muted-foreground">{detail.concern}</p>
                  </div>
                ))}
              </div>
            </div>
            
            <Separator />
            
            {/* Estrogenic Profile */}
            <div>
              <h3 className="font-semibold flex items-center gap-2 mb-3">
                <AlertTriangle className="h-4 w-4 text-warning" />
                Estrogenic Profile
              </h3>
              <div className="grid grid-cols-2 gap-3 mb-3">
                <div className="bg-muted/50 rounded-lg p-3 text-center">
                  <p className="text-xs text-muted-foreground mb-1">Xenoestrogens</p>
                  <Badge variant={cosmetic.estrogenicProfile.hasXenoestrogens ? "destructive" : "secondary"}>
                    {cosmetic.estrogenicProfile.hasXenoestrogens ? "Present" : "None"}
                  </Badge>
                </div>
                <div className="bg-muted/50 rounded-lg p-3 text-center">
                  <p className="text-xs text-muted-foreground mb-1">Potency</p>
                  <Badge className={potencyColor}>
                    {cosmetic.estrogenicProfile.estrogenicPotency.toUpperCase()}
                  </Badge>
                </div>
              </div>
              
              {cosmetic.estrogenicProfile.xenoestrogens && cosmetic.estrogenicProfile.xenoestrogens.length > 0 && (
                <div className="mb-3">
                  <p className="text-xs text-muted-foreground mb-1">Xenoestrogen compounds:</p>
                  <div className="flex flex-wrap gap-1">
                    {cosmetic.estrogenicProfile.xenoestrogens.map((xe, idx) => (
                      <Badge key={idx} variant="outline" className="text-xs text-category-danger border-category-danger">
                        {xe}
                      </Badge>
                    ))}
                  </div>
                </div>
              )}
              
              {cosmetic.estrogenicProfile.healthConcerns && cosmetic.estrogenicProfile.healthConcerns.length > 0 && (
                <div>
                  <p className="text-xs text-muted-foreground mb-1">Health concerns:</p>
                  <ul className="text-sm space-y-1">
                    {cosmetic.estrogenicProfile.healthConcerns.map((concern, idx) => (
                      <li key={idx} className="flex items-start gap-2">
                        <span className="text-category-danger">•</span>
                        {concern}
                      </li>
                    ))}
                  </ul>
                </div>
              )}
            </div>
            
            <Separator />
            
            {/* Health Effects */}
            <div>
              <h3 className="font-semibold flex items-center gap-2 mb-3">
                <Heart className="h-4 w-4 text-destructive" />
                Health Effects
              </h3>
              <ul className="space-y-2">
                {cosmetic.healthEffects.map((effect, idx) => (
                  <li key={idx} className="flex items-start gap-2 text-sm">
                    <CircleAlert className="h-4 w-4 text-muted-foreground flex-shrink-0 mt-0.5" />
                    {effect}
                  </li>
                ))}
              </ul>
            </div>
            
            {/* Safer Alternatives */}
            {cosmetic.saferAlternatives && cosmetic.saferAlternatives.length > 0 && (
              <>
                <Separator />
                <div>
                  <h3 className="font-semibold flex items-center gap-2 mb-3">
                    <Leaf className="h-4 w-4 text-category-safe" />
                    Safer Alternatives
                  </h3>
                  <div className="flex flex-wrap gap-2">
                    {cosmetic.saferAlternatives.map((alt, idx) => (
                      <Badge key={idx} variant="outline" className="text-category-safe border-category-safe">
                        {alt}
                      </Badge>
                    ))}
                  </div>
                </div>
              </>
            )}
            
            {/* Regulatory Notes */}
            {cosmetic.regulatoryNotes && (
              <>
                <Separator />
                <div className="bg-muted/50 rounded-lg p-3 flex items-start gap-2">
                  <Info className="h-4 w-4 text-primary flex-shrink-0 mt-0.5" />
                  <p className="text-sm text-muted-foreground break-words">{cosmetic.regulatoryNotes}</p>
                </div>
              </>
            )}
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
};
