import { useState, useMemo } from 'react';
import { Link } from 'react-router-dom';
import { useAuth } from '@/hooks/useAuth';
import { useFavorites } from '@/contexts/FavoritesContext';
import { useShoppingList } from '@/contexts/ShoppingListContext';
import { useScanHistory, ScanHistoryItem, ScanProductDetails } from '@/hooks/useScanHistory';
import { cosmeticsDatabase, CosmeticItem } from '@/data/cosmeticsDatabase';
import { CosmeticCard } from '@/components/CosmeticCard';
import { ScannedProductMiniCard, ScannedProduct, getScannedProductKey } from '@/components/ScannedProductMiniCard';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Heart, ShoppingCart, History, Star, ArrowRight, ScanLine, Sparkles, CheckCircle2, Filter } from 'lucide-react';
import { CosmeticDetailModal } from '@/components/CosmeticDetailModal';
import { ScannedProductDetailModal } from '@/components/ScannedProductDetailModal';

type CosmeticCategory = CosmeticItem["category"] | null;

interface CosmeticsDashboardProps {
  selectedCategory?: CosmeticCategory;
  selectedRisk?: number | null;
}

// Helper to get risk level from AI safety analysis
function getRiskLevelFromSafety(safetyLevel?: string): number | null {
  switch (safetyLevel) {
    case 'safe': return 4;
    case 'low_concern': return 3;
    case 'moderate_concern': return 2;
    case 'high_concern': return 1;
    default: return null;
  }
}

export function CosmeticsDashboard({ selectedCategory, selectedRisk }: CosmeticsDashboardProps) {
  const { user } = useAuth();
  const { favorites, loading: favoritesLoading } = useFavorites();
  const { items: shoppingItems, loading: shoppingLoading } = useShoppingList();
  const { history: scanHistory, loading: scanLoading } = useScanHistory();
  
  const [selectedCosmetic, setSelectedCosmetic] = useState<CosmeticItem | null>(null);
  const [selectedScannedProduct, setSelectedScannedProduct] = useState<ScanHistoryItem | null>(null);

  const hasFilters = selectedCategory !== null || selectedRisk !== null;

  // Filter for cosmetic products only
  const cosmeticFavorites = useMemo(() => 
    favorites.filter(f => f.item_type === 'cosmetic'), [favorites]);
  
  const cosmeticShopping = useMemo(() => 
    shoppingItems.filter(i => i.item_type === 'cosmetic'), [shoppingItems]);
  
  const cosmeticScans = useMemo(() => 
    scanHistory.filter(s => s.product_type === 'cosmetic'), [scanHistory]);

  // Helper to find product in local DB or scan history
  const resolveProduct = useMemo(() => {
    return (itemId: string, quantity?: number) => {
      // Try local cosmetics database first
      const localCosmetic = cosmeticsDatabase.find(c => c.id === itemId);
      if (localCosmetic) return { type: 'local-cosmetic' as const, item: localCosmetic, quantity };
      
      // Check ALL scan history for external products (by barcode or product_id)
      // Filter for cosmetic products when found
      const scannedProduct = scanHistory.find(s => 
        (s.barcode === itemId || s.product_id === itemId) && s.product_type === 'cosmetic'
      );
      if (scannedProduct) {
        return {
          type: 'scanned' as const,
          item: {
            id: scannedProduct.product_id,
            product_name: scannedProduct.product_name,
            product_type: 'cosmetic' as const,
            image_url: scannedProduct.image_url,
            barcode: scannedProduct.barcode,
            isScanned: true as const,
            quantity,
            scanHistoryItem: scannedProduct
          } satisfies ScannedProduct
        };
      }
      
      return null;
    };
  }, [scanHistory]);

  // Filter functions for category and risk
  const matchesLocalCategory = (cosmetic: CosmeticItem): boolean => {
    if (!selectedCategory) return true;
    return cosmetic.category === selectedCategory;
  };

  const matchesLocalRisk = (cosmetic: CosmeticItem): boolean => {
    if (selectedRisk === null) return true;
    return cosmetic.riskLevel === selectedRisk;
  };

  const matchesScannedRisk = (product: ScannedProduct): boolean => {
    if (selectedRisk === null) return true;
    const safetyLevel = product.scanHistoryItem?.product_details?.aiAnalysis?.safetyLevel;
    const riskLevel = getRiskLevelFromSafety(safetyLevel);
    return riskLevel === selectedRisk;
  };

  if (!user) {
    return null;
  }

  // Get favorite cosmetics with filtering
  const resolvedFavorites = cosmeticFavorites.map(f => resolveProduct(f.item_id)).filter(Boolean);
  const favoriteLocalCosmetics = resolvedFavorites
    .filter(r => r?.type === 'local-cosmetic')
    .map(r => r!.item as CosmeticItem)
    .filter(c => matchesLocalCategory(c) && matchesLocalRisk(c));
  const favoriteScannedProducts = resolvedFavorites
    .filter(r => r?.type === 'scanned')
    .map(r => r!.item as ScannedProduct)
    .filter(p => matchesScannedRisk(p));

  // Get shopping list items - To Buy with filtering
  const toBuyItems = cosmeticShopping.filter(i => !i.is_purchased);
  const resolvedShopping = toBuyItems.map(i => resolveProduct(i.item_id, i.quantity)).filter(Boolean);
  const shoppingLocalCosmetics = resolvedShopping
    .filter(r => r?.type === 'local-cosmetic')
    .map(r => ({ ...r!.item as CosmeticItem, quantity: r!.quantity || 1 }))
    .filter(c => matchesLocalCategory(c) && matchesLocalRisk(c));
  const shoppingScannedProducts = resolvedShopping
    .filter(r => r?.type === 'scanned')
    .map(r => r!.item as ScannedProduct)
    .filter(p => matchesScannedRisk(p));

  // Get purchased items with filtering
  const purchasedItems = cosmeticShopping.filter(i => i.is_purchased);
  const resolvedPurchased = purchasedItems.map(i => resolveProduct(i.item_id, i.quantity)).filter(Boolean);
  const purchasedLocalCosmetics = resolvedPurchased
    .filter(r => r?.type === 'local-cosmetic')
    .map(r => ({ ...r!.item as CosmeticItem, quantity: r!.quantity || 1 }))
    .filter(c => matchesLocalCategory(c) && matchesLocalRisk(c));
  const purchasedScannedProducts = resolvedPurchased
    .filter(r => r?.type === 'scanned')
    .map(r => r!.item as ScannedProduct)
    .filter(p => matchesScannedRisk(p));

  // Get recent cosmetic scans with filtering
  const recentScans = cosmeticScans
    .map(scan => {
      const cosmetic = cosmeticsDatabase.find(c => c.id === scan.product_id);
      return { item: cosmetic, scannedAt: scan.scanned_at, scanData: scan };
    })
    .filter(scan => {
      if (scan.item) return matchesLocalCategory(scan.item) && matchesLocalRisk(scan.item);
      if (selectedRisk !== null) {
        const safetyLevel = scan.scanData.product_details?.aiAnalysis?.safetyLevel;
        const riskLevel = getRiskLevelFromSafety(safetyLevel);
        return riskLevel === selectedRisk;
      }
      return true;
    })
    .slice(0, 8);

  const filteredFavoritesCount = favoriteLocalCosmetics.length + favoriteScannedProducts.length;
  const filteredShoppingCount = shoppingLocalCosmetics.length + shoppingScannedProducts.length;
  const filteredScansCount = recentScans.length;
  const totalFilteredItems = filteredFavoritesCount + filteredShoppingCount + filteredScansCount;
  const totalItems = cosmeticFavorites.length + toBuyItems.length + cosmeticScans.length;

  const handleScannedProductClick = (item: ScanHistoryItem) => {
    setSelectedScannedProduct(item);
  };

  const isLoading = favoritesLoading || shoppingLoading || scanLoading;

  if (isLoading) {
    return (
      <Card className="animate-pulse">
        <CardHeader>
          <div className="h-6 w-48 bg-muted rounded" />
          <div className="h-4 w-32 bg-muted rounded mt-2" />
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            {[1, 2, 3, 4].map(i => (
              <div key={i} className="h-32 bg-muted rounded-lg" />
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }

  if (totalItems === 0) {
    return (
      <Card className="border-dashed border-2 border-primary/20 bg-gradient-to-br from-primary/5 to-transparent">
        <CardContent className="flex flex-col items-center justify-center py-12">
          <div className="rounded-full bg-primary/10 p-4 mb-4">
            <Sparkles className="h-8 w-8 text-primary" />
          </div>
          <h3 className="text-xl font-semibold mb-2">Start Building Your Cosmetics Collection</h3>
          <p className="text-muted-foreground text-center mb-6 max-w-md">
            Scan cosmetics, save favorites, and create shopping lists. Your personalized cosmetics dashboard will appear here!
          </p>
          <div className="flex flex-wrap gap-3 justify-center">
            <Link to="/barcode-scanner">
              <Button className="gap-2">
                <ScanLine className="h-4 w-4" />
                Scan a Cosmetic
              </Button>
            </Link>
            <Link to="/ingredient-scanner">
              <Button variant="outline" className="gap-2">
                <Heart className="h-4 w-4" />
                Scan Ingredients
              </Button>
            </Link>
          </div>
        </CardContent>
      </Card>
    );
  }

  // Show empty state when filters match nothing but there are items
  if (hasFilters && totalFilteredItems === 0 && totalItems > 0) {
    return (
      <Card className="border-dashed border-2 border-muted bg-muted/5">
        <CardContent className="flex flex-col items-center justify-center py-12">
          <div className="rounded-full bg-muted p-4 mb-4">
            <Filter className="h-8 w-8 text-muted-foreground" />
          </div>
          <h3 className="text-lg font-semibold mb-2">No matching products</h3>
          <p className="text-muted-foreground text-center max-w-md">
            None of your {totalItems} tracked cosmetics match the current filters. Try adjusting the category or risk level filters above.
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <>
      <Card className="overflow-hidden">
        <CardHeader className="bg-gradient-to-r from-primary/5 via-transparent to-transparent">
          <div className="flex items-center justify-between flex-wrap gap-2">
            <div>
              <CardTitle className="flex items-center gap-2">
                <Star className="h-5 w-5 text-primary" />
                Your Cosmetics Collection
                {hasFilters && (
                  <Badge variant="outline" className="ml-2 gap-1 text-xs">
                    <Filter className="h-3 w-3" />
                    Filtered
                  </Badge>
                )}
              </CardTitle>
              <CardDescription>
                {hasFilters 
                  ? `${totalFilteredItems} of ${totalItems} items match filters`
                  : `${totalItems} items tracked • Favorites, scans & shopping list`
                }
              </CardDescription>
            </div>
            <div className="flex gap-2">
              <Badge variant="secondary" className="gap-1">
                <Heart className="h-3 w-3" /> {hasFilters ? filteredFavoritesCount : cosmeticFavorites.length}
              </Badge>
              <Badge variant="secondary" className="gap-1">
                <ShoppingCart className="h-3 w-3" /> {hasFilters ? filteredShoppingCount : toBuyItems.length}
              </Badge>
              <Badge variant="secondary" className="gap-1">
                <History className="h-3 w-3" /> {hasFilters ? filteredScansCount : cosmeticScans.length}
              </Badge>
            </div>
          </div>
        </CardHeader>
        <CardContent className="pt-6">
          <Tabs defaultValue="favorites" className="w-full">
            <TabsList className="grid w-full grid-cols-3 mb-6">
              <TabsTrigger value="favorites" className="gap-2">
                <Heart className="h-4 w-4" />
                <span className="hidden sm:inline">Favorites</span>
                <Badge variant="secondary" className="ml-1">{hasFilters ? filteredFavoritesCount : cosmeticFavorites.length}</Badge>
              </TabsTrigger>
              <TabsTrigger value="shopping" className="gap-2">
                <ShoppingCart className="h-4 w-4" />
                <span className="hidden sm:inline">Shopping</span>
                <Badge variant="secondary" className="ml-1">{hasFilters ? filteredShoppingCount : toBuyItems.length}</Badge>
              </TabsTrigger>
              <TabsTrigger value="scans" className="gap-2">
                <History className="h-4 w-4" />
                <span className="hidden sm:inline">Recent Scans</span>
                <Badge variant="secondary" className="ml-1">{hasFilters ? filteredScansCount : cosmeticScans.length}</Badge>
              </TabsTrigger>
            </TabsList>

            <TabsContent value="favorites" className="space-y-4">
              {cosmeticFavorites.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <Heart className="h-12 w-12 mx-auto mb-3 opacity-30" />
                  <p>No favorite cosmetics yet. Browse products and tap the heart icon to save them!</p>
                </div>
              ) : (
                <>
                  {favoriteLocalCosmetics.length > 0 && (
                    <div>
                      <h4 className="text-sm font-medium text-muted-foreground mb-3">Reference Products</h4>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                        {favoriteLocalCosmetics.slice(0, 4).map(cosmetic => (
                          <CosmeticCard 
                            key={cosmetic.id} 
                            cosmetic={cosmetic} 
                            onClick={() => setSelectedCosmetic(cosmetic)}
                          />
                        ))}
                      </div>
                      {favoriteLocalCosmetics.length > 4 && (
                        <Button variant="ghost" className="mt-3 gap-2">
                          View all {favoriteLocalCosmetics.length} reference products <ArrowRight className="h-4 w-4" />
                        </Button>
                      )}
                    </div>
                  )}
                  {favoriteScannedProducts.length > 0 && (
                    <div className="mt-6">
                      <h4 className="text-sm font-medium text-muted-foreground mb-3">Scanned Products</h4>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                        {favoriteScannedProducts.slice(0, 4).map(product => (
                          <ScannedProductMiniCard 
                            key={getScannedProductKey(product)} 
                            product={product} 
                            onProductClick={handleScannedProductClick}
                          />
                        ))}
                      </div>
                      {favoriteScannedProducts.length > 4 && (
                        <Button variant="ghost" className="mt-3 gap-2">
                          View all {favoriteScannedProducts.length} scanned products <ArrowRight className="h-4 w-4" />
                        </Button>
                      )}
                    </div>
                  )}
                </>
              )}
            </TabsContent>

            <TabsContent value="shopping" className="space-y-4">
              {toBuyItems.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <ShoppingCart className="h-12 w-12 mx-auto mb-3 opacity-30" />
                  <p>Your cosmetics shopping list is empty. Add items from product pages!</p>
                  <Link to="/shopping-list">
                    <Button variant="outline" className="mt-4">View Shopping List</Button>
                  </Link>
                </div>
              ) : (
                <>
                  {shoppingLocalCosmetics.length > 0 && (
                    <div>
                      <h4 className="text-sm font-medium text-muted-foreground mb-3">Reference Products to Buy</h4>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                        {shoppingLocalCosmetics.slice(0, 4).map(cosmetic => (
                          <div key={cosmetic.id} className="relative">
                            <CosmeticCard 
                              cosmetic={cosmetic} 
                              onClick={() => setSelectedCosmetic(cosmetic)}
                            />
                            {cosmetic.quantity > 1 && (
                              <Badge className="absolute top-2 right-2 bg-primary">
                                ×{cosmetic.quantity}
                              </Badge>
                            )}
                          </div>
                        ))}
                      </div>
                    </div>
                  )}
                  {shoppingScannedProducts.length > 0 && (
                    <div className="mt-6">
                      <h4 className="text-sm font-medium text-muted-foreground mb-3">Scanned Products to Buy</h4>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                        {shoppingScannedProducts.slice(0, 4).map(product => (
                          <ScannedProductMiniCard 
                            key={getScannedProductKey(product)} 
                            product={product} 
                            showQuantity 
                            onProductClick={handleScannedProductClick}
                          />
                        ))}
                      </div>
                      {shoppingScannedProducts.length > 4 && (
                        <Link to="/shopping-list">
                          <Button variant="ghost" className="mt-3 gap-2">
                            View all {shoppingScannedProducts.length} scanned products <ArrowRight className="h-4 w-4" />
                          </Button>
                        </Link>
                      )}
                    </div>
                  )}
                  <div className="flex justify-center mt-4">
                    <Link to="/shopping-list">
                      <Button variant="outline" className="gap-2">
                        <ShoppingCart className="h-4 w-4" />
                        View Full Shopping List
                      </Button>
                    </Link>
                  </div>
                </>
              )}
              
              {/* Purchased Section */}
              {purchasedItems.length > 0 && (
                <div className="mt-8 pt-6 border-t">
                  <div className="flex items-center gap-2 mb-4">
                    <CheckCircle2 className="h-5 w-5 text-category-safe" />
                    <h3 className="font-medium">Recently Purchased</h3>
                    <Badge variant="secondary" className="ml-auto">{purchasedItems.length}</Badge>
                  </div>
                  
                  <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                    {purchasedLocalCosmetics.slice(0, 2).map(cosmetic => (
                      <div key={cosmetic.id} className="relative opacity-75">
                        <CosmeticCard 
                          cosmetic={cosmetic} 
                          onClick={() => setSelectedCosmetic(cosmetic)}
                        />
                        <Badge className="absolute top-2 right-2 bg-category-safe gap-1">
                          <CheckCircle2 className="h-3 w-3" />
                          Purchased
                        </Badge>
                      </div>
                    ))}
                    {purchasedScannedProducts.slice(0, 2).map(product => (
                      <div key={getScannedProductKey(product)} className="relative opacity-75">
                        <ScannedProductMiniCard 
                          product={product} 
                          onProductClick={handleScannedProductClick}
                        />
                      </div>
                    ))}
                  </div>
                  
                  {purchasedItems.length > 4 && (
                    <Link to="/shopping-list">
                      <Button variant="ghost" className="mt-3 gap-2 text-muted-foreground">
                        View all {purchasedItems.length} purchased items <ArrowRight className="h-4 w-4" />
                      </Button>
                    </Link>
                  )}
                </div>
              )}
            </TabsContent>

            <TabsContent value="scans" className="space-y-4">
              {cosmeticScans.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <ScanLine className="h-12 w-12 mx-auto mb-3 opacity-30" />
                  <p>No cosmetic scans yet. Use the barcode scanner to analyze products!</p>
                  <Link to="/barcode-scanner">
                    <Button variant="outline" className="mt-4 gap-2">
                      <ScanLine className="h-4 w-4" />
                      Scan a Cosmetic
                    </Button>
                  </Link>
                </div>
              ) : (
                <>
                  <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                    {recentScans.map((scan) => {
                      if (scan.item) {
                        return (
                          <CosmeticCard
                            key={`cosmetic-${scan.scanData.barcode}`}
                            cosmetic={scan.item}
                            onClick={() => setSelectedCosmetic(scan.item!)}
                          />
                        );
                      }
                      return (
                        <ScannedProductMiniCard
                          key={scan.scanData.barcode}
                          product={{
                            id: scan.scanData.product_id,
                            product_name: scan.scanData.product_name,
                            product_type: 'cosmetic',
                            image_url: scan.scanData.image_url,
                            barcode: scan.scanData.barcode,
                            isScanned: true,
                            scanHistoryItem: scan.scanData
                          }}
                          onProductClick={handleScannedProductClick}
                        />
                      );
                    })}
                  </div>
                  {cosmeticScans.length > 8 && (
                    <div className="flex justify-center mt-4">
                      <Link to="/barcode-scanner">
                        <Button variant="ghost" className="gap-2">
                          View all {cosmeticScans.length} scans <ArrowRight className="h-4 w-4" />
                        </Button>
                      </Link>
                    </div>
                  )}
                </>
              )}
            </TabsContent>

          </Tabs>
        </CardContent>
      </Card>

      {/* Modals */}
      <CosmeticDetailModal
        cosmetic={selectedCosmetic}
        open={!!selectedCosmetic}
        onClose={() => setSelectedCosmetic(null)}
      />
      <ScannedProductDetailModal
        item={selectedScannedProduct}
        open={!!selectedScannedProduct}
        onClose={() => setSelectedScannedProduct(null)}
      />
    </>
  );
}
