import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Leaf, Star, AlertTriangle } from "lucide-react";

export interface DietaryFilters {
  halal: "all" | "halal" | "haram" | "mashbooh";
  kosher: "all" | "kosher" | "not_kosher" | "pareve";
  organic: "all" | "organic" | "non_organic" | "varies";
  phytoestrogen: "all" | "hide" | "low_only" | "show_estrogenic";
}

interface DietaryFilterProps {
  filters: DietaryFilters;
  onChange: (filters: DietaryFilters) => void;
}

const halalOptions = [
  { id: "all" as const, label: "All", color: "bg-muted" },
  { id: "halal" as const, label: "Halal", color: "bg-category-safe" },
  { id: "mashbooh" as const, label: "Mashbooh", color: "bg-category-warning" },
  { id: "haram" as const, label: "Haram", color: "bg-category-danger" },
];

const kosherOptions = [
  { id: "all" as const, label: "All", color: "bg-muted" },
  { id: "kosher" as const, label: "Kosher", color: "bg-category-safe" },
  { id: "pareve" as const, label: "Pareve", color: "bg-category-therapeutic" },
  { id: "not_kosher" as const, label: "Not Kosher", color: "bg-category-danger" },
];

const organicOptions = [
  { id: "all" as const, label: "All", color: "bg-muted" },
  { id: "organic" as const, label: "Organic", color: "bg-category-safe" },
  { id: "varies" as const, label: "Varies", color: "bg-category-warning" },
  { id: "non_organic" as const, label: "Non-Organic", color: "bg-muted-foreground" },
];

const phytoestrogenOptions = [
  { id: "all" as const, label: "All", color: "bg-muted" },
  { id: "hide" as const, label: "Hide Estrogenic", color: "bg-category-safe" },
  { id: "low_only" as const, label: "Low Only", color: "bg-category-warning" },
  { id: "show_estrogenic" as const, label: "Only Estrogenic", color: "bg-pink-500" },
];

export function DietaryFilter({ filters, onChange }: DietaryFilterProps) {
  return (
    <div className="bg-card border rounded-lg sm:rounded-xl p-3 sm:p-4 space-y-3 sm:space-y-4">
      <h3 className="font-semibold text-xs sm:text-sm text-muted-foreground">Dietary Filters</h3>
      
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-3 sm:gap-4">
        {/* Halal Filter */}
        <div className="space-y-1.5 sm:space-y-2">
          <div className="flex items-center gap-1 sm:gap-1.5 text-xs sm:text-sm font-medium">
            <Star className="h-3.5 w-3.5 sm:h-4 sm:w-4 text-category-safe" />
            Halal Status
          </div>
          <div className="flex flex-wrap gap-1 sm:gap-1.5">
            {halalOptions.map((option) => (
              <Button
                key={option.id}
                variant={filters.halal === option.id ? "default" : "outline"}
                size="sm"
                onClick={() => onChange({ ...filters, halal: option.id })}
                className="h-6 sm:h-7 text-[10px] sm:text-xs px-1.5 sm:px-2"
              >
                <span className={`w-1.5 h-1.5 sm:w-2 sm:h-2 rounded-full ${option.color} mr-0.5 sm:mr-1`} />
                {option.label}
              </Button>
            ))}
          </div>
        </div>
        
        {/* Kosher Filter */}
        <div className="space-y-1.5 sm:space-y-2">
          <div className="flex items-center gap-1 sm:gap-1.5 text-xs sm:text-sm font-medium">
            <Star className="h-3.5 w-3.5 sm:h-4 sm:w-4 text-primary" />
            Kosher Status
          </div>
          <div className="flex flex-wrap gap-1 sm:gap-1.5">
            {kosherOptions.map((option) => (
              <Button
                key={option.id}
                variant={filters.kosher === option.id ? "default" : "outline"}
                size="sm"
                onClick={() => onChange({ ...filters, kosher: option.id })}
                className="h-6 sm:h-7 text-[10px] sm:text-xs px-1.5 sm:px-2"
              >
                <span className={`w-1.5 h-1.5 sm:w-2 sm:h-2 rounded-full ${option.color} mr-0.5 sm:mr-1`} />
                {option.label}
              </Button>
            ))}
          </div>
        </div>
        
        {/* Organic Filter */}
        <div className="space-y-1.5 sm:space-y-2">
          <div className="flex items-center gap-1 sm:gap-1.5 text-xs sm:text-sm font-medium">
            <Leaf className="h-3.5 w-3.5 sm:h-4 sm:w-4 text-category-therapeutic" />
            Organic Status
          </div>
          <div className="flex flex-wrap gap-1 sm:gap-1.5">
            {organicOptions.map((option) => (
              <Button
                key={option.id}
                variant={filters.organic === option.id ? "default" : "outline"}
                size="sm"
                onClick={() => onChange({ ...filters, organic: option.id })}
                className="h-6 sm:h-7 text-[10px] sm:text-xs px-1.5 sm:px-2"
              >
                <span className={`w-1.5 h-1.5 sm:w-2 sm:h-2 rounded-full ${option.color} mr-0.5 sm:mr-1`} />
                {option.label}
              </Button>
            ))}
          </div>
        </div>
        
        {/* Phytoestrogen Filter */}
        <div className="space-y-1.5 sm:space-y-2">
          <div className="flex items-center gap-1 sm:gap-1.5 text-xs sm:text-sm font-medium">
            <AlertTriangle className="h-3.5 w-3.5 sm:h-4 sm:w-4 text-pink-500" />
            Phytoestrogens
          </div>
          <div className="flex flex-wrap gap-1 sm:gap-1.5">
            {phytoestrogenOptions.map((option) => (
              <Button
                key={option.id}
                variant={filters.phytoestrogen === option.id ? "default" : "outline"}
                size="sm"
                onClick={() => onChange({ ...filters, phytoestrogen: option.id })}
                className="h-6 sm:h-7 text-[10px] sm:text-xs px-1.5 sm:px-2"
              >
                <span className={`w-1.5 h-1.5 sm:w-2 sm:h-2 rounded-full ${option.color} mr-0.5 sm:mr-1`} />
                {option.label}
              </Button>
            ))}
          </div>
        </div>
      </div>
      
      {/* Active Filters Display */}
      {(filters.halal !== "all" || filters.kosher !== "all" || filters.organic !== "all" || filters.phytoestrogen !== "all") && (
        <div className="flex items-center gap-2 pt-2 border-t">
          <span className="text-xs text-muted-foreground">Active:</span>
          <div className="flex flex-wrap gap-1.5">
            {filters.halal !== "all" && (
              <Badge variant="secondary" className="text-xs">
                {filters.halal === "mashbooh" ? "Mashbooh" : filters.halal.charAt(0).toUpperCase() + filters.halal.slice(1)}
              </Badge>
            )}
            {filters.kosher !== "all" && (
              <Badge variant="secondary" className="text-xs">
                {filters.kosher === "not_kosher" ? "Not Kosher" : filters.kosher.charAt(0).toUpperCase() + filters.kosher.slice(1)}
              </Badge>
            )}
            {filters.organic !== "all" && (
              <Badge variant="secondary" className="text-xs">
                {filters.organic === "non_organic" ? "Non-Organic" : filters.organic.charAt(0).toUpperCase() + filters.organic.slice(1)}
              </Badge>
            )}
            {filters.phytoestrogen !== "all" && (
              <Badge variant="secondary" className="text-xs bg-pink-100 text-pink-700">
                {filters.phytoestrogen === "hide" ? "No Estrogenic" : 
                 filters.phytoestrogen === "low_only" ? "Low Estrogen" : "Estrogenic Only"}
              </Badge>
            )}
          </div>
          <Button
            variant="ghost"
            size="sm"
            className="h-6 text-xs ml-auto"
            onClick={() => onChange({ halal: "all", kosher: "all", organic: "all", phytoestrogen: "all" })}
          >
            Clear All
          </Button>
        </div>
      )}
    </div>
  );
}
