import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { 
  Users, 
  MessageSquare, 
  Heart, 
  Loader2, 
  Lightbulb, 
  Trophy, 
  Sparkles,
  UserPlus
} from 'lucide-react';
import { useAuth } from '@/hooks/useAuth';
import { useFollows } from '@/hooks/useFollows';
import { communityApi, userPublicApi, API_BASE } from '@/lib/apiServices';
import { formatDistanceToNow } from 'date-fns';

interface FeedPost {
  id: string;
  user_id: string;
  author_name: string;
  content: string;
  category: string;
  likes_count: number;
  replies_count: number;
  created_at: string;
  avatar_url?: string;
  display_name?: string;
}

const categoryIcons: Record<string, React.ElementType> = {
  tip: Lightbulb,
  question: MessageSquare,
  achievement: Trophy,
  discovery: Sparkles,
};

const categoryColors: Record<string, string> = {
  tip: 'bg-green-100 text-green-700 border-green-300 dark:bg-green-950/30 dark:text-green-400',
  question: 'bg-blue-100 text-blue-700 border-blue-300 dark:bg-blue-950/30 dark:text-blue-400',
  achievement: 'bg-amber-100 text-amber-700 border-amber-300 dark:bg-amber-950/30 dark:text-amber-400',
  discovery: 'bg-purple-100 text-purple-700 border-purple-300 dark:bg-purple-950/30 dark:text-purple-400',
};

export const FollowingFeed = () => {
  const { user } = useAuth();
  const { followingList } = useFollows();
  const [posts, setPosts] = useState<FeedPost[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (followingList.length > 0) {
      fetchFollowingPosts();
    } else {
      setPosts([]);
      setLoading(false);
    }
  }, [followingList]);

  const fetchFollowingPosts = async () => {
    setLoading(true);
    try {
      const postsData = await communityApi.getPosts(undefined, 20, 0, undefined, followingList);
      if (postsData?.length) {
        const userIds = [...new Set(postsData.map((p: FeedPost) => p.user_id))];
        const profiles = await Promise.all(
          userIds.map((id: string) => userPublicApi.getProfile(id).catch(() => null))
        );
        const profileMap = new Map(userIds.map((id: string, i: number) => [id, profiles[i]]));
        const enrichedPosts = postsData.map((post: FeedPost) => ({
          ...post,
          display_name: profileMap.get(post.user_id)?.display_name,
          avatar_url: profileMap.get(post.user_id)?.avatar_url,
        }));
        setPosts(enrichedPosts);
      }
    } catch (e) {
      console.error('Error fetching following feed:', e);
    }
    setLoading(false);
  };

  const formatTimeAgo = (dateString: string) => {
    try {
      return formatDistanceToNow(new Date(dateString), { addSuffix: true });
    } catch {
      return 'recently';
    }
  };

  if (!user) {
    return (
      <Card>
        <CardContent className="p-8 text-center">
          <Users className="h-10 w-10 mx-auto text-muted-foreground mb-3" />
          <h3 className="font-semibold mb-2">Sign in to see your feed</h3>
          <p className="text-sm text-muted-foreground mb-4">
            Follow community members to see their posts here
          </p>
          <Link to="/auth">
            <Button>Sign In</Button>
          </Link>
        </CardContent>
      </Card>
    );
  }

  if (loading) {
    return (
      <Card>
        <CardContent className="p-8 flex justify-center">
          <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
        </CardContent>
      </Card>
    );
  }

  if (followingList.length === 0) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg flex items-center gap-2">
            <Users className="h-5 w-5" />
            Following Feed
          </CardTitle>
        </CardHeader>
        <CardContent className="text-center py-6">
          <UserPlus className="h-10 w-10 mx-auto text-muted-foreground mb-3" />
          <h3 className="font-semibold mb-2">Start following people</h3>
          <p className="text-sm text-muted-foreground mb-4">
            Follow community members to see their posts in your personalized feed
          </p>
          <Link to="/community">
            <Button variant="outline">Explore Community</Button>
          </Link>
        </CardContent>
      </Card>
    );
  }

  if (posts.length === 0) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg flex items-center gap-2">
            <Users className="h-5 w-5" />
            Following Feed
          </CardTitle>
        </CardHeader>
        <CardContent className="text-center py-6">
          <MessageSquare className="h-10 w-10 mx-auto text-muted-foreground mb-3" />
          <p className="text-sm text-muted-foreground">
            No recent posts from people you follow
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-lg flex items-center gap-2">
          <Users className="h-5 w-5" />
          Following Feed
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        {posts.map(post => {
          const CategoryIcon = categoryIcons[post.category] || MessageSquare;
          const displayName = post.display_name || post.author_name;

          return (
            <div key={post.id} className="flex gap-3 pb-4 border-b last:border-b-0 last:pb-0">
              <Link to={`/user/${post.user_id}`}>
                <Avatar className="h-10 w-10 cursor-pointer hover:ring-2 hover:ring-primary/50 transition-all">
                  <AvatarImage src={post.avatar_url || undefined} />
                  <AvatarFallback className="bg-primary/10 text-primary text-sm">
                    {displayName.substring(0, 2).toUpperCase()}
                  </AvatarFallback>
                </Avatar>
              </Link>

              <div className="flex-1 min-w-0">
                <div className="flex items-center gap-2 mb-1 flex-wrap">
                  <Link 
                    to={`/user/${post.user_id}`}
                    className="font-semibold text-sm hover:text-primary hover:underline transition-colors"
                  >
                    {displayName}
                  </Link>
                  <Badge variant="outline" className={`${categoryColors[post.category]} text-xs`}>
                    <CategoryIcon className="h-3 w-3 mr-1" />
                    {post.category}
                  </Badge>
                  <span className="text-xs text-muted-foreground">
                    {formatTimeAgo(post.created_at)}
                  </span>
                </div>

                <p className="text-sm mb-2 line-clamp-3">{post.content}</p>

                <div className="flex items-center gap-3 text-xs text-muted-foreground">
                  <span className="flex items-center gap-1">
                    <Heart className="h-3 w-3" />
                    {post.likes_count}
                  </span>
                  <span className="flex items-center gap-1">
                    <MessageSquare className="h-3 w-3" />
                    {post.replies_count}
                  </span>
                </div>
              </div>
            </div>
          );
        })}

        <Link to="/community" className="block">
          <Button variant="outline" className="w-full">
            View All Community Posts
          </Button>
        </Link>
      </CardContent>
    </Card>
  );
};
