import React from "react";
import { FoodItem, getCategoryColor, getIndexRating, getTobinRating } from "@/data/foodDatabase";
import { Card, CardContent, CardHeader } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { AlertTriangle, CheckCircle, Shield, Sparkles, Leaf } from "lucide-react";
import { FavoriteButton } from "./FavoriteButton";
import { ShoppingListButton } from "./ShoppingListButton";
import { PurchasedButton } from "./PurchasedButton";

const getPhytoestrogenBadgeStyle = (potency: string) => {
  switch (potency) {
    case 'high':
      return 'bg-red-500/10 text-red-600 border-red-500/20';
    case 'moderate':
      return 'bg-orange-500/10 text-orange-600 border-orange-500/20';
    case 'low':
      return 'bg-yellow-500/10 text-yellow-600 border-yellow-500/20';
    default:
      return 'bg-muted text-muted-foreground';
  }
};

interface FoodCardProps {
  food: FoodItem;
  onClick?: () => void;
  isSelected?: boolean;
}

const categoryIcons: Record<number, React.ReactNode> = {
  1: <AlertTriangle className="h-4 w-4" />,
  2: <AlertTriangle className="h-4 w-4" />,
  3: <CheckCircle className="h-4 w-4" />,
  4: <Sparkles className="h-4 w-4" />
};

export function FoodCard({ food, onClick, isSelected }: FoodCardProps) {
  const giRating = getIndexRating(food.glycogenicIndex, 'gi');
  const liRating = getIndexRating(food.lipogenicIndex, 'li');
  const categoryColor = getCategoryColor(food.category);
  const tobinRating = food.tobinScore ? getTobinRating(food.tobinScore.total) : null;
  
  return (
    <Card 
      className={`
        cursor-pointer transition-all duration-300 hover:shadow-card hover:-translate-y-1
        ${isSelected ? 'ring-2 ring-primary shadow-glow-primary' : ''}
        ${food.category === 1 ? 'border-category-danger/30' : ''}
        ${food.category === 4 ? 'border-category-therapeutic/30' : ''}
      `}
      onClick={onClick}
    >
      <CardHeader className="pb-2 p-3 sm:p-4 sm:pb-2">
        <div className="flex items-start justify-between gap-2">
          <div className="flex items-center gap-2 sm:gap-3 min-w-0">
            <span className="text-2xl sm:text-3xl flex-shrink-0">{food.emoji}</span>
            <div className="min-w-0">
              <h3 className="font-display font-semibold text-foreground text-sm sm:text-base truncate">{food.name}</h3>
              <div className="flex flex-wrap gap-1 mt-0.5 sm:mt-1">
                <Badge 
                  variant="secondary" 
                  className={`text-${categoryColor} bg-${categoryColor}/10 border-${categoryColor}/20 text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5`}
                >
                  {categoryIcons[food.category]}
                  <span className="ml-0.5 sm:ml-1">{food.categoryName}</span>
                </Badge>
                {tobinRating && (
                  <Badge 
                    variant="outline" 
                    className={`${tobinRating.color} ${tobinRating.bgColor} border-0 text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5`}
                  >
                    TOBIN: {food.tobinScore!.total}/50
                  </Badge>
                )}
                {food.estrogenicProfile?.hasPhytoestrogens && (
                  <Badge 
                    variant="outline" 
                    className={`${getPhytoestrogenBadgeStyle(food.estrogenicProfile.estrogenicPotency)} text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5 gap-0.5`}
                  >
                    <Leaf className="h-3 w-3" />
                    <span className="capitalize">{food.estrogenicProfile.estrogenicPotency}</span>
                  </Badge>
                )}
              </div>
            </div>
          </div>
          <div className="flex items-center gap-0.5 sm:gap-1 flex-shrink-0">
            <PurchasedButton itemId={food.id} itemType="food" size="sm" />
            <ShoppingListButton itemId={food.id} itemType="food" size="sm" />
            <FavoriteButton itemId={food.id} itemType="food" size="sm" />
          </div>
        </div>
      </CardHeader>
      
      <CardContent className="space-y-3 sm:space-y-4 p-3 sm:p-4 pt-0 sm:pt-0">
        {/* Index Meters */}
        <div className="grid grid-cols-2 gap-2 sm:gap-4">
          <div className="space-y-1 sm:space-y-1.5">
            <div className="flex items-center justify-between text-xs sm:text-sm">
              <span className="text-muted-foreground">GI*</span>
              <span className={`font-semibold ${giRating.color}`}>{food.glycogenicIndex.toFixed(2)}</span>
            </div>
            <div className="h-1.5 sm:h-2 bg-muted rounded-full overflow-hidden">
              <div 
                className={`h-full rounded-full transition-all duration-500 ${
                  food.glycogenicIndex < 0.6 ? 'bg-category-safe' : 
                  food.glycogenicIndex < 1.2 ? 'bg-category-warning' : 'bg-category-danger'
                }`}
                style={{ width: `${Math.min(food.glycogenicIndex * 30, 100)}%` }}
              />
            </div>
            <p className={`text-[10px] sm:text-xs ${giRating.color}`}>{giRating.label}</p>
          </div>
          
          <div className="space-y-1 sm:space-y-1.5">
            <div className="flex items-center justify-between text-xs sm:text-sm">
              <span className="text-muted-foreground">LI</span>
              <span className={`font-semibold ${liRating.color}`}>{food.lipogenicIndex.toFixed(2)}</span>
            </div>
            <div className="h-1.5 sm:h-2 bg-muted rounded-full overflow-hidden">
              <div 
                className={`h-full rounded-full transition-all duration-500 ${
                  food.lipogenicIndex < 0.5 ? 'bg-category-safe' : 
                  food.lipogenicIndex < 1.0 ? 'bg-category-warning' : 'bg-category-danger'
                }`}
                style={{ width: `${Math.min(food.lipogenicIndex * 40, 100)}%` }}
              />
            </div>
            <p className={`text-[10px] sm:text-xs ${liRating.color}`}>{liRating.label}</p>
          </div>
        </div>

        {/* TOBIN Mini Display */}
        {food.tobinScore && (
          <div className="flex justify-between text-[10px] sm:text-xs border-t pt-1.5 sm:pt-2">
            <span className="text-red-500" title="Toxins">T:{food.tobinScore.T}</span>
            <span className="text-orange-500" title="Oxidative">O:{food.tobinScore.O}</span>
            <span className="text-purple-500" title="Biological">B:{food.tobinScore.B}</span>
            <span className="text-yellow-600" title="Inflammatory">I:{food.tobinScore.I}</span>
            <span className="text-blue-500" title="Nutrient">N:{food.tobinScore.N}</span>
          </div>
        )}
        
        {/* Quick Stats */}
        <div className="flex flex-wrap gap-1 sm:gap-2 text-[10px] sm:text-xs">
          <span className="px-1.5 sm:px-2 py-0.5 sm:py-1 bg-secondary rounded-full">
            Carbs: {food.netCarbs}g
          </span>
          <span className="px-1.5 sm:px-2 py-0.5 sm:py-1 bg-secondary rounded-full">
            Fiber: {food.fiber}g
          </span>
          <span className="px-1.5 sm:px-2 py-0.5 sm:py-1 bg-secondary rounded-full">
            Protein: {food.protein}g
          </span>
        </div>
      </CardContent>
    </Card>
  );
}
