import { FoodItem, getCategoryColor, getIndexRating, getHalalStatusColor, getKosherStatusColor, getOrganicStatusColor } from "@/data/foodDatabase";
import { getEstrogenicPotencyColor } from "@/data/chemicalDatabase";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { TobinScoreDisplay } from "@/components/TobinScoreDisplay";
import { AlertTriangle, CheckCircle, Sparkles, Lightbulb, ArrowRight, Zap, Activity, Flame, FlaskConical, CircleAlert, Leaf, Star } from "lucide-react";

interface FoodDetailModalProps {
  food: FoodItem | null;
  open: boolean;
  onClose: () => void;
}

export function FoodDetailModal({ food, open, onClose }: FoodDetailModalProps) {
  if (!food) return null;
  
  const giRating = getIndexRating(food.glycogenicIndex, 'gi');
  const liRating = getIndexRating(food.lipogenicIndex, 'li');
  const categoryColor = getCategoryColor(food.category);
  
  return (
    <Dialog open={open} onOpenChange={onClose}>
      <DialogContent className="flex max-h-[90dvh] w-[calc(100vw-2rem)] max-w-2xl flex-col overflow-hidden p-4 sm:w-full sm:p-6">
        <DialogHeader className="shrink-0">
          <div className="flex items-center gap-3 sm:gap-4">
            <span className="text-4xl sm:text-5xl">{food.emoji}</span>
            <div className="min-w-0 flex-1">
              <DialogTitle className="text-xl font-display break-words sm:text-2xl">{food.name}</DialogTitle>
              <Badge 
                variant="secondary" 
                className={`mt-2 text-${categoryColor} bg-${categoryColor}/10`}
              >
                Category {food.category}: {food.categoryName}
              </Badge>
            </div>
          </div>
        </DialogHeader>
        
        <div className="min-h-0 flex-1 overflow-y-auto overscroll-contain space-y-6 mt-4">
          {/* Index Display */}
          <div className="grid grid-cols-2 gap-4 sm:gap-6">
            <div className="p-4 rounded-lg bg-secondary/50 space-y-3">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <Zap className="h-5 w-5 text-primary" />
                  <span className="font-semibold">Glycogenic Index (GI*)</span>
                </div>
                <span className={`text-2xl font-display font-bold ${giRating.color}`}>
                  {food.glycogenicIndex.toFixed(2)}
                </span>
              </div>
              <div className="h-3 bg-muted rounded-full overflow-hidden">
                <div 
                  className={`h-full rounded-full transition-all duration-500 ${
                    food.glycogenicIndex < 0.6 ? 'bg-category-safe' : 
                    food.glycogenicIndex < 1.2 ? 'bg-category-warning' : 'bg-category-danger'
                  }`}
                  style={{ width: `${Math.min(food.glycogenicIndex * 30, 100)}%` }}
                />
              </div>
              <p className="text-sm text-muted-foreground">
                Measures glycogen filling & insulin release
              </p>
            </div>
            
            <div className="p-4 rounded-lg bg-secondary/50 space-y-3">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <Flame className="h-5 w-5 text-destructive" />
                  <span className="font-semibold">Lipogenic Index (LI)</span>
                </div>
                <span className={`text-2xl font-display font-bold ${liRating.color}`}>
                  {food.lipogenicIndex.toFixed(2)}
                </span>
              </div>
              <div className="h-3 bg-muted rounded-full overflow-hidden">
                <div 
                  className={`h-full rounded-full transition-all duration-500 ${
                    food.lipogenicIndex < 0.5 ? 'bg-category-safe' : 
                    food.lipogenicIndex < 1.0 ? 'bg-category-warning' : 'bg-category-danger'
                  }`}
                  style={{ width: `${Math.min(food.lipogenicIndex * 40, 100)}%` }}
                />
              </div>
              <p className="text-sm text-muted-foreground">
                Predicts fat creation & storage potential
              </p>
            </div>
          </div>
          
          {/* TOBIN Score Section */}
          {food.tobinScore && (
            <>
              <Separator />
              <TobinScoreDisplay tobinScore={food.tobinScore} />
            </>
          )}
          
          {/* Mechanism */}
          <div className="p-4 rounded-lg border bg-card">
            <div className="flex items-start gap-3">
              <Activity className="h-5 w-5 text-primary mt-0.5" />
              <div>
                <h4 className="font-semibold mb-1">Metabolic Mechanism</h4>
                <p className="text-muted-foreground">{food.mechanism}</p>
              </div>
            </div>
          </div>
          
          {/* Synergy Warning */}
          {food.synergyWarning && (
            <div className={`p-4 rounded-lg border ${
              food.category === 1 ? 'bg-destructive/5 border-destructive/20' : 
              food.category === 2 ? 'bg-warning/5 border-warning/20' : 'bg-muted'
            }`}>
              <div className="flex items-start gap-3">
                <AlertTriangle className={`h-5 w-5 mt-0.5 ${
                  food.category === 1 ? 'text-destructive' : 'text-warning'
                }`} />
                <div>
                  <h4 className="font-semibold mb-1">Synergistic Effects Warning</h4>
                  <p className="text-muted-foreground">{food.synergyWarning}</p>
                </div>
              </div>
            </div>
          )}
          
          {/* Estrogenic Profile */}
          {food.estrogenicProfile && (food.estrogenicProfile.hasPhytoestrogens || food.estrogenicProfile.hasXenoestrogens || food.estrogenicProfile.estrogenicPotency !== "none") && (
            <div className={`p-4 rounded-lg border ${
              food.estrogenicProfile.estrogenicPotency === "high" ? "bg-pink-50 border-pink-200 dark:bg-pink-950/20 dark:border-pink-800" :
              food.estrogenicProfile.estrogenicPotency === "moderate" ? "bg-orange-50 border-orange-200 dark:bg-orange-950/20 dark:border-orange-800" :
              "bg-purple-50 border-purple-200 dark:bg-purple-950/20 dark:border-purple-800"
            }`}>
              <div className="flex items-start gap-3">
                <FlaskConical className="h-5 w-5 mt-0.5 text-pink-600" />
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-2">
                    <h4 className="font-semibold">Estrogenic Profile</h4>
                    <Badge className={getEstrogenicPotencyColor(food.estrogenicProfile.estrogenicPotency)}>
                      {food.estrogenicProfile.estrogenicPotency} potency
                    </Badge>
                  </div>
                  <div className="space-y-2 text-sm">
                    {food.estrogenicProfile.phytoestrogens && food.estrogenicProfile.phytoestrogens.length > 0 && (
                      <div>
                        <span className="text-muted-foreground">Phytoestrogens: </span>
                        <span>{food.estrogenicProfile.phytoestrogens.join(", ")}</span>
                      </div>
                    )}
                    {food.estrogenicProfile.xenoestrogens && food.estrogenicProfile.xenoestrogens.length > 0 && (
                      <div>
                        <span className="text-muted-foreground">Xenoestrogens: </span>
                        <span className="text-destructive">{food.estrogenicProfile.xenoestrogens.join(", ")}</span>
                      </div>
                    )}
                    {food.estrogenicProfile.healthConcerns && food.estrogenicProfile.healthConcerns.length > 0 && (
                      <div className="mt-2 text-muted-foreground italic">
                        {food.estrogenicProfile.healthConcerns.join("; ")}
                      </div>
                    )}
                  </div>
                </div>
              </div>
            </div>
          )}
          
          {/* Chemical Contaminants */}
          {food.chemicalContaminants && food.chemicalContaminants.length > 0 && (
            <div className="p-4 rounded-lg border bg-amber-50 border-amber-200 dark:bg-amber-950/20 dark:border-amber-800">
              <div className="flex items-start gap-3">
                <CircleAlert className="h-5 w-5 mt-0.5 text-amber-600" />
                <div>
                  <h4 className="font-semibold mb-2">CDC Chemical Contaminants</h4>
                  <div className="flex flex-wrap gap-2 mb-2">
                    {food.chemicalContaminants.map((chem, i) => (
                      <Badge key={i} variant="outline" className="bg-amber-100 border-amber-300 text-amber-800 dark:bg-amber-900/50 dark:text-amber-200">
                        {chem}
                      </Badge>
                    ))}
                  </div>
                  {food.contaminantWarning && (
                    <p className="text-sm text-muted-foreground">{food.contaminantWarning}</p>
                  )}
                </div>
              </div>
            </div>
          )}
          
          {/* Dietary Classifications */}
          {food.dietaryClassification && (
            <div className="p-4 rounded-lg border bg-slate-50 border-slate-200 dark:bg-slate-950/20 dark:border-slate-800">
              <div className="flex items-start gap-3">
                <Star className="h-5 w-5 mt-0.5 text-slate-600" />
                <div className="flex-1">
                  <h4 className="font-semibold mb-3">Dietary Classifications</h4>
                  <div className="grid grid-cols-3 gap-3">
                    {/* Halal Status */}
                    <div className="text-center p-2 rounded-lg bg-background border">
                      <p className="text-xs text-muted-foreground mb-1">Halal</p>
                      <Badge className={getHalalStatusColor(food.dietaryClassification.halal)}>
                        {food.dietaryClassification.halal === "mashbooh" ? "Doubtful" : 
                         food.dietaryClassification.halal.charAt(0).toUpperCase() + food.dietaryClassification.halal.slice(1).replace("_", " ")}
                      </Badge>
                    </div>
                    
                    {/* Kosher Status */}
                    <div className="text-center p-2 rounded-lg bg-background border">
                      <p className="text-xs text-muted-foreground mb-1">Kosher</p>
                      <Badge className={getKosherStatusColor(food.dietaryClassification.kosher)}>
                        {food.dietaryClassification.kosher.charAt(0).toUpperCase() + food.dietaryClassification.kosher.slice(1).replace("_", " ")}
                      </Badge>
                    </div>
                    
                    {/* Organic Status */}
                    <div className="text-center p-2 rounded-lg bg-background border">
                      <p className="text-xs text-muted-foreground mb-1">Organic</p>
                      <Badge className={getOrganicStatusColor(food.dietaryClassification.organic)}>
                        <Leaf className="h-3 w-3 mr-1" />
                        {food.dietaryClassification.organic.charAt(0).toUpperCase() + food.dietaryClassification.organic.slice(1).replace("_", " ")}
                      </Badge>
                    </div>
                  </div>
                  {food.dietaryClassification.certificationNotes && (
                    <p className="text-sm text-muted-foreground mt-2 italic">
                      {food.dietaryClassification.certificationNotes}
                    </p>
                  )}
                </div>
              </div>
            </div>
          )}
          
          {/* Benefits */}
          {food.benefits && food.benefits.length > 0 && (
            <div className="p-4 rounded-lg border bg-category-safe/5 border-category-safe/20">
              <div className="flex items-start gap-3">
                <Sparkles className="h-5 w-5 text-category-safe mt-0.5" />
                <div>
                  <h4 className="font-semibold mb-2">Health Benefits</h4>
                  <div className="flex flex-wrap gap-2">
                    {food.benefits.map((benefit, i) => (
                      <Badge key={i} variant="secondary" className="bg-category-safe/10 text-category-safe">
                        <CheckCircle className="h-3 w-3 mr-1" />
                        {benefit}
                      </Badge>
                    ))}
                  </div>
                </div>
              </div>
            </div>
          )}
          
          <Separator />
          
          {/* Nutritional Breakdown */}
          <div>
            <h4 className="font-semibold mb-3">Nutritional Profile</h4>
            <div className="grid grid-cols-4 gap-4">
              <div className="text-center p-3 rounded-lg bg-secondary">
                <p className="text-2xl font-display font-bold text-foreground">{food.netCarbs}g</p>
                <p className="text-xs text-muted-foreground">Net Carbs</p>
              </div>
              <div className="text-center p-3 rounded-lg bg-secondary">
                <p className="text-2xl font-display font-bold text-foreground">{food.fiber}g</p>
                <p className="text-xs text-muted-foreground">Fiber</p>
              </div>
              <div className="text-center p-3 rounded-lg bg-secondary">
                <p className="text-2xl font-display font-bold text-foreground">{food.protein}g</p>
                <p className="text-xs text-muted-foreground">Protein</p>
              </div>
              <div className="text-center p-3 rounded-lg bg-secondary">
                <p className="text-2xl font-display font-bold text-foreground">{food.fat}g</p>
                <p className="text-xs text-muted-foreground">Fat</p>
              </div>
            </div>
          </div>
          
          {/* Alternatives */}
          {food.alternatives && food.alternatives.length > 0 && (
            <div className="p-4 rounded-lg bg-primary/5 border border-primary/20">
              <div className="flex items-start gap-3">
                <Lightbulb className="h-5 w-5 text-primary mt-0.5" />
                <div>
                  <h4 className="font-semibold mb-2">Healthier Alternatives</h4>
                  <div className="space-y-2">
                    {food.alternatives.map((alt, i) => (
                      <div key={i} className="flex items-center gap-2 text-sm">
                        <ArrowRight className="h-4 w-4 text-primary" />
                        <span>{alt}</span>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            </div>
          )}
          
          {/* Equation Display */}
          <div className="p-4 rounded-lg bg-muted/50 border">
            <h4 className="font-semibold mb-3 text-sm">Calculation Factors</h4>
            <div className="grid grid-cols-2 gap-x-6 gap-y-2 text-sm">
              <div className="flex justify-between">
                <span className="text-muted-foreground">Absorption Coefficient:</span>
                <span className="font-mono">{food.absorptionCoefficient}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">Insulin Secretion:</span>
                <span className="font-mono">{food.insulinSecretionFactor}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">AMPK Score:</span>
                <span className="font-mono">{food.ampkScore}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">Mito Efficiency:</span>
                <span className="font-mono">{food.mitoEfficiency}</span>
              </div>
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}
