import { useState, useMemo } from "react";
import { ShoppingCart, Loader2, Check, Copy, ChevronDown, ChevronUp, Users, Minus, Plus, CalendarDays } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Checkbox } from "@/components/ui/checkbox";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Label } from "@/components/ui/label";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { aiApi } from "@/lib/apiServices";
import { toast } from "sonner";
import { useMealPlanner } from "@/hooks/useMealPlanner";
import { format } from "date-fns";

interface GroceryItem {
  name: string;
  quantity: string;
  unit: string;
  originalQuantity?: number;
  checked?: boolean;
}

interface GroceryCategory {
  name: string;
  items: GroceryItem[];
}

interface GroceryList {
  categories: GroceryCategory[];
  summary: {
    totalItems: number;
    recipesProcessed: number;
  };
}

interface Recipe {
  id: string;
  title: string;
  content: string;
  servings: number;
}

interface GroceryListGeneratorProps {
  recipes: Recipe[];
  selectedRecipeIds: string[];
  onSelectionChange: (ids: string[]) => void;
}

// Parse quantity string to number for scaling
const parseQuantity = (qty: string): { value: number; isNumeric: boolean } => {
  // Handle fractions like "1/2", "1 1/2"
  const fractionMatch = qty.match(/^(\d+)?\s*(\d+)\/(\d+)$/);
  if (fractionMatch) {
    const whole = fractionMatch[1] ? parseInt(fractionMatch[1]) : 0;
    const num = parseInt(fractionMatch[2]);
    const denom = parseInt(fractionMatch[3]);
    return { value: whole + num / denom, isNumeric: true };
  }
  
  // Handle ranges like "2-3"
  const rangeMatch = qty.match(/^(\d+(?:\.\d+)?)\s*-\s*(\d+(?:\.\d+)?)$/);
  if (rangeMatch) {
    const avg = (parseFloat(rangeMatch[1]) + parseFloat(rangeMatch[2])) / 2;
    return { value: avg, isNumeric: true };
  }
  
  const num = parseFloat(qty);
  return { value: isNaN(num) ? 1 : num, isNumeric: !isNaN(num) };
};

// Format scaled quantity nicely
const formatQuantity = (value: number, originalQty: string): string => {
  // If original wasn't numeric, return as-is
  if (parseQuantity(originalQty).isNumeric === false && isNaN(parseFloat(originalQty))) {
    return originalQty;
  }
  
  // Round to reasonable precision
  if (value === Math.floor(value)) {
    return value.toString();
  }
  
  // Convert to nice fractions for common values
  const fractions: Record<number, string> = {
    0.25: "1/4", 0.33: "1/3", 0.5: "1/2", 0.67: "2/3", 0.75: "3/4"
  };
  
  const whole = Math.floor(value);
  const decimal = value - whole;
  const roundedDecimal = Math.round(decimal * 100) / 100;
  
  for (const [dec, frac] of Object.entries(fractions)) {
    if (Math.abs(roundedDecimal - parseFloat(dec)) < 0.05) {
      return whole > 0 ? `${whole} ${frac}` : frac;
    }
  }
  
  return value.toFixed(1).replace(/\.0$/, '');
};

export function GroceryListGenerator({ 
  recipes, 
  selectedRecipeIds, 
  onSelectionChange 
}: GroceryListGeneratorProps) {
  const [groceryList, setGroceryList] = useState<GroceryList | null>(null);
  const [isGenerating, setIsGenerating] = useState(false);
  const [checkedItems, setCheckedItems] = useState<Set<string>>(new Set());
  const [copied, setCopied] = useState(false);
  const [expandedCategories, setExpandedCategories] = useState<Set<string>>(new Set());
  const [servingsMultiplier, setServingsMultiplier] = useState(1);
  const [baseServings, setBaseServings] = useState(0);
  const [sourceTab, setSourceTab] = useState<"recipes" | "mealplan">("recipes");

  const { mealPlans, currentWeekStart, getWeekDays } = useMealPlanner();

  const toggleRecipeSelection = (id: string) => {
    if (selectedRecipeIds.includes(id)) {
      onSelectionChange(selectedRecipeIds.filter(r => r !== id));
    } else {
      onSelectionChange([...selectedRecipeIds, id]);
    }
  };

  const selectAll = () => {
    onSelectionChange(recipes.map(r => r.id));
  };

  const deselectAll = () => {
    onSelectionChange([]);
  };

  // Get unique recipes from meal plans for the current week
  const weeklyMealRecipes = useMemo(() => {
    const recipeMap = new Map<string, { recipe: Recipe; count: number }>();
    
    mealPlans.forEach(plan => {
      if (plan.recipe) {
        const existingRecipe = recipes.find(r => r.id === plan.recipe_id);
        if (existingRecipe) {
          const existing = recipeMap.get(plan.recipe_id);
          if (existing) {
            recipeMap.set(plan.recipe_id, { ...existing, count: existing.count + 1 });
          } else {
            recipeMap.set(plan.recipe_id, { recipe: existingRecipe, count: 1 });
          }
        }
      }
    });
    
    return Array.from(recipeMap.values());
  }, [mealPlans, recipes]);

  // Calculate total servings from weekly meal plan
  const totalWeeklyServings = useMemo(() => {
    return weeklyMealRecipes.reduce((sum, { recipe, count }) => 
      sum + (recipe.servings || 2) * count, 0
    );
  }, [weeklyMealRecipes]);

  // Calculate total servings from selected recipes
  const totalSelectedServings = useMemo(() => {
    return recipes
      .filter(r => selectedRecipeIds.includes(r.id))
      .reduce((sum, r) => sum + (r.servings || 2), 0);
  }, [recipes, selectedRecipeIds]);

  const generateGroceryList = async () => {
    if (selectedRecipeIds.length === 0) {
      toast.error("Please select at least one recipe");
      return;
    }

    setIsGenerating(true);
    setGroceryList(null);
    setCheckedItems(new Set());
    setServingsMultiplier(1);
    setBaseServings(totalSelectedServings);

    try {
      const selectedRecipes = recipes.filter(r => selectedRecipeIds.includes(r.id));
      
      const data = await aiApi.extractIngredients(
        selectedRecipes.map((r) => ({ title: r.title, content: r.content }))
      );

      // Store original quantities for scaling
      const enhancedData = {
        ...data,
        categories: data.categories.map((cat: GroceryCategory) => ({
          ...cat,
          items: cat.items.map((item: GroceryItem) => ({
            ...item,
            originalQuantity: parseQuantity(item.quantity).value
          }))
        }))
      };
      
      setGroceryList(enhancedData);
      setExpandedCategories(new Set(data.categories.map((c: GroceryCategory) => c.name)));
      toast.success("Grocery list generated!");
    } catch (error) {
      console.error('Error generating grocery list:', error);
      toast.error("Failed to generate grocery list");
    } finally {
      setIsGenerating(false);
    }
  };

  const generateFromMealPlan = async () => {
    if (weeklyMealRecipes.length === 0) {
      toast.error("No meals planned for this week");
      return;
    }

    setIsGenerating(true);
    setGroceryList(null);
    setCheckedItems(new Set());
    setServingsMultiplier(1);
    setBaseServings(totalWeeklyServings);

    try {
      // Build recipes array with adjusted quantities for multiple occurrences
      const recipesForExtraction = weeklyMealRecipes.map(({ recipe, count }) => ({
        title: `${recipe.title}${count > 1 ? ` (×${count})` : ''}`,
        content: recipe.content,
        multiplier: count
      }));
      
      const data = await aiApi.extractIngredients(recipesForExtraction);

      // Store original quantities for scaling
      const enhancedData = {
        ...data,
        categories: data.categories.map((cat: GroceryCategory) => ({
          ...cat,
          items: cat.items.map((item: GroceryItem) => ({
            ...item,
            originalQuantity: parseQuantity(item.quantity).value
          }))
        }))
      };
      
      setGroceryList(enhancedData);
      setExpandedCategories(new Set(data.categories.map((c: GroceryCategory) => c.name)));
      toast.success("Weekly grocery list generated!");
    } catch (error) {
      console.error('Error generating grocery list:', error);
      toast.error("Failed to generate grocery list");
    } finally {
      setIsGenerating(false);
    }
  };

  // Scale quantities based on multiplier
  const scaledGroceryList = useMemo(() => {
    if (!groceryList || servingsMultiplier === 1) return groceryList;
    
    return {
      ...groceryList,
      categories: groceryList.categories.map(cat => ({
        ...cat,
        items: cat.items.map(item => {
          const originalValue = item.originalQuantity || parseQuantity(item.quantity).value;
          const scaledValue = originalValue * servingsMultiplier;
          return {
            ...item,
            quantity: formatQuantity(scaledValue, item.quantity)
          };
        })
      }))
    };
  }, [groceryList, servingsMultiplier]);

  const adjustServings = (delta: number) => {
    const newMultiplier = Math.max(0.5, Math.min(4, servingsMultiplier + delta));
    setServingsMultiplier(newMultiplier);
  };

  const currentServings = Math.round(baseServings * servingsMultiplier);

  const toggleItemCheck = (categoryName: string, itemName: string) => {
    const key = `${categoryName}-${itemName}`;
    const newChecked = new Set(checkedItems);
    if (newChecked.has(key)) {
      newChecked.delete(key);
    } else {
      newChecked.add(key);
    }
    setCheckedItems(newChecked);
  };

  const toggleCategory = (categoryName: string) => {
    const newExpanded = new Set(expandedCategories);
    if (newExpanded.has(categoryName)) {
      newExpanded.delete(categoryName);
    } else {
      newExpanded.add(categoryName);
    }
    setExpandedCategories(newExpanded);
  };

  const copyToClipboard = async () => {
    if (!scaledGroceryList) return;

    const text = scaledGroceryList.categories
      .map(cat => {
        const items = cat.items
          .map(item => `  □ ${item.name}: ${item.quantity} ${item.unit}`)
          .join('\n');
        return `${cat.name}\n${items}`;
      })
      .join('\n\n');

    await navigator.clipboard.writeText(text);
    setCopied(true);
    toast.success("Grocery list copied!");
    setTimeout(() => setCopied(false), 2000);
  };

  const checkedCount = checkedItems.size;
  const totalItems = scaledGroceryList?.summary.totalItems || 0;

  const weekDays = getWeekDays();
  const weekLabel = `${format(weekDays[0], 'MMM d')} - ${format(weekDays[6], 'MMM d, yyyy')}`;

  return (
    <div className="space-y-4">
      {/* Source Selection Tabs */}
      <Card>
        <CardHeader className="p-4 pb-2">
          <CardTitle className="text-sm sm:text-base flex items-center gap-2">
            <ShoppingCart className="h-4 w-4 text-primary" />
            Generate Grocery List
          </CardTitle>
          <CardDescription className="text-xs">
            Choose recipes or use your weekly meal plan
          </CardDescription>
        </CardHeader>
        <CardContent className="p-4 pt-2">
          <Tabs value={sourceTab} onValueChange={(v) => setSourceTab(v as "recipes" | "mealplan")}>
            <TabsList className="grid w-full grid-cols-2 mb-4">
              <TabsTrigger value="recipes" className="gap-1.5 text-xs sm:text-sm">
                <ShoppingCart className="h-3.5 w-3.5" />
                Select Recipes
              </TabsTrigger>
              <TabsTrigger value="mealplan" className="gap-1.5 text-xs sm:text-sm">
                <CalendarDays className="h-3.5 w-3.5" />
                Weekly Plan
              </TabsTrigger>
            </TabsList>

            <TabsContent value="recipes" className="mt-0 space-y-4">
              <div className="flex justify-end gap-2">
                <Button variant="ghost" size="sm" onClick={selectAll} className="text-xs h-7">
                  Select All
                </Button>
                <Button variant="ghost" size="sm" onClick={deselectAll} className="text-xs h-7">
                  Clear
                </Button>
              </div>
              <div className="space-y-2 max-h-[200px] overflow-y-auto">
                {recipes.map((recipe) => (
                  <label
                    key={recipe.id}
                    className="flex items-center gap-3 p-2 rounded-lg border cursor-pointer hover:bg-muted/50 transition-colors"
                  >
                    <Checkbox
                      checked={selectedRecipeIds.includes(recipe.id)}
                      onCheckedChange={() => toggleRecipeSelection(recipe.id)}
                    />
                    <span className="text-sm flex-1 truncate">{recipe.title}</span>
                    <Badge variant="outline" className="text-xs shrink-0">
                      {recipe.servings || 2} srv
                    </Badge>
                  </label>
                ))}
                {recipes.length === 0 && (
                  <p className="text-sm text-muted-foreground text-center py-4">
                    No recipes in your cookbook yet
                  </p>
                )}
              </div>
              <Button 
                onClick={generateGroceryList} 
                disabled={isGenerating || selectedRecipeIds.length === 0}
                className="w-full gap-2"
              >
                {isGenerating ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin" />
                    Extracting Ingredients...
                  </>
                ) : (
                  <>
                    <ShoppingCart className="h-4 w-4" />
                    Generate List ({selectedRecipeIds.length} {selectedRecipeIds.length === 1 ? 'recipe' : 'recipes'})
                  </>
                )}
              </Button>
            </TabsContent>

            <TabsContent value="mealplan" className="mt-0 space-y-4">
              <div className="p-3 bg-muted/30 rounded-lg">
                <div className="flex items-center gap-2 mb-2">
                  <CalendarDays className="h-4 w-4 text-primary" />
                  <span className="text-sm font-medium">Week of {weekLabel}</span>
                </div>
                {weeklyMealRecipes.length > 0 ? (
                  <div className="space-y-1.5">
                    {weeklyMealRecipes.map(({ recipe, count }) => (
                      <div key={recipe.id} className="flex items-center justify-between text-sm">
                        <span className="truncate flex-1">{recipe.title}</span>
                        <div className="flex items-center gap-2 shrink-0">
                          {count > 1 && (
                            <Badge variant="secondary" className="text-xs">
                              ×{count}
                            </Badge>
                          )}
                          <Badge variant="outline" className="text-xs">
                            {(recipe.servings || 2) * count} srv
                          </Badge>
                        </div>
                      </div>
                    ))}
                    <div className="pt-2 mt-2 border-t text-xs text-muted-foreground">
                      {mealPlans.length} meals planned • {totalWeeklyServings} total servings
                    </div>
                  </div>
                ) : (
                  <p className="text-sm text-muted-foreground">
                    No meals planned for this week. Go to the Planner tab to schedule meals.
                  </p>
                )}
              </div>
              <Button 
                onClick={generateFromMealPlan} 
                disabled={isGenerating || weeklyMealRecipes.length === 0}
                className="w-full gap-2"
              >
                {isGenerating ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin" />
                    Extracting Ingredients...
                  </>
                ) : (
                  <>
                    <CalendarDays className="h-4 w-4" />
                    Generate Weekly Grocery List
                  </>
                )}
              </Button>
            </TabsContent>
          </Tabs>
        </CardContent>
      </Card>

      {/* Generated Grocery List */}
      {scaledGroceryList && (
        <Card>
          <CardHeader className="p-4 pb-2">
            <div className="flex items-center justify-between flex-wrap gap-2">
              <div>
                <CardTitle className="text-sm sm:text-base">Your Grocery List</CardTitle>
                <CardDescription className="text-xs">
                  {checkedCount} of {totalItems} items checked
                </CardDescription>
              </div>
              <Button 
                variant="outline" 
                size="sm" 
                onClick={copyToClipboard}
                className="gap-1.5"
              >
                {copied ? <Check className="h-3.5 w-3.5" /> : <Copy className="h-3.5 w-3.5" />}
                <span className="hidden sm:inline">{copied ? "Copied" : "Copy"}</span>
              </Button>
            </div>
            
            {/* Servings Scaler */}
            <div className="flex items-center justify-between mt-3 p-3 bg-muted/30 rounded-lg">
              <div className="flex items-center gap-2">
                <Users className="h-4 w-4 text-muted-foreground" />
                <Label className="text-xs sm:text-sm">Scale for servings:</Label>
              </div>
              <div className="flex items-center gap-2">
                <Button
                  variant="outline"
                  size="icon"
                  className="h-7 w-7"
                  onClick={() => adjustServings(-0.5)}
                  disabled={servingsMultiplier <= 0.5}
                >
                  <Minus className="h-3 w-3" />
                </Button>
                <span className="w-16 text-center text-sm font-medium">
                  {currentServings} srv
                </span>
                <Button
                  variant="outline"
                  size="icon"
                  className="h-7 w-7"
                  onClick={() => adjustServings(0.5)}
                  disabled={servingsMultiplier >= 4}
                >
                  <Plus className="h-3 w-3" />
                </Button>
              </div>
            </div>
            {servingsMultiplier !== 1 && (
              <p className="text-xs text-muted-foreground mt-1">
                Quantities scaled {servingsMultiplier > 1 ? 'up' : 'down'} by {servingsMultiplier}x
              </p>
            )}
          </CardHeader>
          <CardContent className="p-4 pt-2">
            <ScrollArea className="h-[400px] sm:h-[500px]">
              <div className="space-y-3 pr-4">
                {scaledGroceryList.categories.map((category) => {
                  const categoryCheckedCount = category.items.filter(
                    item => checkedItems.has(`${category.name}-${item.name}`)
                  ).length;
                  const isExpanded = expandedCategories.has(category.name);

                  return (
                    <div key={category.name} className="border rounded-lg overflow-hidden">
                      <button
                        onClick={() => toggleCategory(category.name)}
                        className="w-full flex items-center justify-between p-3 bg-muted/30 hover:bg-muted/50 transition-colors"
                      >
                        <div className="flex items-center gap-2">
                          <span className="font-medium text-sm">{category.name}</span>
                          <Badge variant="secondary" className="text-xs">
                            {categoryCheckedCount}/{category.items.length}
                          </Badge>
                        </div>
                        {isExpanded ? (
                          <ChevronUp className="h-4 w-4 text-muted-foreground" />
                        ) : (
                          <ChevronDown className="h-4 w-4 text-muted-foreground" />
                        )}
                      </button>
                      
                      {isExpanded && (
                        <div className="p-2 space-y-1">
                          {category.items.map((item) => {
                            const isChecked = checkedItems.has(`${category.name}-${item.name}`);
                            return (
                              <label
                                key={item.name}
                                className={`flex items-center gap-3 p-2 rounded cursor-pointer hover:bg-muted/30 transition-colors ${
                                  isChecked ? 'opacity-50' : ''
                                }`}
                              >
                                <Checkbox
                                  checked={isChecked}
                                  onCheckedChange={() => toggleItemCheck(category.name, item.name)}
                                />
                                <span className={`text-sm flex-1 ${isChecked ? 'line-through' : ''}`}>
                                  {item.name}
                                </span>
                                <span className="text-xs text-muted-foreground whitespace-nowrap">
                                  {item.quantity} {item.unit}
                                </span>
                              </label>
                            );
                          })}
                        </div>
                      )}
                    </div>
                  );
                })}
              </div>
            </ScrollArea>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
