import { useState, useRef, useEffect, useCallback } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { ScrollArea } from "@/components/ui/scroll-area";
import { MessageCircle, Send, X, Bot, User, Loader2, Minimize2, Maximize2, Mic, MicOff, Volume2, VolumeX, Square } from "lucide-react";
import { toast } from "sonner";
import { useSpeechRecognition } from "@/hooks/useSpeechRecognition";
import { useSpeechSynthesis } from "@/hooks/useSpeechSynthesis";

type Message = {
  role: "user" | "assistant";
  content: string;
};

import { aiApi } from "@/lib/apiServices";

export function HealthChatbot() {
  const [isOpen, setIsOpen] = useState(false);
  const [isExpanded, setIsExpanded] = useState(false);
  const [messages, setMessages] = useState<Message[]>([
    {
      role: "assistant",
      content: "👋 Hello! I'm your metabolic health advisor. Ask me anything about:\n\n• **Lipogenic & Glycogenic Indices** of foods\n• **Weight management** strategies\n• **Insulin biology** and metabolic health\n• **Healthy meal construction**\n• **Food alternatives** for better health\n\nYou can type or tap the 🎤 to speak your question!",
    },
  ]);
  const [input, setInput] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [autoSpeak, setAutoSpeak] = useState(true); // Auto-read responses
  const scrollRef = useRef<HTMLDivElement>(null);
  const inputRef = useRef<HTMLInputElement>(null);

  // Speech recognition hook
  const {
    isListening,
    transcript,
    isSupported: isVoiceSupported,
    toggleListening,
    stopListening,
  } = useSpeechRecognition({
    onResult: (finalTranscript) => {
      setInput((prev) => prev + (prev ? " " : "") + finalTranscript);
    },
    onError: (error) => {
      toast.error(error);
    },
  });

  // Speech synthesis hook for text-to-speech
  const {
    speak,
    stop: stopSpeaking,
    isSpeaking,
    isSupported: isTTSSupported,
  } = useSpeechSynthesis({
    rate: 1.0,
    pitch: 1.0,
    onError: (error) => {
      console.error("TTS error:", error);
    },
  });

  // Update input with live transcript while listening
  useEffect(() => {
    if (isListening && transcript) {
      // Show live transcript in input field
      setInput((prev) => {
        const baseInput = prev.replace(/\s*\[listening\.\.\.\].*$/, "");
        return baseInput + (baseInput ? " " : "") + transcript;
      });
    }
  }, [transcript, isListening]);

  useEffect(() => {
    if (scrollRef.current) {
      scrollRef.current.scrollTop = scrollRef.current.scrollHeight;
    }
  }, [messages]);

  useEffect(() => {
    if (isOpen && inputRef.current && !isListening) {
      inputRef.current.focus();
    }
  }, [isOpen, isListening]);

  const sendMessage = useCallback(async () => {
    const messageText = input.trim();
    if (!messageText || isLoading) return;

    // Stop listening if active
    if (isListening) {
      stopListening();
    }

    const userMsg: Message = { role: "user", content: messageText };
    setMessages((prev) => [...prev, userMsg]);
    setInput("");
    setIsLoading(true);

    try {
      const conversationHistory = messages.slice(1).map((m) => ({ role: m.role, content: m.content }));
      const { response } = await aiApi.healthChat(messageText, conversationHistory);
      setMessages((prev) => [...prev, { role: "assistant", content: response || "" }]);
      if (autoSpeak && isTTSSupported && response) {
        setTimeout(() => speak(response), 100);
      }
    } catch (error) {
      const errMsg = error instanceof Error ? error.message : "Failed to get response";
      toast.error(errMsg);
      setMessages((prev) => [
        ...prev,
        { role: "assistant", content: `Sorry, something went wrong: ${errMsg}` },
      ]);
    } finally {
      setIsLoading(false);
    }
  }, [input, isLoading, isListening, messages, stopListening, autoSpeak, isTTSSupported, speak]);

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  };

  const handleVoiceClick = () => {
    if (isLoading) return;
    toggleListening();
  };

  // Render markdown-like formatting
  const renderContent = (content: string) => {
    return content.split("\n").map((line, i) => {
      // Bold text
      const parts = line.split(/(\*\*[^*]+\*\*)/g);
      const rendered = parts.map((part, j) => {
        if (part.startsWith("**") && part.endsWith("**")) {
          return <strong key={j}>{part.slice(2, -2)}</strong>;
        }
        return part;
      });

      // Bullet points
      if (line.startsWith("• ") || line.startsWith("- ")) {
        return (
          <div key={i} className="flex gap-2 ml-2">
            <span>•</span>
            <span>{rendered.slice(1)}</span>
          </div>
        );
      }

      return (
        <p key={i} className={line === "" ? "h-2" : ""}>
          {rendered}
        </p>
      );
    });
  };

  if (!isOpen) {
    return (
      <Button
        onClick={() => setIsOpen(true)}
        className="fixed bottom-6 right-6 h-14 w-14 rounded-full gradient-hero shadow-lg hover:shadow-xl transition-all duration-300 z-50"
        size="icon"
      >
        <MessageCircle className="h-6 w-6" />
      </Button>
    );
  }

  return (
    <Card
      className={`fixed z-50 shadow-2xl transition-all duration-300 ${
        isExpanded
          ? "inset-4 md:inset-8"
          : "bottom-6 right-6 w-[380px] h-[600px] max-h-[80vh]"
      }`}
    >
      <CardHeader className="pb-3 border-b">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-3">
            <div className="p-2 rounded-lg gradient-hero">
              <Bot className="h-5 w-5 text-primary-foreground" />
            </div>
            <div>
              <CardTitle className="text-lg font-display">Health Advisor</CardTitle>
              <p className="text-xs text-muted-foreground flex items-center gap-1">
                {(isVoiceSupported || isTTSSupported) && <Volume2 className="h-3 w-3" />}
                Metabolic health & nutrition
              </p>
            </div>
          </div>
          <div className="flex items-center gap-1">
            {/* TTS toggle button */}
            {isTTSSupported && (
              <Button
                variant="ghost"
                size="icon"
                onClick={() => {
                  if (isSpeaking) stopSpeaking();
                  setAutoSpeak(!autoSpeak);
                }}
                className={`h-8 w-8 ${autoSpeak ? "text-primary" : "text-muted-foreground"}`}
                title={autoSpeak ? "Auto-speak is ON" : "Auto-speak is OFF"}
              >
                {autoSpeak ? <Volume2 className="h-4 w-4" /> : <VolumeX className="h-4 w-4" />}
              </Button>
            )}
            <Button
              variant="ghost"
              size="icon"
              onClick={() => setIsExpanded(!isExpanded)}
              className="h-8 w-8"
            >
              {isExpanded ? (
                <Minimize2 className="h-4 w-4" />
              ) : (
                <Maximize2 className="h-4 w-4" />
              )}
            </Button>
            <Button
              variant="ghost"
              size="icon"
              onClick={() => {
                if (isListening) stopListening();
                if (isSpeaking) stopSpeaking();
                setIsOpen(false);
              }}
              className="h-8 w-8"
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </CardHeader>

      <CardContent className="p-0 flex flex-col h-[calc(100%-80px)]">
        <ScrollArea className="flex-1 p-4" ref={scrollRef}>
          <div className="space-y-4">
            {messages.map((msg, i) => (
              <div
                key={i}
                className={`flex gap-3 ${msg.role === "user" ? "justify-end" : "justify-start"}`}
              >
                {msg.role === "assistant" && (
                  <div className="flex-shrink-0 w-8 h-8 rounded-full gradient-hero flex items-center justify-center">
                    <Bot className="h-4 w-4 text-primary-foreground" />
                  </div>
                )}
                <div
                  className={`max-w-[85%] rounded-2xl px-4 py-3 text-sm ${
                    msg.role === "user"
                      ? "bg-primary text-primary-foreground rounded-br-md"
                      : "bg-secondary text-secondary-foreground rounded-bl-md"
                  }`}
                >
                  <div className="space-y-1">{renderContent(msg.content)}</div>
                  {/* Speak button for assistant messages */}
                  {msg.role === "assistant" && isTTSSupported && (
                    <button
                      onClick={() => {
                        if (isSpeaking) {
                          stopSpeaking();
                        } else {
                          speak(msg.content);
                        }
                      }}
                      className="mt-2 flex items-center gap-1 text-xs text-muted-foreground hover:text-foreground transition-colors"
                      title={isSpeaking ? "Stop speaking" : "Read aloud"}
                    >
                      {isSpeaking ? (
                        <>
                          <Square className="h-3 w-3" />
                          <span>Stop</span>
                        </>
                      ) : (
                        <>
                          <Volume2 className="h-3 w-3" />
                          <span>Read aloud</span>
                        </>
                      )}
                    </button>
                  )}
                </div>
                {msg.role === "user" && (
                  <div className="flex-shrink-0 w-8 h-8 rounded-full bg-muted flex items-center justify-center">
                    <User className="h-4 w-4 text-muted-foreground" />
                  </div>
                )}
              </div>
            ))}
            {isLoading && messages[messages.length - 1]?.role === "user" && (
              <div className="flex gap-3 justify-start">
                <div className="flex-shrink-0 w-8 h-8 rounded-full gradient-hero flex items-center justify-center">
                  <Bot className="h-4 w-4 text-primary-foreground" />
                </div>
                <div className="bg-secondary rounded-2xl rounded-bl-md px-4 py-3">
                  <Loader2 className="h-5 w-5 animate-spin text-muted-foreground" />
                </div>
              </div>
            )}
          </div>
        </ScrollArea>

        <div className="p-4 border-t">
          {/* Speaking indicator */}
          {isSpeaking && (
            <div className="mb-3 flex items-center justify-center gap-2 text-sm text-primary">
              <div className="flex gap-1">
                <span className="w-1.5 h-3 bg-primary rounded-full animate-pulse" />
                <span className="w-1.5 h-4 bg-primary rounded-full animate-pulse" style={{ animationDelay: "100ms" }} />
                <span className="w-1.5 h-2 bg-primary rounded-full animate-pulse" style={{ animationDelay: "200ms" }} />
                <span className="w-1.5 h-4 bg-primary rounded-full animate-pulse" style={{ animationDelay: "300ms" }} />
                <span className="w-1.5 h-3 bg-primary rounded-full animate-pulse" style={{ animationDelay: "400ms" }} />
              </div>
              <span>Speaking...</span>
              <button
                onClick={stopSpeaking}
                className="ml-2 px-2 py-0.5 text-xs bg-primary/10 rounded hover:bg-primary/20 transition-colors"
              >
                Stop
              </button>
            </div>
          )}
          
          {/* Voice indicator */}
          {isListening && (
            <div className="mb-3 flex items-center justify-center gap-2 text-sm text-primary animate-pulse">
              <div className="flex gap-1">
                <span className="w-1 h-4 bg-primary rounded-full animate-bounce" style={{ animationDelay: "0ms" }} />
                <span className="w-1 h-4 bg-primary rounded-full animate-bounce" style={{ animationDelay: "150ms" }} />
                <span className="w-1 h-4 bg-primary rounded-full animate-bounce" style={{ animationDelay: "300ms" }} />
              </div>
              <span>Listening... Speak your question</span>
            </div>
          )}
          
          <div className="flex gap-2">
            {/* Voice input button */}
            {isVoiceSupported && (
              <Button
                onClick={handleVoiceClick}
                disabled={isLoading}
                variant={isListening ? "default" : "outline"}
                size="icon"
                className={`flex-shrink-0 transition-all duration-300 ${
                  isListening 
                    ? "gradient-hero animate-pulse-ring shadow-glow-primary" 
                    : "hover:border-primary hover:text-primary"
                }`}
                title={isListening ? "Stop listening" : "Start voice input"}
              >
                {isListening ? (
                  <MicOff className="h-4 w-4" />
                ) : (
                  <Mic className="h-4 w-4" />
                )}
              </Button>
            )}
            
            <Input
              ref={inputRef}
              value={input}
              onChange={(e) => setInput(e.target.value)}
              onKeyDown={handleKeyPress}
              placeholder={isListening ? "Listening..." : "Ask about nutrition, health..."}
              disabled={isLoading}
              className={`flex-1 transition-all duration-300 ${isListening ? "border-primary" : ""}`}
            />
            <Button
              onClick={sendMessage}
              disabled={!input.trim() || isLoading}
              size="icon"
              className="gradient-hero"
            >
              <Send className="h-4 w-4" />
            </Button>
          </div>
          <p className="text-xs text-muted-foreground mt-2 text-center">
            For educational purposes only. Consult a healthcare provider for medical advice.
          </p>
        </div>
      </CardContent>
    </Card>
  );
}
