import { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Switch } from '@/components/ui/switch';
import { Slider } from '@/components/ui/slider';
import { Label } from '@/components/ui/label';
import { Button } from '@/components/ui/button';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Separator } from '@/components/ui/separator';
import { Badge } from '@/components/ui/badge';
import { 
  Heart, 
  Shield, 
  Sparkles, 
  Bug, 
  Flame, 
  Leaf, 
  AlertTriangle,
  Bell,
  Activity,
  Brain,
  Pill,
  HeartPulse,
  Eye,
  Baby,
  Zap,
  Wind,
  Droplets,
  Users,
  Check
} from 'lucide-react';
import { useHealthProfile, HealthProfile } from '@/hooks/useHealthProfile';

interface HealthPreset {
  id: string;
  name: string;
  icon: React.ElementType;
  description: string;
  color: string;
  bgColor: string;
  profile: Partial<HealthProfile>;
}

const healthPresets: HealthPreset[] = [
  {
    id: 'pregnancy',
    name: 'Pregnancy Safe',
    icon: Baby,
    description: 'Extra strict on toxins & hormonal disruptors',
    color: 'text-pink-600',
    bgColor: 'bg-pink-100 dark:bg-pink-950/30 border-pink-200 dark:border-pink-800',
    profile: {
      has_hormonal_imbalance: true,
      toxin_threshold: 2,
      oxidative_threshold: 3,
      biological_threshold: 2,
      inflammatory_threshold: 4,
      nutrient_threshold: 3,
      total_threshold: 15,
      show_alerts: true,
    },
  },
  {
    id: 'autoimmune',
    name: 'Autoimmune Protocol',
    icon: Shield,
    description: 'Focus on inflammation & immune triggers',
    color: 'text-purple-600',
    bgColor: 'bg-purple-100 dark:bg-purple-950/30 border-purple-200 dark:border-purple-800',
    profile: {
      has_autoimmune: true,
      has_digestive_issues: true,
      toxin_threshold: 3,
      oxidative_threshold: 3,
      biological_threshold: 2,
      inflammatory_threshold: 2,
      nutrient_threshold: 3,
      total_threshold: 15,
      show_alerts: true,
    },
  },
  {
    id: 'low-inflammation',
    name: 'Low Inflammation',
    icon: Flame,
    description: 'Reduce inflammatory compounds',
    color: 'text-orange-600',
    bgColor: 'bg-orange-100 dark:bg-orange-950/30 border-orange-200 dark:border-orange-800',
    profile: {
      toxin_threshold: 5,
      oxidative_threshold: 4,
      biological_threshold: 5,
      inflammatory_threshold: 2,
      nutrient_threshold: 5,
      total_threshold: 20,
      show_alerts: true,
    },
  },
  {
    id: 'heart-healthy',
    name: 'Heart Healthy',
    icon: HeartPulse,
    description: 'Optimize for cardiovascular health',
    color: 'text-red-600',
    bgColor: 'bg-red-100 dark:bg-red-950/30 border-red-200 dark:border-red-800',
    profile: {
      has_cardiovascular_issues: true,
      toxin_threshold: 4,
      oxidative_threshold: 2,
      biological_threshold: 5,
      inflammatory_threshold: 3,
      nutrient_threshold: 3,
      total_threshold: 18,
      show_alerts: true,
    },
  },
  {
    id: 'gut-friendly',
    name: 'Gut Friendly',
    icon: Activity,
    description: 'Gentle on digestive system',
    color: 'text-green-600',
    bgColor: 'bg-green-100 dark:bg-green-950/30 border-green-200 dark:border-green-800',
    profile: {
      has_digestive_issues: true,
      has_allergies: true,
      toxin_threshold: 4,
      oxidative_threshold: 5,
      biological_threshold: 2,
      inflammatory_threshold: 3,
      nutrient_threshold: 4,
      total_threshold: 18,
      show_alerts: true,
    },
  },
  {
    id: 'sensitive-skin',
    name: 'Sensitive Skin',
    icon: Droplets,
    description: 'Avoid skin irritants & allergens',
    color: 'text-cyan-600',
    bgColor: 'bg-cyan-100 dark:bg-cyan-950/30 border-cyan-200 dark:border-cyan-800',
    profile: {
      has_skin_sensitivity: true,
      has_allergies: true,
      toxin_threshold: 3,
      oxidative_threshold: 4,
      biological_threshold: 3,
      inflammatory_threshold: 3,
      nutrient_threshold: 5,
      total_threshold: 18,
      show_alerts: true,
    },
  },
  {
    id: 'balanced',
    name: 'Balanced',
    icon: Users,
    description: 'Moderate settings for general wellness',
    color: 'text-blue-600',
    bgColor: 'bg-blue-100 dark:bg-blue-950/30 border-blue-200 dark:border-blue-800',
    profile: {
      has_autoimmune: false,
      has_allergies: false,
      has_digestive_issues: false,
      has_hormonal_imbalance: false,
      has_cardiovascular_issues: false,
      has_skin_sensitivity: false,
      toxin_threshold: 5,
      oxidative_threshold: 5,
      biological_threshold: 5,
      inflammatory_threshold: 5,
      nutrient_threshold: 5,
      total_threshold: 25,
      show_alerts: true,
    },
  },
  {
    id: 'strict',
    name: 'Ultra Strict',
    icon: Zap,
    description: 'Maximum sensitivity for all categories',
    color: 'text-amber-600',
    bgColor: 'bg-amber-100 dark:bg-amber-950/30 border-amber-200 dark:border-amber-800',
    profile: {
      toxin_threshold: 2,
      oxidative_threshold: 2,
      biological_threshold: 2,
      inflammatory_threshold: 2,
      nutrient_threshold: 2,
      total_threshold: 10,
      show_alerts: true,
    },
  },
];

const HealthProfileSettings = () => {
  const { profile, loading, updateProfile, getConditionRecommendations } = useHealthProfile();
  const [localProfile, setLocalProfile] = useState<HealthProfile>(profile);
  const [hasChanges, setHasChanges] = useState(false);
  const [selectedPreset, setSelectedPreset] = useState<string | null>(null);

  useEffect(() => {
    setLocalProfile(profile);
  }, [profile]);

  const handleConditionChange = (condition: keyof HealthProfile, value: boolean) => {
    setLocalProfile(prev => ({ ...prev, [condition]: value }));
    setHasChanges(true);
    setSelectedPreset(null);
  };

  const handleThresholdChange = (threshold: keyof HealthProfile, value: number) => {
    setLocalProfile(prev => ({ ...prev, [threshold]: value }));
    setHasChanges(true);
    setSelectedPreset(null);
  };

  const handlePresetSelect = (preset: HealthPreset) => {
    setLocalProfile(prev => ({ ...prev, ...preset.profile }));
    setSelectedPreset(preset.id);
    setHasChanges(true);
  };

  const handleSave = async () => {
    const success = await updateProfile(localProfile);
    if (success) {
      setHasChanges(false);
    }
  };

  const recommendations = getConditionRecommendations();

  const healthConditions = [
    { key: 'has_autoimmune', label: 'Autoimmune Condition', icon: Shield, description: 'Lupus, rheumatoid arthritis, MS, etc.' },
    { key: 'has_allergies', label: 'Allergies', icon: Bug, description: 'Food allergies, environmental sensitivities' },
    { key: 'has_digestive_issues', label: 'Digestive Issues', icon: Activity, description: 'IBS, Crohn\'s, celiac, GERD' },
    { key: 'has_hormonal_imbalance', label: 'Hormonal Imbalance', icon: Brain, description: 'Thyroid issues, PCOS, menopause' },
    { key: 'has_cardiovascular_issues', label: 'Cardiovascular Issues', icon: HeartPulse, description: 'Heart disease, hypertension' },
    { key: 'has_skin_sensitivity', label: 'Skin Sensitivity', icon: Eye, description: 'Eczema, psoriasis, rosacea' },
  ];

  const tobinThresholds = [
    { key: 'toxin_threshold', label: 'Toxin Threshold', icon: Pill, color: 'text-purple-500', description: 'Chemical contaminants, heavy metals' },
    { key: 'oxidative_threshold', label: 'Oxidative Stress Threshold', icon: Sparkles, color: 'text-orange-500', description: 'Free radicals, cellular damage' },
    { key: 'biological_threshold', label: 'Biological Risk Threshold', icon: Bug, color: 'text-red-500', description: 'Pathogens, allergens' },
    { key: 'inflammatory_threshold', label: 'Inflammatory Threshold', icon: Flame, color: 'text-yellow-500', description: 'Pro-inflammatory compounds' },
    { key: 'nutrient_threshold', label: 'Nutrient Deficit Threshold', icon: Leaf, color: 'text-green-500', description: 'Missing essential nutrients' },
  ];

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      {/* Quick Presets */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Zap className="h-5 w-5 text-primary" />
            Quick Presets
          </CardTitle>
          <CardDescription>
            Choose a preset profile to quickly configure your TOBIN thresholds
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
            {healthPresets.map((preset) => {
              const Icon = preset.icon;
              const isSelected = selectedPreset === preset.id;
              
              return (
                <button
                  key={preset.id}
                  onClick={() => handlePresetSelect(preset)}
                  className={`
                    relative p-4 rounded-xl border-2 transition-all duration-200
                    hover:scale-[1.02] hover:shadow-md
                    ${isSelected 
                      ? `${preset.bgColor} border-current ${preset.color} ring-2 ring-offset-2 ring-current` 
                      : 'bg-card border-border hover:border-primary/50'
                    }
                  `}
                >
                  {isSelected && (
                    <div className="absolute top-2 right-2">
                      <Check className="h-4 w-4" />
                    </div>
                  )}
                  <Icon className={`h-6 w-6 mb-2 ${isSelected ? '' : preset.color}`} />
                  <h4 className="font-semibold text-sm mb-1">{preset.name}</h4>
                  <p className="text-xs text-muted-foreground line-clamp-2">{preset.description}</p>
                </button>
              );
            })}
          </div>
          
          {selectedPreset && (
            <div className="mt-4 p-3 rounded-lg bg-muted/50 border">
              <p className="text-sm text-muted-foreground">
                <strong className="text-foreground">
                  {healthPresets.find(p => p.id === selectedPreset)?.name}
                </strong>
                {' '}preset applied. Thresholds and conditions have been updated below. 
                Click <strong>Save Health Profile</strong> to apply changes.
              </p>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Health Conditions */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Heart className="h-5 w-5 text-primary" />
            Health Conditions
          </CardTitle>
          <CardDescription>
            Select any conditions that apply to you for personalized threshold recommendations
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          {healthConditions.map(({ key, label, icon: Icon, description }) => (
            <div key={key} className="flex items-center justify-between p-3 rounded-lg bg-muted/50 hover:bg-muted transition-colors">
              <div className="flex items-center gap-3">
                <Icon className="h-5 w-5 text-muted-foreground" />
                <div>
                  <Label htmlFor={key} className="font-medium cursor-pointer">{label}</Label>
                  <p className="text-sm text-muted-foreground">{description}</p>
                </div>
              </div>
              <Switch
                id={key}
                checked={localProfile[key as keyof HealthProfile] as boolean}
                onCheckedChange={(checked) => handleConditionChange(key as keyof HealthProfile, checked)}
              />
            </div>
          ))}
        </CardContent>
      </Card>

      {/* Personalized Recommendations */}
      {recommendations.length > 0 && (
        <Alert>
          <AlertTriangle className="h-4 w-4" />
          <AlertDescription>
            <p className="font-medium mb-2">Based on your conditions:</p>
            <ul className="list-disc list-inside space-y-1">
              {recommendations.map((rec, index) => (
                <li key={index} className="text-sm">{rec}</li>
              ))}
            </ul>
          </AlertDescription>
        </Alert>
      )}

      {/* TOBIN Category Thresholds */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Shield className="h-5 w-5 text-primary" />
            TOBIN Sensitivity Thresholds
          </CardTitle>
          <CardDescription>
            Set your personal thresholds for each TOBIN category. Lower values = more sensitive alerts.
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          {tobinThresholds.map(({ key, label, icon: Icon, color, description }) => (
            <div key={key} className="space-y-3">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <Icon className={`h-5 w-5 ${color}`} />
                  <div>
                    <Label className="font-medium">{label}</Label>
                    <p className="text-xs text-muted-foreground">{description}</p>
                  </div>
                </div>
                <span className="text-lg font-bold tabular-nums">
                  {localProfile[key as keyof HealthProfile]}/10
                </span>
              </div>
              <Slider
                value={[localProfile[key as keyof HealthProfile] as number]}
                onValueChange={([value]) => handleThresholdChange(key as keyof HealthProfile, value)}
                min={1}
                max={10}
                step={1}
                className="w-full"
              />
              <div className="flex justify-between text-xs text-muted-foreground">
                <span>Very Sensitive</span>
                <span>Less Sensitive</span>
              </div>
            </div>
          ))}

          <Separator className="my-4" />

          {/* Total Threshold */}
          <div className="space-y-3">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Shield className="h-5 w-5 text-primary" />
                <div>
                  <Label className="font-medium">Total TOBIN Threshold</Label>
                  <p className="text-xs text-muted-foreground">Combined score limit (5-50)</p>
                </div>
              </div>
              <span className="text-lg font-bold tabular-nums">
                {localProfile.total_threshold}/50
              </span>
            </div>
            <Slider
              value={[localProfile.total_threshold]}
              onValueChange={([value]) => handleThresholdChange('total_threshold', value)}
              min={5}
              max={50}
              step={1}
              className="w-full"
            />
            <div className="flex justify-between text-xs text-muted-foreground">
              <span>Very Strict</span>
              <span>More Lenient</span>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Alert Settings */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Bell className="h-5 w-5 text-primary" />
            Alert Settings
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex items-center justify-between p-3 rounded-lg bg-muted/50">
            <div className="flex items-center gap-3">
              <AlertTriangle className="h-5 w-5 text-muted-foreground" />
              <div>
                <Label htmlFor="show_alerts" className="font-medium cursor-pointer">Show Health Alerts</Label>
                <p className="text-sm text-muted-foreground">Get alerts when items exceed your thresholds</p>
              </div>
            </div>
            <Switch
              id="show_alerts"
              checked={localProfile.show_alerts}
              onCheckedChange={(checked) => handleConditionChange('show_alerts', checked)}
            />
          </div>
        </CardContent>
      </Card>

      {/* Save Button */}
      {hasChanges && (
        <div className="flex justify-end">
          <Button onClick={handleSave} size="lg">
            Save Health Profile
          </Button>
        </div>
      )}
    </div>
  );
};

export default HealthProfileSettings;
