import { useState, DragEvent } from "react";
import { format, isToday } from "date-fns";
import { ChevronLeft, ChevronRight, Calendar, Plus, X, CalendarDays, Bell, BellOff, GripVertical } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Calendar as CalendarPicker } from "@/components/ui/calendar";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { useMealPlanner, MEAL_SLOTS, MealSlot, REMINDER_OPTIONS, ReminderTime } from "@/hooks/useMealPlanner";
import { useSavedRecipes } from "@/hooks/useSavedRecipes";
import { useAuth } from "@/hooks/useAuth";
import { cn } from "@/lib/utils";

interface DragData {
  mealId: string;
  recipeTitle: string;
}

export function MealPlannerCalendar() {
  const { user } = useAuth();
  const { recipes } = useSavedRecipes();
  const {
    loading,
    currentWeekStart,
    getWeekDays,
    getMealsForDayAndSlot,
    addMealPlan,
    removeMealPlan,
    moveMealPlan,
    setReminder,
    goToNextWeek,
    goToPreviousWeek,
    goToCurrentWeek,
  } = useMealPlanner();

  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [selectedDate, setSelectedDate] = useState<Date | null>(null);
  const [selectedSlot, setSelectedSlot] = useState<MealSlot>("dinner");
  const [selectedRecipeId, setSelectedRecipeId] = useState<string>("");
  const [selectedReminder, setSelectedReminder] = useState<ReminderTime>(null);
  const [draggedMeal, setDraggedMeal] = useState<DragData | null>(null);
  const [dropTarget, setDropTarget] = useState<{ date: string; slot: string } | null>(null);

  const weekDays = getWeekDays();

  const handleAddMeal = async () => {
    if (!selectedDate || !selectedRecipeId) return;
    
    const success = await addMealPlan(
      selectedRecipeId, 
      selectedDate, 
      selectedSlot, 
      undefined,
      selectedReminder
    );
    if (success) {
      setIsAddDialogOpen(false);
      setSelectedRecipeId("");
      setSelectedReminder(null);
    }
  };

  const openAddDialog = (date: Date, slot: MealSlot) => {
    setSelectedDate(date);
    setSelectedSlot(slot);
    setIsAddDialogOpen(true);
  };

  // Drag and drop handlers
  const handleDragStart = (e: DragEvent<HTMLDivElement>, mealId: string, recipeTitle: string) => {
    e.dataTransfer.effectAllowed = 'move';
    e.dataTransfer.setData('text/plain', mealId);
    setDraggedMeal({ mealId, recipeTitle });
  };

  const handleDragEnd = () => {
    setDraggedMeal(null);
    setDropTarget(null);
  };

  const handleDragOver = (e: DragEvent<HTMLDivElement>, date: Date, slot: MealSlot) => {
    e.preventDefault();
    e.dataTransfer.dropEffect = 'move';
    const targetKey = `${format(date, 'yyyy-MM-dd')}-${slot}`;
    if (!dropTarget || dropTarget.date !== format(date, 'yyyy-MM-dd') || dropTarget.slot !== slot) {
      setDropTarget({ date: format(date, 'yyyy-MM-dd'), slot });
    }
  };

  const handleDragLeave = () => {
    setDropTarget(null);
  };

  const handleDrop = async (e: DragEvent<HTMLDivElement>, date: Date, slot: MealSlot) => {
    e.preventDefault();
    const mealId = e.dataTransfer.getData('text/plain');
    if (mealId && draggedMeal) {
      await moveMealPlan(mealId, date, slot);
    }
    setDraggedMeal(null);
    setDropTarget(null);
  };

  const isDropTarget = (date: Date, slot: MealSlot) => {
    return dropTarget?.date === format(date, 'yyyy-MM-dd') && dropTarget?.slot === slot;
  };

  if (!user) {
    return (
      <Card className="bg-muted/30">
        <CardContent className="p-4 sm:p-6 text-center">
          <CalendarDays className="h-10 w-10 mx-auto text-muted-foreground mb-3" />
          <h3 className="font-semibold text-sm sm:text-base mb-2">Meal Planner</h3>
          <p className="text-xs sm:text-sm text-muted-foreground">
            Sign in to plan your weekly meals
          </p>
        </CardContent>
      </Card>
    );
  }

  if (recipes.length === 0) {
    return (
      <Card className="bg-muted/30">
        <CardContent className="p-4 sm:p-6 text-center">
          <CalendarDays className="h-10 w-10 mx-auto text-muted-foreground mb-3" />
          <h3 className="font-semibold text-sm sm:text-base mb-2">No Recipes Yet</h3>
          <p className="text-xs sm:text-sm text-muted-foreground">
            Save some recipes to your cookbook first, then plan your meals
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader className="p-4 sm:p-6 pb-2">
        <div className="flex items-center justify-between">
          <div>
            <CardTitle className="text-base sm:text-lg flex items-center gap-2">
              <CalendarDays className="h-5 w-5 text-primary" />
              Weekly Meal Plan
            </CardTitle>
            <CardDescription className="text-xs sm:text-sm">
              {format(currentWeekStart, "MMM d")} - {format(weekDays[6], "MMM d, yyyy")}
              <span className="ml-2 text-muted-foreground/70">• Drag meals to reschedule</span>
            </CardDescription>
          </div>
          <div className="flex items-center gap-1">
            <Button variant="ghost" size="icon" onClick={goToPreviousWeek} className="h-8 w-8">
              <ChevronLeft className="h-4 w-4" />
            </Button>
            <Button variant="outline" size="sm" onClick={goToCurrentWeek} className="text-xs h-8">
              Today
            </Button>
            <Button variant="ghost" size="icon" onClick={goToNextWeek} className="h-8 w-8">
              <ChevronRight className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </CardHeader>
      <CardContent className="p-4 sm:p-6 pt-2">
        <ScrollArea className="w-full">
          <div className="min-w-[600px]">
            {/* Header - Days of week */}
            <div className="grid grid-cols-8 gap-1 mb-2">
              <div className="p-2 text-xs font-medium text-muted-foreground">Meal</div>
              {weekDays.map((day) => (
                <div
                  key={day.toISOString()}
                  className={cn(
                    "p-2 text-center rounded-lg",
                    isToday(day) && "bg-primary/10"
                  )}
                >
                  <p className="text-xs font-medium text-muted-foreground">
                    {format(day, "EEE")}
                  </p>
                  <p className={cn(
                    "text-sm font-bold",
                    isToday(day) && "text-primary"
                  )}>
                    {format(day, "d")}
                  </p>
                </div>
              ))}
            </div>

            {/* Meal Slots */}
            {MEAL_SLOTS.map((slot) => (
              <div key={slot.value} className="grid grid-cols-8 gap-1 mb-1">
                <div className="p-2 flex items-center gap-1 text-xs font-medium text-muted-foreground">
                  <span>{slot.emoji}</span>
                  <span className="hidden sm:inline">{slot.label}</span>
                </div>
                {weekDays.map((day) => {
                  const meals = getMealsForDayAndSlot(day, slot.value);
                  const isTarget = isDropTarget(day, slot.value);
                  return (
                    <div
                      key={`${day.toISOString()}-${slot.value}`}
                      className={cn(
                        "min-h-[60px] p-1 rounded-lg border border-dashed transition-all",
                        isToday(day) && "bg-primary/5",
                        isTarget 
                          ? "border-primary border-solid bg-primary/10" 
                          : "border-muted-foreground/20 hover:border-primary/50"
                      )}
                      onDragOver={(e) => handleDragOver(e, day, slot.value)}
                      onDragLeave={handleDragLeave}
                      onDrop={(e) => handleDrop(e, day, slot.value)}
                    >
                      {meals.length > 0 ? (
                        <div className="space-y-1">
                          {meals.map((meal) => (
                            <div
                              key={meal.id}
                              draggable
                              onDragStart={(e) => handleDragStart(e, meal.id, meal.recipe?.title || "Recipe")}
                              onDragEnd={handleDragEnd}
                              className={cn(
                                "group relative p-1.5 bg-primary/10 rounded text-xs cursor-grab active:cursor-grabbing",
                                draggedMeal?.mealId === meal.id && "opacity-50"
                              )}
                            >
                              <div className="flex items-start gap-1">
                                <GripVertical className="h-3 w-3 text-muted-foreground/50 shrink-0 mt-0.5" />
                                <p className="font-medium truncate flex-1 text-[10px] sm:text-xs pr-6">
                                  {meal.recipe?.title || "Recipe"}
                                </p>
                              </div>
                              
                              {/* Reminder indicator */}
                              {meal.reminder_time && (
                                <Bell className="absolute bottom-1 left-1 h-2.5 w-2.5 text-primary" />
                              )}
                              
                              {/* Actions */}
                              <div className="absolute top-1 right-1 flex items-center gap-0.5 opacity-0 group-hover:opacity-100 transition-opacity">
                                <DropdownMenu>
                                  <DropdownMenuTrigger asChild>
                                    <button className="p-0.5 hover:bg-muted rounded">
                                      {meal.reminder_time ? (
                                        <Bell className="h-3 w-3 text-primary" />
                                      ) : (
                                        <BellOff className="h-3 w-3 text-muted-foreground" />
                                      )}
                                    </button>
                                  </DropdownMenuTrigger>
                                  <DropdownMenuContent align="end" className="w-48">
                                    {REMINDER_OPTIONS.map((option) => (
                                      <DropdownMenuItem
                                        key={option.value || 'none'}
                                        onClick={() => setReminder(meal.id, option.value)}
                                        className={cn(
                                          meal.reminder_time === option.value && "bg-primary/10"
                                        )}
                                      >
                                        {option.value ? (
                                          <Bell className="h-3.5 w-3.5 mr-2" />
                                        ) : (
                                          <BellOff className="h-3.5 w-3.5 mr-2" />
                                        )}
                                        {option.label}
                                      </DropdownMenuItem>
                                    ))}
                                  </DropdownMenuContent>
                                </DropdownMenu>
                                <button
                                  onClick={() => removeMealPlan(meal.id)}
                                  className="p-0.5 hover:bg-muted rounded"
                                >
                                  <X className="h-3 w-3 text-muted-foreground hover:text-destructive" />
                                </button>
                              </div>
                            </div>
                          ))}
                        </div>
                      ) : (
                        <button
                          onClick={() => openAddDialog(day, slot.value)}
                          className="w-full h-full flex items-center justify-center opacity-0 hover:opacity-100 transition-opacity"
                        >
                          <Plus className="h-4 w-4 text-muted-foreground" />
                        </button>
                      )}
                      {meals.length > 0 && (
                        <button
                          onClick={() => openAddDialog(day, slot.value)}
                          className="w-full mt-1 flex items-center justify-center"
                        >
                          <Plus className="h-3 w-3 text-muted-foreground hover:text-primary" />
                        </button>
                      )}
                    </div>
                  );
                })}
              </div>
            ))}
          </div>
        </ScrollArea>

        {/* Add Meal Dialog */}
        <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
          <DialogContent className="sm:max-w-md">
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2">
                <Calendar className="h-5 w-5 text-primary" />
                Schedule Meal
              </DialogTitle>
              <DialogDescription>
                {selectedDate && `Add a recipe for ${format(selectedDate, "EEEE, MMMM d")}`}
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4 pt-4">
              <div className="space-y-2">
                <label className="text-sm font-medium">Date</label>
                <Popover>
                  <PopoverTrigger asChild>
                    <Button variant="outline" className="w-full justify-start">
                      <Calendar className="h-4 w-4 mr-2" />
                      {selectedDate ? format(selectedDate, "PPP") : "Select date"}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-0" align="start">
                    <CalendarPicker
                      mode="single"
                      selected={selectedDate || undefined}
                      onSelect={(date) => date && setSelectedDate(date)}
                      initialFocus
                      className={cn("p-3 pointer-events-auto")}
                    />
                  </PopoverContent>
                </Popover>
              </div>

              <div className="space-y-2">
                <label className="text-sm font-medium">Meal</label>
                <Select value={selectedSlot} onValueChange={(v) => setSelectedSlot(v as MealSlot)}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {MEAL_SLOTS.map((slot) => (
                      <SelectItem key={slot.value} value={slot.value}>
                        <span className="flex items-center gap-2">
                          <span>{slot.emoji}</span>
                          <span>{slot.label}</span>
                        </span>
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <label className="text-sm font-medium">Recipe</label>
                <Select value={selectedRecipeId} onValueChange={setSelectedRecipeId}>
                  <SelectTrigger>
                    <SelectValue placeholder="Select a recipe" />
                  </SelectTrigger>
                  <SelectContent>
                    <ScrollArea className="h-[200px]">
                      {recipes.map((recipe) => (
                        <SelectItem key={recipe.id} value={recipe.id}>
                          <span className="truncate">{recipe.title}</span>
                        </SelectItem>
                      ))}
                    </ScrollArea>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <label className="text-sm font-medium flex items-center gap-2">
                  <Bell className="h-4 w-4" />
                  Reminder
                </label>
                <Select 
                  value={selectedReminder || "none"} 
                  onValueChange={(v) => setSelectedReminder(v === "none" ? null : v as ReminderTime)}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {REMINDER_OPTIONS.map((option) => (
                      <SelectItem key={option.value || "none"} value={option.value || "none"}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="flex justify-end gap-2 pt-2">
                <Button variant="outline" onClick={() => setIsAddDialogOpen(false)}>
                  Cancel
                </Button>
                <Button onClick={handleAddMeal} disabled={!selectedRecipeId}>
                  Add to Plan
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      </CardContent>
    </Card>
  );
}