import { useState, useEffect, useRef } from 'react';
import { Link } from 'react-router-dom';
import { 
  Sheet, 
  SheetContent, 
  SheetHeader, 
  SheetTitle, 
  SheetTrigger 
} from '@/components/ui/sheet';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';
import { Checkbox } from '@/components/ui/checkbox';
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from '@/components/ui/popover';
import { 
  MessageCircle, 
  ArrowLeft, 
  Send, 
  Loader2,
  User,
  Check,
  CheckCheck,
  SmilePlus,
  Search,
  X,
  Paperclip,
  Image,
  FileText,
  Download,
  Archive,
  ArchiveRestore,
  MoreVertical,
  Mic,
  Square,
  Play,
  Pause,
  Users,
  Plus,
  LogOut
} from 'lucide-react';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
  DropdownMenuSeparator,
} from '@/components/ui/dropdown-menu';
import { useMessaging, Conversation, Message, GroupParticipant } from '@/hooks/useMessaging';
import { useAuth } from '@/hooks/useAuth';
import { useTypingIndicator } from '@/hooks/useTypingIndicator';
import { useMessageReactions, REACTION_EMOJIS, GroupedReaction } from '@/hooks/useMessageReactions';
import { useVoiceRecorder, formatDuration } from '@/hooks/useVoiceRecorder';
import { userApi } from '@/lib/apiServices';
import { formatDistanceToNow } from 'date-fns';
import { cn } from '@/lib/utils';
import { motion, AnimatePresence } from 'framer-motion';

// Typing Indicator Component
const TypingIndicator = ({ text }: { text: string }) => (
  <motion.div
    initial={{ opacity: 0, y: 10 }}
    animate={{ opacity: 1, y: 0 }}
    exit={{ opacity: 0, y: 10 }}
    className="flex items-center gap-2 px-4 py-2"
  >
    <div className="flex gap-1">
      <motion.span
        className="w-2 h-2 bg-muted-foreground/50 rounded-full"
        animate={{ y: [0, -4, 0] }}
        transition={{ duration: 0.6, repeat: Infinity, delay: 0 }}
      />
      <motion.span
        className="w-2 h-2 bg-muted-foreground/50 rounded-full"
        animate={{ y: [0, -4, 0] }}
        transition={{ duration: 0.6, repeat: Infinity, delay: 0.2 }}
      />
      <motion.span
        className="w-2 h-2 bg-muted-foreground/50 rounded-full"
        animate={{ y: [0, -4, 0] }}
        transition={{ duration: 0.6, repeat: Infinity, delay: 0.4 }}
      />
    </div>
    <span className="text-xs text-muted-foreground">{text}</span>
  </motion.div>
);

// Message Reactions Component
const MessageReactions = ({ 
  reactions, 
  onToggle 
}: { 
  reactions: GroupedReaction[];
  onToggle: (emoji: string) => void;
}) => {
  if (reactions.length === 0) return null;
  
  return (
    <div className="flex flex-wrap gap-1 mt-1">
      {reactions.map(r => (
        <button
          key={r.emoji}
          onClick={() => onToggle(r.emoji)}
          className={cn(
            "inline-flex items-center gap-0.5 px-1.5 py-0.5 rounded-full text-xs transition-colors",
            r.userReacted 
              ? "bg-primary/20 hover:bg-primary/30" 
              : "bg-muted hover:bg-muted/80"
          )}
        >
          <span>{r.emoji}</span>
          <span className="text-muted-foreground">{r.count}</span>
        </button>
      ))}
    </div>
  );
};

// Reaction Picker Component
const ReactionPicker = ({ 
  onSelect, 
  isOwn 
}: { 
  onSelect: (emoji: string) => void;
  isOwn: boolean;
}) => {
  const [open, setOpen] = useState(false);
  
  return (
    <Popover open={open} onOpenChange={setOpen}>
      <PopoverTrigger asChild>
        <button
          className={cn(
            "opacity-0 group-hover:opacity-100 transition-opacity p-1 rounded hover:bg-muted",
            isOwn ? "order-first mr-1" : "order-last ml-1"
          )}
        >
          <SmilePlus className="h-4 w-4 text-muted-foreground" />
        </button>
      </PopoverTrigger>
      <PopoverContent className="w-auto p-2" side="top">
        <div className="flex gap-1">
          {REACTION_EMOJIS.map(emoji => (
            <button
              key={emoji}
              onClick={() => {
                onSelect(emoji);
                setOpen(false);
              }}
              className="text-xl hover:scale-125 transition-transform p-1"
            >
              {emoji}
            </button>
          ))}
        </div>
      </PopoverContent>
    </Popover>
  );
};

// Search Results Component
interface SearchResult {
  message: Message;
  conversation: Conversation;
}

const SearchResults = ({ 
  results, 
  loading,
  query,
  onSelectResult,
  onClear
}: { 
  results: SearchResult[];
  loading: boolean;
  query: string;
  onSelectResult: (result: SearchResult) => void;
  onClear: () => void;
}) => {
  const highlightMatch = (text: string, query: string) => {
    if (!query.trim()) return text;
    const regex = new RegExp(`(${query.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')})`, 'gi');
    const parts = text.split(regex);
    return parts.map((part, i) => 
      regex.test(part) ? (
        <mark key={i} className="bg-yellow-200 dark:bg-yellow-800 rounded px-0.5">{part}</mark>
      ) : part
    );
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center py-12">
        <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
      </div>
    );
  }

  if (results.length === 0 && query) {
    return (
      <div className="text-center py-12 px-4">
        <Search className="h-12 w-12 mx-auto text-muted-foreground mb-3" />
        <h3 className="font-semibold mb-2">No results found</h3>
        <p className="text-sm text-muted-foreground">
          Try a different search term
        </p>
        <Button variant="ghost" size="sm" className="mt-4" onClick={onClear}>
          Clear search
        </Button>
      </div>
    );
  }

  return (
    <div className="divide-y">
      {results.map(result => {
        const displayName = result.conversation.other_user?.display_name || 'User';
        const initials = displayName.substring(0, 2).toUpperCase();

        return (
          <button
            key={result.message.id}
            onClick={() => onSelectResult(result)}
            className="w-full flex items-start gap-3 p-4 hover:bg-muted/50 transition-colors text-left"
          >
            <Avatar className="mt-0.5">
              <AvatarImage src={result.conversation.other_user?.avatar_url || undefined} />
              <AvatarFallback className="bg-primary/10 text-primary">
                {initials}
              </AvatarFallback>
            </Avatar>
            
            <div className="flex-1 min-w-0">
              <div className="flex items-center justify-between">
                <span className="font-medium truncate">{displayName}</span>
                <span className="text-xs text-muted-foreground">
                  {formatDistanceToNow(new Date(result.message.created_at), { addSuffix: true })}
                </span>
              </div>
              <p className="text-sm text-muted-foreground line-clamp-2">
                {highlightMatch(result.message.content, query)}
              </p>
            </div>
          </button>
        );
      })}
    </div>
  );
};

// Group Avatar Component
const GroupAvatarStack = ({ participants }: { participants: GroupParticipant[] }) => {
  const displayParticipants = participants.slice(0, 3);
  
  return (
    <div className="relative h-10 w-10">
      {displayParticipants.map((p, i) => (
        <Avatar 
          key={p.user_id}
          className={cn(
            "h-6 w-6 absolute border-2 border-background",
            i === 0 && "top-0 left-0",
            i === 1 && "top-0 right-0",
            i === 2 && "bottom-0 left-1/2 -translate-x-1/2"
          )}
        >
          <AvatarImage src={p.avatar_url || undefined} />
          <AvatarFallback className="text-[10px] bg-primary/10 text-primary">
            {(p.display_name || 'U').substring(0, 1).toUpperCase()}
          </AvatarFallback>
        </Avatar>
      ))}
      {participants.length === 0 && (
        <div className="h-10 w-10 rounded-full bg-muted flex items-center justify-center">
          <Users className="h-5 w-5 text-muted-foreground" />
        </div>
      )}
    </div>
  );
};

// Create Group Dialog
const CreateGroupDialog = ({ 
  open, 
  onOpenChange,
  onCreateGroup
}: { 
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onCreateGroup: (name: string, participants: string[]) => Promise<void>;
}) => {
  const [groupName, setGroupName] = useState('');
  const [searchQuery, setSearchQuery] = useState('');
  const [searchResults, setSearchResults] = useState<{ user_id: string; display_name: string | null; avatar_url: string | null }[]>([]);
  const [selectedUsers, setSelectedUsers] = useState<{ user_id: string; display_name: string | null; avatar_url: string | null }[]>([]);
  const [searching, setSearching] = useState(false);
  const [creating, setCreating] = useState(false);
  const { user } = useAuth();
  const searchTimeoutRef = useRef<NodeJS.Timeout | null>(null);

  const handleSearch = (query: string) => {
    setSearchQuery(query);
    
    if (searchTimeoutRef.current) {
      clearTimeout(searchTimeoutRef.current);
    }

    if (!query.trim()) {
      setSearchResults([]);
      setSearching(false);
      return;
    }

    setSearching(true);
    searchTimeoutRef.current = setTimeout(async () => {
      try {
        const data = await userApi.searchProfiles(query);
        setSearchResults(data || []);
      } catch {
        setSearchResults([]);
      }
      setSearching(false);
    }, 300);
  };

  const toggleUser = (profile: { user_id: string; display_name: string | null; avatar_url: string | null }) => {
    setSelectedUsers(prev => {
      const exists = prev.find(u => u.user_id === profile.user_id);
      if (exists) {
        return prev.filter(u => u.user_id !== profile.user_id);
      }
      return [...prev, profile];
    });
  };

  const handleCreate = async () => {
    if (selectedUsers.length < 2) return;
    setCreating(true);
    await onCreateGroup(groupName, selectedUsers.map(u => u.user_id));
    setCreating(false);
    setGroupName('');
    setSelectedUsers([]);
    setSearchQuery('');
    setSearchResults([]);
    onOpenChange(false);
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle>Create Group Chat</DialogTitle>
        </DialogHeader>
        
        <div className="space-y-4">
          <Input
            placeholder="Group name"
            value={groupName}
            onChange={(e) => setGroupName(e.target.value)}
          />

          {/* Selected users */}
          {selectedUsers.length > 0 && (
            <div className="flex flex-wrap gap-2">
              {selectedUsers.map(u => (
                <Badge key={u.user_id} variant="secondary" className="flex items-center gap-1">
                  {u.display_name || 'User'}
                  <button onClick={() => toggleUser(u)} className="ml-1 hover:text-destructive">
                    <X className="h-3 w-3" />
                  </button>
                </Badge>
              ))}
            </div>
          )}

          {/* Search users */}
          <div className="relative">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search users to add..."
              value={searchQuery}
              onChange={(e) => handleSearch(e.target.value)}
              className="pl-9"
            />
          </div>

          {/* Search results */}
          <ScrollArea className="h-48">
            {searching ? (
              <div className="flex items-center justify-center py-8">
                <Loader2 className="h-5 w-5 animate-spin text-muted-foreground" />
              </div>
            ) : searchResults.length > 0 ? (
              <div className="space-y-1">
                {searchResults.map(profile => {
                  const isSelected = selectedUsers.some(u => u.user_id === profile.user_id);
                  return (
                    <button
                      key={profile.user_id}
                      onClick={() => toggleUser(profile)}
                      className={cn(
                        "w-full flex items-center gap-3 p-2 rounded-lg transition-colors",
                        isSelected ? "bg-primary/10" : "hover:bg-muted"
                      )}
                    >
                      <Avatar className="h-8 w-8">
                        <AvatarImage src={profile.avatar_url || undefined} />
                        <AvatarFallback className="text-xs">
                          {(profile.display_name || 'U').substring(0, 2).toUpperCase()}
                        </AvatarFallback>
                      </Avatar>
                      <span className="flex-1 text-left text-sm">{profile.display_name || 'User'}</span>
                      {isSelected && <Check className="h-4 w-4 text-primary" />}
                    </button>
                  );
                })}
              </div>
            ) : searchQuery ? (
              <p className="text-center text-sm text-muted-foreground py-8">No users found</p>
            ) : (
              <p className="text-center text-sm text-muted-foreground py-8">Search for users to add</p>
            )}
          </ScrollArea>
        </div>

        <DialogFooter>
          <Button variant="outline" onClick={() => onOpenChange(false)}>Cancel</Button>
          <Button 
            onClick={handleCreate} 
            disabled={selectedUsers.length < 2 || creating}
          >
            {creating ? <Loader2 className="h-4 w-4 animate-spin" /> : 'Create Group'}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
};

// Conversation List View
const ConversationList = ({ 
  conversations, 
  loading,
  onSelectConversation,
  onArchive,
  onUnarchive,
  onLeaveGroup,
  showArchived,
  onToggleArchived,
  archivedCount,
  onCreateGroup
}: { 
  conversations: Conversation[];
  loading: boolean;
  onSelectConversation: (conv: Conversation) => void;
  onArchive: (id: string) => void;
  onUnarchive: (id: string) => void;
  onLeaveGroup: (id: string) => void;
  showArchived: boolean;
  onToggleArchived: () => void;
  archivedCount: number;
  onCreateGroup: () => void;
}) => {
  if (loading) {
    return (
      <div className="flex items-center justify-center py-12">
        <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
      </div>
    );
  }

  if (conversations.length === 0) {
    return (
      <div className="text-center py-12 px-4">
        <MessageCircle className="h-12 w-12 mx-auto text-muted-foreground mb-3" />
        <h3 className="font-semibold mb-2">
          {showArchived ? 'No archived chats' : 'No messages yet'}
        </h3>
        <p className="text-sm text-muted-foreground">
          {showArchived 
            ? 'Archived conversations will appear here'
            : 'Start a conversation by visiting someone\'s profile and clicking "Message"'
          }
        </p>
        {showArchived && (
          <Button variant="ghost" size="sm" className="mt-4" onClick={onToggleArchived}>
            Back to messages
          </Button>
        )}
      </div>
    );
  }

  return (
    <div>
      {/* Create group button */}
      {!showArchived && (
        <button
          onClick={onCreateGroup}
          className="w-full flex items-center gap-3 p-4 hover:bg-muted/50 transition-colors text-left border-b"
        >
          <div className="h-10 w-10 rounded-full bg-primary/10 flex items-center justify-center">
            <Plus className="h-5 w-5 text-primary" />
          </div>
          <span className="font-medium">Create Group Chat</span>
        </button>
      )}

      {/* Archived toggle */}
      {!showArchived && archivedCount > 0 && (
        <button
          onClick={onToggleArchived}
          className="w-full flex items-center gap-3 p-4 hover:bg-muted/50 transition-colors text-left border-b"
        >
          <div className="h-10 w-10 rounded-full bg-muted flex items-center justify-center">
            <Archive className="h-5 w-5 text-muted-foreground" />
          </div>
          <div>
            <span className="font-medium">Archived</span>
            <p className="text-sm text-muted-foreground">{archivedCount} conversation{archivedCount !== 1 ? 's' : ''}</p>
          </div>
        </button>
      )}

      {showArchived && (
        <button
          onClick={onToggleArchived}
          className="w-full flex items-center gap-2 p-3 hover:bg-muted/50 transition-colors text-left border-b text-sm text-muted-foreground"
        >
          <ArrowLeft className="h-4 w-4" />
          Back to messages
        </button>
      )}

      <div className="divide-y">
        {conversations.map(conv => {
          const isGroup = conv.is_group;
          const displayName = isGroup 
            ? conv.name || 'Group Chat'
            : conv.other_user?.display_name || 'User';
          const initials = displayName.substring(0, 2).toUpperCase();

          return (
            <div
              key={conv.id}
              className="flex items-center gap-3 p-4 hover:bg-muted/50 transition-colors group"
            >
              <button
                onClick={() => onSelectConversation(conv)}
                className="flex items-center gap-3 flex-1 text-left"
              >
                {isGroup ? (
                  <GroupAvatarStack participants={conv.participants || []} />
                ) : (
                  <Avatar>
                    <AvatarImage src={conv.other_user?.avatar_url || undefined} />
                    <AvatarFallback className="bg-primary/10 text-primary">
                      {initials}
                    </AvatarFallback>
                  </Avatar>
                )}
                
                <div className="flex-1 min-w-0">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <span className="font-medium truncate">{displayName}</span>
                      {isGroup && <Users className="h-3.5 w-3.5 text-muted-foreground" />}
                    </div>
                    {(conv.unread_count || 0) > 0 && (
                      <Badge variant="default" className="ml-2">
                        {conv.unread_count}
                      </Badge>
                    )}
                  </div>
                  {conv.last_message && (
                    <p className="text-sm text-muted-foreground truncate">
                      {conv.last_message}
                    </p>
                  )}
                </div>
              </button>

              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button 
                    variant="ghost" 
                    size="icon" 
                    className="h-8 w-8 opacity-0 group-hover:opacity-100 transition-opacity"
                  >
                    <MoreVertical className="h-4 w-4" />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end">
                  {isGroup && (
                    <>
                      <DropdownMenuItem onClick={() => onLeaveGroup(conv.id)}>
                        <LogOut className="h-4 w-4 mr-2" />
                        Leave Group
                      </DropdownMenuItem>
                      <DropdownMenuSeparator />
                    </>
                  )}
                  {conv.isArchived ? (
                    <DropdownMenuItem onClick={() => onUnarchive(conv.id)}>
                      <ArchiveRestore className="h-4 w-4 mr-2" />
                      Unarchive
                    </DropdownMenuItem>
                  ) : (
                    <DropdownMenuItem onClick={() => onArchive(conv.id)}>
                      <Archive className="h-4 w-4 mr-2" />
                      Archive
                    </DropdownMenuItem>
                  )}
                </DropdownMenuContent>
              </DropdownMenu>
            </div>
          );
        })}
      </div>
    </div>
  );
};

// Chat View
const ChatView = ({ 
  conversation,
  onBack 
}: { 
  conversation: Conversation;
  onBack: () => void;
}) => {
  const { user } = useAuth();
  const { sendMessage, uploadAttachment, uploadVoiceMessage, getMessages, markAsRead } = useMessaging();
  const { typingText, isTyping, startTyping, stopTyping } = useTypingIndicator(conversation.id);
  const { fetchReactions, toggleReaction, getGroupedReactions } = useMessageReactions(conversation.id);
  const voiceRecorder = useVoiceRecorder();
  const [messages, setMessages] = useState<Message[]>([]);
  const [newMessage, setNewMessage] = useState('');
  const [loading, setLoading] = useState(true);
  const [sending, setSending] = useState(false);
  const [pendingAttachment, setPendingAttachment] = useState<{file: File; preview?: string} | null>(null);
  const [uploading, setUploading] = useState(false);
  const [playingVoiceId, setPlayingVoiceId] = useState<string | null>(null);
  const audioRef = useRef<HTMLAudioElement | null>(null);
  const scrollRef = useRef<HTMLDivElement>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const isGroup = conversation.is_group;
  const displayName = isGroup 
    ? conversation.name || 'Group Chat' 
    : conversation.other_user?.display_name || 'User';

  useEffect(() => {
    loadMessages();
    markAsRead(conversation.id);
    const interval = setInterval(() => loadMessages(), 4000);
    return () => clearInterval(interval);
  }, [conversation.id]);

  useEffect(() => {
    if (scrollRef.current) scrollRef.current.scrollTop = scrollRef.current.scrollHeight;
  }, [messages]);

  const loadMessages = async () => {
    setLoading(true);
    const msgs = await getMessages(conversation.id);
    setMessages(msgs);
    if (msgs.length > 0) fetchReactions(msgs.map(m => m.id));
    setLoading(false);
  };

  const handleSend = async () => {
    if ((!newMessage.trim() && !pendingAttachment) || sending) return;
    setSending(true);
    stopTyping();

    let attachment: { url: string; type: string; name: string } | undefined;
    if (pendingAttachment) {
      setUploading(true);
      const url = await uploadAttachment(pendingAttachment.file, conversation.id);
      setUploading(false);
      if (url) attachment = { url, type: pendingAttachment.file.type, name: pendingAttachment.file.name };
    }

    const success = await sendMessage(conversation.id, newMessage.trim() || '', attachment);
    if (success) { setNewMessage(''); setPendingAttachment(null); }
    setSending(false);
  };

  const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file || file.size > 20 * 1024 * 1024) return;
    if (file.type.startsWith('image/')) {
      const reader = new FileReader();
      reader.onload = (e) => setPendingAttachment({ file, preview: e.target?.result as string });
      reader.readAsDataURL(file);
    } else {
      setPendingAttachment({ file });
    }
  };

  const handleSendVoice = async () => {
    if (!voiceRecorder.audioBlob) return;
    setSending(true);
    setUploading(true);
    const voiceData = await uploadVoiceMessage(voiceRecorder.audioBlob, conversation.id, voiceRecorder.duration);
    setUploading(false);
    if (voiceData) await sendMessage(conversation.id, '', undefined, voiceData);
    voiceRecorder.resetRecording();
    setSending(false);
  };

  const playVoiceMessage = (messageId: string, url: string) => {
    if (playingVoiceId === messageId) { audioRef.current?.pause(); setPlayingVoiceId(null); return; }
    if (audioRef.current) audioRef.current.pause();
    const audio = new Audio(url);
    audioRef.current = audio;
    audio.onended = () => setPlayingVoiceId(null);
    audio.onerror = () => setPlayingVoiceId(null);
    audio.play();
    setPlayingVoiceId(messageId);
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setNewMessage(e.target.value);
    if (e.target.value.trim()) startTyping(user?.email?.split('@')[0] || 'User');
    else stopTyping();
  };

  const formatTime = (dateString: string) => {
    try { return formatDistanceToNow(new Date(dateString), { addSuffix: true }); }
    catch { return ''; }
  };

  return (
    <div className="flex flex-col h-full">
      <div className="flex items-center gap-3 p-4 border-b">
        <Button variant="ghost" size="icon" onClick={onBack}><ArrowLeft className="h-5 w-5" /></Button>
        {isGroup ? (
          <div className="flex items-center gap-3 flex-1">
            <GroupAvatarStack participants={conversation.participants || []} />
            <div className="flex-1 min-w-0">
              <span className="font-medium">{displayName}</span>
              <p className="text-xs text-muted-foreground">{conversation.participants?.length || 0} members</p>
            </div>
          </div>
        ) : (
          <Link to={`/user/${conversation.other_user?.user_id}`} className="flex items-center gap-3 flex-1">
            <Avatar>
              <AvatarImage src={conversation.other_user?.avatar_url || undefined} />
              <AvatarFallback className="bg-primary/10 text-primary">{displayName.substring(0, 2).toUpperCase()}</AvatarFallback>
            </Avatar>
            <span className="font-medium hover:underline">{displayName}</span>
          </Link>
        )}
      </div>

      <ScrollArea ref={scrollRef} className="flex-1 p-4">
        {loading ? (
          <div className="flex items-center justify-center py-12"><Loader2 className="h-6 w-6 animate-spin text-muted-foreground" /></div>
        ) : messages.length === 0 ? (
          <div className="text-center py-12"><p className="text-sm text-muted-foreground">Start the conversation by sending a message</p></div>
        ) : (
          <div className="space-y-4">
            {messages.map(msg => {
              const isOwn = msg.sender_id === user?.id;
              const messageReactions = getGroupedReactions(msg.id);
              const senderName = msg.sender_profile?.display_name || 'User';
              return (
                <div key={msg.id} className={cn("flex group", isOwn ? "justify-end" : "justify-start")}>
                  <div className="flex items-start gap-2">
                    {isGroup && !isOwn && (
                      <Avatar className="h-7 w-7 mt-0.5">
                        <AvatarImage src={msg.sender_profile?.avatar_url || undefined} />
                        <AvatarFallback className="text-xs bg-muted">{senderName.substring(0, 1).toUpperCase()}</AvatarFallback>
                      </Avatar>
                    )}
                    <div className="flex items-center">
                      <ReactionPicker onSelect={(emoji) => toggleReaction(msg.id, emoji)} isOwn={isOwn} />
                      <div className={cn("max-w-[75%] rounded-2xl px-4 py-2", isOwn ? "bg-primary text-primary-foreground rounded-br-md" : "bg-muted rounded-bl-md")}>
                        {isGroup && !isOwn && <p className="text-xs font-medium mb-1 opacity-70">{senderName}</p>}
                        {msg.voice_url && (
                          <div className="mb-2">
                            <button onClick={() => playVoiceMessage(msg.id, msg.voice_url!)} className={cn("flex items-center gap-3 p-2 rounded-lg min-w-[140px]", isOwn ? "bg-primary-foreground/10" : "bg-background")}>
                              <div className={cn("h-8 w-8 rounded-full flex items-center justify-center", isOwn ? "bg-primary-foreground/20" : "bg-primary/10")}>
                                {playingVoiceId === msg.id ? <Pause className="h-4 w-4" /> : <Play className="h-4 w-4 ml-0.5" />}
                              </div>
                              <span className="text-xs opacity-70">{formatDuration(msg.voice_duration || 0)}</span>
                            </button>
                          </div>
                        )}
                        {msg.attachment_url && (
                          <div className="mb-2">
                            {msg.attachment_type?.startsWith('image/') ? (
                              <a href={msg.attachment_url} target="_blank" rel="noopener noreferrer">
                                <img src={msg.attachment_url} alt={msg.attachment_name || 'Image'} className="max-w-full rounded-lg max-h-48 object-cover" />
                              </a>
                            ) : (
                              <a href={msg.attachment_url} target="_blank" rel="noopener noreferrer" className={cn("flex items-center gap-2 p-2 rounded-lg", isOwn ? "bg-primary-foreground/10" : "bg-background")}>
                                <FileText className="h-5 w-5 flex-shrink-0" />
                                <span className="text-sm truncate flex-1">{msg.attachment_name}</span>
                                <Download className="h-4 w-4 flex-shrink-0" />
                              </a>
                            )}
                          </div>
                        )}
                        {msg.content && <p className="text-sm whitespace-pre-wrap">{msg.content}</p>}
                        <div className={cn("flex items-center justify-end gap-1 mt-1", isOwn ? "text-primary-foreground/70" : "text-muted-foreground")}>
                          <span className="text-xs">{formatTime(msg.created_at)}</span>
                          {isOwn && (msg.is_read ? <CheckCheck className="h-3.5 w-3.5 text-blue-400" /> : <Check className="h-3.5 w-3.5" />)}
                        </div>
                        <MessageReactions reactions={messageReactions} onToggle={(emoji) => toggleReaction(msg.id, emoji)} />
                      </div>
                    </div>
                  </div>
                </div>
              );
            })}
          </div>
        )}
      </ScrollArea>

      <AnimatePresence>{isTyping && typingText && <TypingIndicator text={typingText} />}</AnimatePresence>

      <AnimatePresence>
        {pendingAttachment && (
          <motion.div initial={{ height: 0, opacity: 0 }} animate={{ height: 'auto', opacity: 1 }} exit={{ height: 0, opacity: 0 }} className="px-4 pt-2 border-t">
            <div className="flex items-center gap-2 p-2 bg-muted rounded-lg">
              {pendingAttachment.preview ? <img src={pendingAttachment.preview} alt="Preview" className="h-12 w-12 object-cover rounded" /> : <div className="h-12 w-12 bg-background rounded flex items-center justify-center"><FileText className="h-6 w-6 text-muted-foreground" /></div>}
              <div className="flex-1 min-w-0"><p className="text-sm font-medium truncate">{pendingAttachment.file.name}</p><p className="text-xs text-muted-foreground">{(pendingAttachment.file.size / 1024).toFixed(1)} KB</p></div>
              <Button variant="ghost" size="icon" className="h-8 w-8" onClick={() => setPendingAttachment(null)}><X className="h-4 w-4" /></Button>
            </div>
          </motion.div>
        )}
      </AnimatePresence>

      <AnimatePresence>
        {(voiceRecorder.isRecording || voiceRecorder.audioBlob) && (
          <motion.div initial={{ height: 0, opacity: 0 }} animate={{ height: 'auto', opacity: 1 }} exit={{ height: 0, opacity: 0 }} className="px-4 pt-2 border-t">
            <div className="flex items-center gap-3 p-3 bg-muted rounded-lg">
              {voiceRecorder.isRecording ? (
                <>
                  <motion.div className="h-3 w-3 bg-red-500 rounded-full" animate={{ opacity: [1, 0.3, 1] }} transition={{ duration: 1, repeat: Infinity }} />
                  <span className="text-sm font-medium flex-1">Recording... {formatDuration(voiceRecorder.duration)}</span>
                  <Button variant="ghost" size="icon" className="h-8 w-8" onClick={voiceRecorder.cancelRecording}><X className="h-4 w-4" /></Button>
                  <Button variant="default" size="icon" className="h-8 w-8" onClick={voiceRecorder.stopRecording}><Square className="h-4 w-4" /></Button>
                </>
              ) : voiceRecorder.audioBlob && (
                <>
                  <Button variant="ghost" size="icon" className="h-8 w-8" onClick={() => { if (voiceRecorder.audioUrl) new Audio(voiceRecorder.audioUrl).play(); }}><Play className="h-4 w-4" /></Button>
                  <span className="text-sm flex-1">{formatDuration(voiceRecorder.duration)}</span>
                  <Button variant="ghost" size="icon" className="h-8 w-8" onClick={voiceRecorder.resetRecording}><X className="h-4 w-4" /></Button>
                  <Button size="icon" className="h-8 w-8" onClick={handleSendVoice} disabled={sending}>{sending ? <Loader2 className="h-4 w-4 animate-spin" /> : <Send className="h-4 w-4" />}</Button>
                </>
              )}
            </div>
          </motion.div>
        )}
      </AnimatePresence>

      <div className="p-4 border-t">
        <div className="flex gap-2">
          <input type="file" ref={fileInputRef} onChange={handleFileSelect} className="hidden" accept="image/*,.pdf,.doc,.docx,.txt" />
          <Button variant="ghost" size="icon" onClick={() => fileInputRef.current?.click()} disabled={sending || uploading || voiceRecorder.isRecording}><Paperclip className="h-5 w-5" /></Button>
          <Input placeholder="Type a message..." value={newMessage} onChange={handleInputChange} onKeyDown={(e) => e.key === 'Enter' && !e.shiftKey && handleSend()} onBlur={stopTyping} disabled={sending || voiceRecorder.isRecording} className="flex-1" />
          {!newMessage.trim() && !pendingAttachment ? (
            <Button variant="ghost" size="icon" onClick={voiceRecorder.isRecording ? voiceRecorder.stopRecording : voiceRecorder.startRecording} disabled={sending || uploading} className={cn(voiceRecorder.isRecording && "text-red-500")}><Mic className="h-5 w-5" /></Button>
          ) : (
            <Button onClick={handleSend} disabled={(!newMessage.trim() && !pendingAttachment) || sending || uploading}>{sending || uploading ? <Loader2 className="h-4 w-4 animate-spin" /> : <Send className="h-4 w-4" />}</Button>
          )}
        </div>
      </div>
    </div>
  );
};

// Main Messages Component
export const MessagesDrawer = () => {
  const { user } = useAuth();
  const { 
    activeConversations, 
    archivedConversations, 
    loading, 
    totalUnread, 
    searchMessages,
    archiveConversation,
    unarchiveConversation,
    createGroupChat,
    leaveGroup,
  } = useMessaging();
  const [selectedConversation, setSelectedConversation] = useState<Conversation | null>(null);
  const [open, setOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [searchResults, setSearchResults] = useState<SearchResult[]>([]);
  const [searching, setSearching] = useState(false);
  const [showSearch, setShowSearch] = useState(false);
  const [showArchived, setShowArchived] = useState(false);
  const [showCreateGroup, setShowCreateGroup] = useState(false);
  const searchTimeoutRef = useRef<NodeJS.Timeout | null>(null);

  const handleSearch = (query: string) => {
    setSearchQuery(query);
    
    if (searchTimeoutRef.current) {
      clearTimeout(searchTimeoutRef.current);
    }

    if (!query.trim()) {
      setSearchResults([]);
      setSearching(false);
      return;
    }

    setSearching(true);
    searchTimeoutRef.current = setTimeout(async () => {
      const results = await searchMessages(query);
      setSearchResults(results);
      setSearching(false);
    }, 300);
  };

  const handleClearSearch = () => {
    setSearchQuery('');
    setSearchResults([]);
    setShowSearch(false);
  };

  const handleSelectSearchResult = (result: SearchResult) => {
    setSelectedConversation(result.conversation);
    handleClearSearch();
  };

  const handleArchive = async (id: string) => {
    await archiveConversation(id);
  };

  const handleUnarchive = async (id: string) => {
    await unarchiveConversation(id);
  };

  const handleLeaveGroup = async (id: string) => {
    await leaveGroup(id);
  };

  const handleCreateGroup = async (name: string, participants: string[]) => {
    await createGroupChat(name, participants);
  };

  if (!user) {
    return null;
  }

  return (
    <Sheet open={open} onOpenChange={setOpen}>
      <SheetTrigger asChild>
        <Button variant="ghost" size="icon" className="relative">
          <MessageCircle className="h-5 w-5" />
          {totalUnread > 0 && (
            <Badge 
              variant="destructive" 
              className="absolute -top-1 -right-1 h-5 w-5 flex items-center justify-center p-0 text-xs"
            >
              {totalUnread > 9 ? '9+' : totalUnread}
            </Badge>
          )}
        </Button>
      </SheetTrigger>
      
      <SheetContent className="w-full sm:max-w-md p-0 flex flex-col">
        {selectedConversation ? (
          <ChatView 
            conversation={selectedConversation} 
            onBack={() => setSelectedConversation(null)}
          />
        ) : (
          <>
            <SheetHeader className="p-4 border-b space-y-3">
              <div className="flex items-center justify-between">
                <SheetTitle className="flex items-center gap-2">
                  <MessageCircle className="h-5 w-5" />
                  Messages
                </SheetTitle>
                <Button 
                  variant="ghost" 
                  size="icon"
                  onClick={() => setShowSearch(!showSearch)}
                  className={cn(showSearch && "bg-muted")}
                >
                  <Search className="h-4 w-4" />
                </Button>
              </div>
              
              <AnimatePresence>
                {showSearch && (
                  <motion.div
                    initial={{ height: 0, opacity: 0 }}
                    animate={{ height: 'auto', opacity: 1 }}
                    exit={{ height: 0, opacity: 0 }}
                    className="overflow-hidden"
                  >
                    <div className="relative">
                      <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                      <Input
                        placeholder="Search messages..."
                        value={searchQuery}
                        onChange={(e) => handleSearch(e.target.value)}
                        className="pl-9 pr-9"
                        autoFocus
                      />
                      {searchQuery && (
                        <Button
                          variant="ghost"
                          size="icon"
                          className="absolute right-1 top-1/2 -translate-y-1/2 h-7 w-7"
                          onClick={handleClearSearch}
                        >
                          <X className="h-3 w-3" />
                        </Button>
                      )}
                    </div>
                  </motion.div>
                )}
              </AnimatePresence>
            </SheetHeader>
            
            <ScrollArea className="flex-1">
              {searchQuery ? (
                <SearchResults
                  results={searchResults}
                  loading={searching}
                  query={searchQuery}
                  onSelectResult={handleSelectSearchResult}
                  onClear={handleClearSearch}
                />
              ) : (
                <ConversationList 
                  conversations={showArchived ? archivedConversations : activeConversations}
                  loading={loading}
                  onSelectConversation={setSelectedConversation}
                  onArchive={handleArchive}
                  onUnarchive={handleUnarchive}
                  onLeaveGroup={handleLeaveGroup}
                  showArchived={showArchived}
                  onToggleArchived={() => setShowArchived(!showArchived)}
                  archivedCount={archivedConversations.length}
                  onCreateGroup={() => setShowCreateGroup(true)}
                />
              )}
            </ScrollArea>

            <CreateGroupDialog
              open={showCreateGroup}
              onOpenChange={setShowCreateGroup}
              onCreateGroup={handleCreateGroup}
            />
          </>
        )}
      </SheetContent>
    </Sheet>
  );
};

// Message Button for profiles
export const MessageButton = ({ 
  userId,
  variant = 'outline',
  size = 'default'
}: { 
  userId: string;
  variant?: 'default' | 'outline' | 'ghost';
  size?: 'default' | 'sm' | 'lg' | 'icon';
}) => {
  const { user } = useAuth();
  const { startConversation } = useMessaging();
  const [loading, setLoading] = useState(false);

  if (!user || user.id === userId) {
    return null;
  }

  const handleClick = async () => {
    setLoading(true);
    await startConversation(userId);
    setLoading(false);
    // Could open the messages drawer here
  };

  return (
    <Button 
      variant={variant} 
      size={size} 
      onClick={handleClick}
      disabled={loading}
      className="gap-2"
    >
      {loading ? (
        <Loader2 className="h-4 w-4 animate-spin" />
      ) : (
        <MessageCircle className="h-4 w-4" />
      )}
      Message
    </Button>
  );
};
