import { useState } from 'react';
import { Link } from 'react-router-dom';
import { Bell, Check, CheckCheck, Trash2, MessageSquare, Heart, Trophy, X, UserPlus, Mail } from 'lucide-react';
import { Button } from '@/components/ui/button';
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from '@/components/ui/popover';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Badge } from '@/components/ui/badge';
import { useNotifications, Notification } from '@/hooks/useNotifications';
import { cn } from '@/lib/utils';

const notificationIcons: Record<string, React.ElementType> = {
  reply: MessageSquare,
  like: Heart,
  achievement: Trophy,
  follow: UserPlus,
  message: Mail,
  default: Bell,
};

const NotificationItem = ({ 
  notification, 
  onMarkAsRead, 
  onDelete,
  formatTimeAgo 
}: { 
  notification: Notification;
  onMarkAsRead: (id: string) => void;
  onDelete: (id: string) => void;
  formatTimeAgo: (date: string) => string;
}) => {
  const Icon = notificationIcons[notification.type] || notificationIcons.default;
  
  return (
    <div 
      className={cn(
        "flex gap-3 p-3 border-b last:border-b-0 transition-colors",
        !notification.is_read && "bg-primary/5"
      )}
    >
      <div className={cn(
        "h-8 w-8 rounded-full flex items-center justify-center flex-shrink-0",
        notification.type === 'reply' && "bg-blue-100 text-blue-600 dark:bg-blue-950 dark:text-blue-400",
        notification.type === 'like' && "bg-red-100 text-red-600 dark:bg-red-950 dark:text-red-400",
        notification.type === 'achievement' && "bg-amber-100 text-amber-600 dark:bg-amber-950 dark:text-amber-400",
        notification.type === 'follow' && "bg-emerald-100 text-emerald-600 dark:bg-emerald-950 dark:text-emerald-400",
        notification.type === 'message' && "bg-purple-100 text-purple-600 dark:bg-purple-950 dark:text-purple-400",
        !['reply', 'like', 'achievement', 'follow', 'message'].includes(notification.type) && "bg-muted text-muted-foreground"
      )}>
        <Icon className="h-4 w-4" />
      </div>
      
      <div className="flex-1 min-w-0">
        <div className="flex items-start justify-between gap-2">
          <div>
            <p className={cn(
              "text-sm",
              !notification.is_read && "font-medium"
            )}>
              {notification.title}
            </p>
            <p className="text-xs text-muted-foreground line-clamp-2">
              {notification.message}
            </p>
          </div>
          {!notification.is_read && (
            <div className="h-2 w-2 rounded-full bg-primary flex-shrink-0 mt-1.5" />
          )}
        </div>
        
        <div className="flex items-center justify-between mt-1">
          <span className="text-xs text-muted-foreground">
            {formatTimeAgo(notification.created_at)}
          </span>
          
          <div className="flex items-center gap-1">
            {notification.link && (
              <Link to={notification.link}>
                <Button variant="ghost" size="sm" className="h-6 px-2 text-xs">
                  View
                </Button>
              </Link>
            )}
            {!notification.is_read && (
              <Button 
                variant="ghost" 
                size="sm" 
                className="h-6 w-6 p-0"
                onClick={() => onMarkAsRead(notification.id)}
              >
                <Check className="h-3 w-3" />
              </Button>
            )}
            <Button 
              variant="ghost" 
              size="sm" 
              className="h-6 w-6 p-0 text-muted-foreground hover:text-destructive"
              onClick={() => onDelete(notification.id)}
            >
              <X className="h-3 w-3" />
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

export const NotificationBell = () => {
  const [open, setOpen] = useState(false);
  const {
    notifications,
    unreadCount,
    loading,
    markAsRead,
    markAllAsRead,
    deleteNotification,
    clearAllNotifications,
    formatTimeAgo,
  } = useNotifications();

  return (
    <Popover open={open} onOpenChange={setOpen}>
      <PopoverTrigger asChild>
        <Button variant="ghost" size="icon" className="relative">
          <Bell className="h-5 w-5" />
          {unreadCount > 0 && (
            <Badge 
              variant="destructive" 
              className="absolute -top-1 -right-1 h-5 w-5 flex items-center justify-center p-0 text-xs"
            >
              {unreadCount > 9 ? '9+' : unreadCount}
            </Badge>
          )}
        </Button>
      </PopoverTrigger>
      
      <PopoverContent className="w-80 p-0" align="end">
        <div className="flex items-center justify-between p-3 border-b">
          <h3 className="font-semibold">Notifications</h3>
          {notifications.length > 0 && (
            <div className="flex gap-1">
              {unreadCount > 0 && (
                <Button 
                  variant="ghost" 
                  size="sm" 
                  className="h-7 text-xs gap-1"
                  onClick={markAllAsRead}
                >
                  <CheckCheck className="h-3 w-3" />
                  Mark all read
                </Button>
              )}
              <Button 
                variant="ghost" 
                size="sm" 
                className="h-7 text-xs gap-1 text-muted-foreground hover:text-destructive"
                onClick={clearAllNotifications}
              >
                <Trash2 className="h-3 w-3" />
              </Button>
            </div>
          )}
        </div>
        
        <ScrollArea className="max-h-[400px]">
          {loading ? (
            <div className="p-8 text-center text-muted-foreground">
              Loading...
            </div>
          ) : notifications.length === 0 ? (
            <div className="p-8 text-center">
              <Bell className="h-8 w-8 mx-auto text-muted-foreground mb-2" />
              <p className="text-sm text-muted-foreground">No notifications yet</p>
            </div>
          ) : (
            notifications.map(notification => (
              <NotificationItem
                key={notification.id}
                notification={notification}
                onMarkAsRead={markAsRead}
                onDelete={deleteNotification}
                formatTimeAgo={formatTimeAgo}
              />
            ))
          )}
        </ScrollArea>
      </PopoverContent>
    </Popover>
  );
};
