import { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Loader2, AlertTriangle, CheckCircle, Leaf, Sparkles, Plus, Check } from 'lucide-react';
import { usePhytoestrogenAnalysis, PhytoAnalysisResult, PhytoCompound } from '@/hooks/usePhytoestrogenAnalysis';
import { usePhytoestrogenTracking } from '@/hooks/usePhytoestrogenTracking';
import { useAuth } from '@/hooks/useAuth';
import { Link } from 'react-router-dom';

interface PhytoestrogenAnalysisCardProps {
  ingredients: string | null;
  productName: string;
}

const getRiskColor = (risk: PhytoAnalysisResult['overallRisk']) => {
  switch (risk) {
    case 'high': return 'bg-destructive text-destructive-foreground';
    case 'moderate': return 'bg-orange-500 text-white';
    case 'low': return 'bg-yellow-500 text-black';
    case 'none': return 'bg-green-500 text-white';
  }
};

const getPotencyColor = (potency: 'weak' | 'moderate' | 'strong') => {
  switch (potency) {
    case 'strong': return 'bg-destructive/10 text-destructive border-destructive/30';
    case 'moderate': return 'bg-orange-100 text-orange-700 border-orange-300';
    case 'weak': return 'bg-yellow-100 text-yellow-700 border-yellow-300';
  }
};

export const PhytoestrogenAnalysisCard = ({ ingredients, productName }: PhytoestrogenAnalysisCardProps) => {
  const { analyzeIngredients, loading, result, reset } = usePhytoestrogenAnalysis();
  const { logExposure, isLogging } = usePhytoestrogenTracking();
  const { user } = useAuth();
  const [useAI, setUseAI] = useState(false);
  const [tracked, setTracked] = useState(false);

  const handleAnalyze = () => {
    if (ingredients) {
      setTracked(false);
      analyzeIngredients(ingredients, productName, useAI);
    }
  };

  const handleTrackExposure = () => {
    if (!result || !result.compounds.length || !user) return;

    // Log each detected compound to the user's exposure tracking
    result.compounds.forEach((compound: PhytoCompound) => {
      logExposure({
        phytoestrogen: {
          id: compound.name.toLowerCase().replace(/\s+/g, '-'),
          name: compound.name,
          class: 'other' as const,
          potency: compound.potency,
          foodSources: [compound.source],
          recommendation: 'avoid' as const,
          healthConcerns: ['Detected via product scan'],
          notes: `Detected in ${productName}`,
        },
        foodSource: `${productName} (${compound.source})`,
        servings: 1,
        notes: `Auto-tracked from barcode scan`,
      });
    });

    setTracked(true);
  };

  if (!ingredients) {
    return null;
  }

  return (
    <Card className="border-primary/20">
      <CardHeader className="pb-3">
        <CardTitle className="flex items-center gap-2 text-base">
          <Leaf className="h-4 w-4 text-primary" />
          Phytoestrogen Analysis
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-3">
        {!result && !loading && (
          <div className="space-y-3">
            <p className="text-sm text-muted-foreground">
              Analyze this product's ingredients for phytoestrogen content.
            </p>
            <div className="flex gap-2">
              <Button 
                onClick={handleAnalyze} 
                size="sm" 
                className="flex-1"
              >
                <Leaf className="h-4 w-4 mr-1" />
                Quick Analyze
              </Button>
              <Button 
                onClick={() => { setUseAI(true); handleAnalyze(); }} 
                size="sm" 
                variant="outline"
                className="flex-1"
              >
                <Sparkles className="h-4 w-4 mr-1" />
                AI Analysis
              </Button>
            </div>
          </div>
        )}

        {loading && (
          <div className="flex items-center justify-center py-4">
            <Loader2 className="h-5 w-5 animate-spin text-primary mr-2" />
            <span className="text-sm">Analyzing ingredients...</span>
          </div>
        )}

        {result && (
          <div className="space-y-3">
            {/* Risk Badge */}
            <div className="flex items-center gap-2">
              <span className="text-sm font-medium">Risk Level:</span>
              <Badge className={getRiskColor(result.overallRisk)}>
                {result.overallRisk === 'none' ? (
                  <><CheckCircle className="h-3 w-3 mr-1" /> None Detected</>
                ) : (
                  <><AlertTriangle className="h-3 w-3 mr-1" /> {result.overallRisk.charAt(0).toUpperCase() + result.overallRisk.slice(1)}</>
                )}
              </Badge>
            </div>

            {/* Compounds Found */}
            {result.compounds.length > 0 && (
              <div className="space-y-2">
                <span className="text-sm font-medium">Detected Compounds:</span>
                <div className="flex flex-wrap gap-1">
                  {result.compounds.map((compound, idx) => (
                    <Badge 
                      key={idx} 
                      variant="outline" 
                      className={`text-xs ${getPotencyColor(compound.potency)}`}
                    >
                      {compound.source} ({compound.potency})
                    </Badge>
                  ))}
                </div>
              </div>
            )}

            {/* Summary */}
            <div className="bg-muted/50 rounded-lg p-3">
              <p className="text-sm">{result.summary}</p>
              {result.overallRisk !== 'none' && (
                <p className="text-xs text-muted-foreground mt-2 italic">
                  {result.recommendation}
                </p>
              )}
            </div>

            {/* Track Exposure Button - only show if compounds detected */}
            {result.compounds.length > 0 && (
              <div className="pt-1">
                {user ? (
                  tracked ? (
                    <div className="flex items-center justify-center gap-2 py-2 text-sm text-primary bg-primary/10 rounded-lg">
                      <Check className="h-4 w-4" />
                      <span>Added to your daily tracking</span>
                    </div>
                  ) : (
                    <Button 
                      onClick={handleTrackExposure}
                      disabled={isLogging}
                      size="sm"
                      variant="secondary"
                      className="w-full"
                    >
                      {isLogging ? (
                        <>
                          <Loader2 className="h-4 w-4 mr-1 animate-spin" />
                          Tracking...
                        </>
                      ) : (
                        <>
                          <Plus className="h-4 w-4 mr-1" />
                          Track This Exposure
                        </>
                      )}
                    </Button>
                  )
                ) : (
                  <div className="text-center">
                    <p className="text-xs text-muted-foreground mb-2">
                      Sign in to track your phytoestrogen exposure
                    </p>
                    <Link to="/auth">
                      <Button variant="outline" size="sm">Sign In</Button>
                    </Link>
                  </div>
                )}
              </div>
            )}

            <Button 
              variant="ghost" 
              size="sm" 
              onClick={() => { reset(); setTracked(false); }}
              className="w-full"
            >
              Analyze Again
            </Button>
          </div>
        )}
      </CardContent>
    </Card>
  );
};
