import { useState } from "react";
import { useNavigate } from "react-router-dom";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Leaf, Shield, Sparkles, ChefHat, AlertTriangle, Loader2, BookOpen, Save } from "lucide-react";
import { usePhytoestrogenFreeRecipes } from "@/hooks/usePhytoestrogenFreeRecipes";
import { useSavedRecipes } from "@/hooks/useSavedRecipes";
import { useAuth } from "@/hooks/useAuth";
import { toast } from "sonner";

const AVOIDED_INGREDIENTS = [
  "Soy products",
  "Legumes/Beans",
  "Flaxseeds",
  "Sesame/Tahini",
  "Oats",
  "Beer/Hops",
];

const SAFE_INGREDIENTS = [
  "Grass-fed meats",
  "Wild fish",
  "Pasture-raised eggs",
  "Cruciferous vegetables",
  "Avocado & olive oil",
  "Macadamia nuts",
];

export function PhytoestrogenFreeRecipes() {
  const navigate = useNavigate();
  const { user } = useAuth();
  const { saveRecipe } = useSavedRecipes();
  
  const [mealType, setMealType] = useState("daily");
  const [servings, setServings] = useState("2");
  const [preferences, setPreferences] = useState("");
  const [saveDialogOpen, setSaveDialogOpen] = useState(false);
  const [recipeTitle, setRecipeTitle] = useState("");
  const [isSaving, setIsSaving] = useState(false);
  
  const { content, isGenerating, error, generateRecipes, reset } = usePhytoestrogenFreeRecipes();

  const handleGenerate = () => {
    generateRecipes({
      mealType,
      servings: parseInt(servings) || 2,
      preferences: preferences.trim() || undefined,
    });
  };

  const handleOpenSaveDialog = () => {
    if (!user) {
      toast.error("Please sign in to save recipes to your cookbook");
      return;
    }
    // Generate a default title
    const defaultTitle = `Phytoestrogen-Free ${mealType.charAt(0).toUpperCase() + mealType.slice(1)} Plan`;
    setRecipeTitle(defaultTitle);
    setSaveDialogOpen(true);
  };

  const handleSaveRecipe = async () => {
    if (!recipeTitle.trim()) {
      toast.error("Please enter a recipe title");
      return;
    }

    setIsSaving(true);
    const success = await saveRecipe(
      recipeTitle.trim(),
      content,
      mealType,
      parseInt(servings) || 2
    );
    setIsSaving(false);

    if (success) {
      setSaveDialogOpen(false);
      setRecipeTitle("");
      toast.success("Recipe saved to your cookbook!");
    }
  };

  return (
    <div className="space-y-6">
      {/* Header Card */}
      <Card className="border-category-therapeutic/30 bg-gradient-to-br from-background to-category-therapeutic/5">
        <CardHeader>
          <div className="flex items-center gap-3">
            <div className="p-2 rounded-full bg-category-therapeutic/10">
              <Shield className="h-6 w-6 text-category-therapeutic" />
            </div>
            <div>
              <CardTitle className="text-xl flex items-center gap-2">
                Phytoestrogen-Free Recipe Generator
                <Badge variant="outline" className="bg-category-safe/10 text-category-safe border-category-safe/20">
                  Hormone-Safe
                </Badge>
              </CardTitle>
              <CardDescription>
                Generate delicious recipes using only non-estrogenic ingredients
              </CardDescription>
            </div>
          </div>
        </CardHeader>
        <CardContent className="space-y-4">
          {/* Info Badges */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <h4 className="text-sm font-medium text-destructive flex items-center gap-1">
                <AlertTriangle className="h-4 w-4" />
                Strictly Avoided:
              </h4>
              <div className="flex flex-wrap gap-1">
                {AVOIDED_INGREDIENTS.map((item) => (
                  <Badge key={item} variant="outline" className="bg-destructive/10 text-destructive border-destructive/20 text-xs">
                    ❌ {item}
                  </Badge>
                ))}
              </div>
            </div>
            <div className="space-y-2">
              <h4 className="text-sm font-medium text-category-safe flex items-center gap-1">
                <Sparkles className="h-4 w-4" />
                Safe Ingredients:
              </h4>
              <div className="flex flex-wrap gap-1">
                {SAFE_INGREDIENTS.map((item) => (
                  <Badge key={item} variant="outline" className="bg-category-safe/10 text-category-safe border-category-safe/20 text-xs">
                    ✓ {item}
                  </Badge>
                ))}
              </div>
            </div>
          </div>

          <Separator />

          {/* Form */}
          <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
            <div className="space-y-2">
              <Label htmlFor="mealType">Meal Plan Type</Label>
              <Select value={mealType} onValueChange={setMealType}>
                <SelectTrigger id="mealType">
                  <SelectValue placeholder="Select type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="daily">Full Day</SelectItem>
                  <SelectItem value="breakfast">Breakfast Only</SelectItem>
                  <SelectItem value="lunch">Lunch Only</SelectItem>
                  <SelectItem value="dinner">Dinner Only</SelectItem>
                  <SelectItem value="snacks">Snacks Only</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="servings">Servings</Label>
              <Input
                id="servings"
                type="number"
                min="1"
                max="10"
                value={servings}
                onChange={(e) => setServings(e.target.value)}
                placeholder="2"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="preferences">Preferences (optional)</Label>
              <Input
                id="preferences"
                value={preferences}
                onChange={(e) => setPreferences(e.target.value)}
                placeholder="e.g., high protein, quick meals"
              />
            </div>
          </div>

          <div className="flex gap-2">
            <Button 
              onClick={handleGenerate} 
              disabled={isGenerating}
              className="bg-category-therapeutic hover:bg-category-therapeutic/90"
            >
              {isGenerating ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Generating...
                </>
              ) : (
                <>
                  <ChefHat className="h-4 w-4 mr-2" />
                  Generate Hormone-Safe Recipes
                </>
              )}
            </Button>
            {content && (
              <Button variant="outline" onClick={reset}>
                Clear
              </Button>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Recipe Output */}
      {(content || isGenerating) && (
        <Card>
          <CardHeader className="pb-2">
            <div className="flex items-center justify-between">
              <CardTitle className="text-lg flex items-center gap-2">
                <Leaf className="h-5 w-5 text-category-therapeutic" />
                Your Phytoestrogen-Free Meal Plan
                {isGenerating && (
                  <Badge variant="secondary" className="animate-pulse">
                    Generating...
                  </Badge>
                )}
              </CardTitle>
              {content && !isGenerating && (
                <div className="flex gap-2">
                  <Button 
                    variant="outline" 
                    size="sm"
                    onClick={handleOpenSaveDialog}
                    className="gap-1.5"
                  >
                    <Save className="h-4 w-4" />
                    Save to Cookbook
                  </Button>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => navigate("/recipes")}
                    className="gap-1.5"
                  >
                    <BookOpen className="h-4 w-4" />
                    View Cookbook
                  </Button>
                </div>
              )}
            </div>
          </CardHeader>
          <CardContent>
            <ScrollArea className="h-[500px] rounded-md border p-4">
              <div className="prose prose-sm dark:prose-invert max-w-none whitespace-pre-wrap">
                {content || (
                  <div className="flex items-center gap-2 text-muted-foreground">
                    <Loader2 className="h-4 w-4 animate-spin" />
                    Creating your hormone-safe recipes...
                  </div>
                )}
              </div>
            </ScrollArea>
          </CardContent>
        </Card>
      )}

      {error && !content && (
        <Card className="border-destructive/50">
          <CardContent className="pt-6">
            <p className="text-destructive text-center">{error}</p>
          </CardContent>
        </Card>
      )}

      {/* Save Dialog */}
      <Dialog open={saveDialogOpen} onOpenChange={setSaveDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <BookOpen className="h-5 w-5 text-category-therapeutic" />
              Save to Cookbook
            </DialogTitle>
            <DialogDescription>
              Give your phytoestrogen-free meal plan a name to save it to your personal cookbook.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="recipeTitle">Recipe Title</Label>
              <Input
                id="recipeTitle"
                value={recipeTitle}
                onChange={(e) => setRecipeTitle(e.target.value)}
                placeholder="Enter a name for this recipe"
              />
            </div>
            <div className="flex gap-2 text-sm text-muted-foreground">
              <Badge variant="secondary">{mealType}</Badge>
              <Badge variant="secondary">{servings} servings</Badge>
              <Badge variant="outline" className="bg-category-therapeutic/10 text-category-therapeutic border-category-therapeutic/20">
                Phytoestrogen-Free
              </Badge>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setSaveDialogOpen(false)}>
              Cancel
            </Button>
            <Button 
              onClick={handleSaveRecipe} 
              disabled={isSaving || !recipeTitle.trim()}
              className="bg-category-therapeutic hover:bg-category-therapeutic/90"
            >
              {isSaving ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Saving...
                </>
              ) : (
                <>
                  <Save className="h-4 w-4 mr-2" />
                  Save Recipe
                </>
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
