import { useState, useMemo, useEffect } from "react";
import { format, subDays } from "date-fns";
import {
  Leaf,
  TrendingUp,
  TrendingDown,
  Calendar,
  AlertTriangle,
  Plus,
  Trash2,
  BarChart3,
  Settings,
  Target,
  Bell,
  BellOff,
  CheckCircle2,
} from "lucide-react";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Switch } from "@/components/ui/switch";
import { Slider } from "@/components/ui/slider";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  LineChart,
  Line,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  BarChart,
  Bar,
  Cell,
  ReferenceLine,
} from "recharts";
import { usePhytoestrogenTracking, DailyExposure } from "@/hooks/usePhytoestrogenTracking";
import { usePhytoestrogenGoals } from "@/hooks/usePhytoestrogenGoals";
import { phytoestrogensDatabase, PhytoestrogensData } from "@/data/chemicalDatabase";
import { useAuth } from "@/hooks/useAuth";
import { cn } from "@/lib/utils";
import { toast } from "sonner";

const potencyColors = {
  weak: "hsl(var(--chart-3))",
  moderate: "hsl(var(--chart-2))",
  strong: "hsl(var(--destructive))",
};

const potencyBadgeStyles = {
  weak: "bg-yellow-500/20 text-yellow-700 border-yellow-500/30",
  moderate: "bg-orange-500/20 text-orange-700 border-orange-500/30",
  strong: "bg-red-500/20 text-red-700 border-red-500/30",
};

export function PhytoestrogenTracker() {
  const { user } = useAuth();
  const {
    exposureLogs,
    isLoading,
    logExposure,
    isLogging,
    deleteExposure,
    isDeleting,
    getDailyExposures,
    getWeeklySummary,
    getTodaysExposure,
    getTopExposureSources,
  } = usePhytoestrogenTracking();

  const {
    goals,
    isLoading: isLoadingGoals,
    saveGoals,
    isSaving,
    checkProgress,
  } = usePhytoestrogenGoals();

  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [isGoalsDialogOpen, setIsGoalsDialogOpen] = useState(false);
  const [selectedPhytoestrogen, setSelectedPhytoestrogen] = useState<string>("");
  const [selectedFood, setSelectedFood] = useState<string>("");
  const [servings, setServings] = useState("1");
  const [notes, setNotes] = useState("");
  const [timeRange, setTimeRange] = useState<"7" | "14" | "30">("7");
  
  // Goals form state
  const [goalDailyLimit, setGoalDailyLimit] = useState(goals.daily_limit);
  const [goalWeeklyLimit, setGoalWeeklyLimit] = useState(goals.weekly_limit);
  const [goalAlertPercentage, setGoalAlertPercentage] = useState(goals.alert_at_percentage);
  const [goalShowAlerts, setGoalShowAlerts] = useState(goals.show_alerts);

  // Sync form state with goals
  useEffect(() => {
    setGoalDailyLimit(goals.daily_limit);
    setGoalWeeklyLimit(goals.weekly_limit);
    setGoalAlertPercentage(goals.alert_at_percentage);
    setGoalShowAlerts(goals.show_alerts);
  }, [goals]);

  const dailyExposures = useMemo(() => getDailyExposures(parseInt(timeRange)), [getDailyExposures, timeRange]);
  const weeklySummary = useMemo(() => getWeeklySummary(), [getWeeklySummary]);
  const todaysExposure = useMemo(() => getTodaysExposure(), [getTodaysExposure]);
  const topSources = useMemo(() => getTopExposureSources(5), [getTopExposureSources]);

  // Check progress against goals
  const progress = useMemo(() => 
    checkProgress(todaysExposure.total, weeklySummary.totalScore),
    [checkProgress, todaysExposure.total, weeklySummary.totalScore]
  );

  // Show alerts when limits are approached
  const [alertShown, setAlertShown] = useState(false);
  useEffect(() => {
    if (progress.shouldAlert && !alertShown && goals.show_alerts) {
      if (progress.daily.isExceeded) {
        toast.error("Daily phytoestrogen limit exceeded!", {
          description: `You've consumed ${progress.daily.current} of your ${progress.daily.limit} daily limit.`,
          duration: 5000,
        });
      } else if (progress.weekly.isExceeded) {
        toast.error("Weekly phytoestrogen limit exceeded!", {
          description: `You've consumed ${progress.weekly.current.toFixed(1)} of your ${progress.weekly.limit} weekly limit.`,
          duration: 5000,
        });
      } else if (progress.daily.isWarning || progress.weekly.isWarning) {
        toast.warning("Approaching phytoestrogen limit", {
          description: `Daily: ${progress.daily.percentage.toFixed(0)}% | Weekly: ${progress.weekly.percentage.toFixed(0)}%`,
          duration: 4000,
        });
      }
      setAlertShown(true);
    }
  }, [progress, alertShown, goals.show_alerts]);

  const selectedPhyto = useMemo(() => 
    phytoestrogensDatabase.find(p => p.id === selectedPhytoestrogen),
    [selectedPhytoestrogen]
  );

  const handleLogExposure = () => {
    if (!selectedPhyto || !selectedFood) return;

    logExposure({
      phytoestrogen: selectedPhyto,
      foodSource: selectedFood,
      servings: parseFloat(servings) || 1,
      notes: notes || undefined,
    });

    setIsDialogOpen(false);
    setSelectedPhytoestrogen("");
    setSelectedFood("");
    setServings("1");
    setNotes("");
    setAlertShown(false); // Reset alert to check new progress
  };

  const handleSaveGoals = () => {
    saveGoals({
      daily_limit: goalDailyLimit,
      weekly_limit: goalWeeklyLimit,
      alert_at_percentage: goalAlertPercentage,
      show_alerts: goalShowAlerts,
    });
    setIsGoalsDialogOpen(false);
    setAlertShown(false);
  };

  const chartData = useMemo(() => 
    dailyExposures.map(d => ({
      date: format(new Date(d.date), "MMM d"),
      total: d.total,
      strong: d.strong * 3,
      moderate: d.moderate * 2,
      weak: d.weak,
    })),
    [dailyExposures]
  );

  if (!user) {
    return (
      <Card className="border-dashed">
        <CardContent className="flex flex-col items-center justify-center py-8">
          <Leaf className="h-12 w-12 text-muted-foreground mb-4" />
          <p className="text-muted-foreground text-center">
            Sign in to track your phytoestrogen exposure
          </p>
        </CardContent>
      </Card>
    );
  }

  const trendPercentage = dailyExposures.length >= 2
    ? ((dailyExposures[dailyExposures.length - 1]?.total || 0) - 
       (dailyExposures[dailyExposures.length - 2]?.total || 0))
    : 0;

  return (
    <div className="space-y-4">
      {/* Progress Alert Banner */}
      {(progress.daily.isWarning || progress.weekly.isWarning) && goals.show_alerts && (
        <Card className={cn(
          "border-l-4",
          progress.daily.isExceeded || progress.weekly.isExceeded
            ? "border-l-destructive bg-destructive/5"
            : "border-l-category-warning bg-category-warning/5"
        )}>
          <CardContent className="p-4">
            <div className="flex items-start gap-3">
              <AlertTriangle className={cn(
                "h-5 w-5 mt-0.5",
                progress.daily.isExceeded || progress.weekly.isExceeded
                  ? "text-destructive"
                  : "text-category-warning"
              )} />
              <div className="flex-1">
                <p className="font-semibold text-sm">
                  {progress.daily.isExceeded || progress.weekly.isExceeded
                    ? "Exposure Limit Exceeded!"
                    : "Approaching Exposure Limit"}
                </p>
                <p className="text-xs text-muted-foreground mt-1">
                  Daily: {progress.daily.current}/{progress.daily.limit} ({progress.daily.percentage.toFixed(0)}%) • 
                  Weekly: {progress.weekly.current.toFixed(1)}/{progress.weekly.limit} ({progress.weekly.percentage.toFixed(0)}%)
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Header with Buttons */}
      <div className="flex items-center justify-between flex-wrap gap-2">
        <div>
          <h2 className="text-xl font-bold flex items-center gap-2">
            <Leaf className="h-5 w-5 text-destructive" />
            Phytoestrogen Tracker
          </h2>
          <p className="text-sm text-muted-foreground">
            Monitor your daily estrogenic compound intake
          </p>
        </div>
        <div className="flex gap-2">
          {/* Goals Dialog */}
          <Dialog open={isGoalsDialogOpen} onOpenChange={setIsGoalsDialogOpen}>
            <DialogTrigger asChild>
              <Button size="sm" variant="outline" className="gap-2">
                <Target className="h-4 w-4" />
                <span className="hidden sm:inline">Goals</span>
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-md">
              <DialogHeader>
                <DialogTitle className="flex items-center gap-2">
                  <Target className="h-5 w-5" />
                  Set Exposure Goals
                </DialogTitle>
              </DialogHeader>
              <div className="space-y-6 pt-4">
                <div className="space-y-3">
                  <Label>Daily Limit (score)</Label>
                  <div className="flex items-center gap-4">
                    <Slider
                      value={[goalDailyLimit]}
                      onValueChange={([v]) => setGoalDailyLimit(v)}
                      min={1}
                      max={50}
                      step={1}
                      className="flex-1"
                    />
                    <span className="text-lg font-bold w-12 text-right">{goalDailyLimit}</span>
                  </div>
                  <p className="text-xs text-muted-foreground">
                    Score = (Strong × 3) + (Moderate × 2) + (Weak × 1) per serving
                  </p>
                </div>

                <div className="space-y-3">
                  <Label>Weekly Limit (score)</Label>
                  <div className="flex items-center gap-4">
                    <Slider
                      value={[goalWeeklyLimit]}
                      onValueChange={([v]) => setGoalWeeklyLimit(v)}
                      min={5}
                      max={200}
                      step={5}
                      className="flex-1"
                    />
                    <span className="text-lg font-bold w-12 text-right">{goalWeeklyLimit}</span>
                  </div>
                </div>

                <div className="space-y-3">
                  <Label>Alert at {goalAlertPercentage}% of limit</Label>
                  <div className="flex items-center gap-4">
                    <Slider
                      value={[goalAlertPercentage]}
                      onValueChange={([v]) => setGoalAlertPercentage(v)}
                      min={50}
                      max={100}
                      step={5}
                      className="flex-1"
                    />
                    <span className="text-sm font-medium w-12 text-right">{goalAlertPercentage}%</span>
                  </div>
                </div>

                <div className="flex items-center justify-between">
                  <div className="space-y-1">
                    <Label>Show Alerts</Label>
                    <p className="text-xs text-muted-foreground">
                      Get notified when approaching limits
                    </p>
                  </div>
                  <Switch
                    checked={goalShowAlerts}
                    onCheckedChange={setGoalShowAlerts}
                  />
                </div>

                <Button 
                  className="w-full" 
                  onClick={handleSaveGoals}
                  disabled={isSaving}
                >
                  {isSaving ? "Saving..." : "Save Goals"}
                </Button>
              </div>
            </DialogContent>
          </Dialog>

          {/* Log Exposure Dialog */}
          <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
            <DialogTrigger asChild>
              <Button size="sm" className="gap-2">
                <Plus className="h-4 w-4" />
                Log Exposure
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-md">
              <DialogHeader>
                <DialogTitle>Log Phytoestrogen Exposure</DialogTitle>
              </DialogHeader>
              <div className="space-y-4 pt-4">
                <div className="space-y-2">
                  <Label>Phytoestrogen Compound</Label>
                  <Select value={selectedPhytoestrogen} onValueChange={setSelectedPhytoestrogen}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select compound..." />
                    </SelectTrigger>
                    <SelectContent className="max-h-[300px]">
                      {phytoestrogensDatabase.map(p => (
                        <SelectItem key={p.id} value={p.id}>
                          <div className="flex items-center gap-2">
                            <span>{p.name}</span>
                            <Badge variant="outline" className={cn("text-[10px]", potencyBadgeStyles[p.potency])}>
                              {p.potency}
                            </Badge>
                          </div>
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                {selectedPhyto && (
                  <div className="space-y-2">
                    <Label>Food Source</Label>
                    <Select value={selectedFood} onValueChange={setSelectedFood}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select food source..." />
                      </SelectTrigger>
                      <SelectContent>
                        {selectedPhyto.foodSources.map(food => (
                          <SelectItem key={food} value={food}>{food}</SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                )}

                <div className="space-y-2">
                  <Label>Servings</Label>
                  <Input
                    type="number"
                    min="0.5"
                    step="0.5"
                    value={servings}
                    onChange={(e) => setServings(e.target.value)}
                  />
                </div>

                <div className="space-y-2">
                  <Label>Notes (optional)</Label>
                  <Textarea
                    value={notes}
                    onChange={(e) => setNotes(e.target.value)}
                    placeholder="Any additional notes..."
                  />
                </div>

                <Button 
                  className="w-full" 
                  onClick={handleLogExposure}
                  disabled={!selectedPhytoestrogen || !selectedFood || isLogging}
                >
                  {isLogging ? "Logging..." : "Log Exposure"}
                </Button>
              </div>
            </DialogContent>
          </Dialog>
        </div>
      </div>

      {/* Goal Progress Cards */}
      <div className="grid grid-cols-2 gap-3">
        <Card className={cn(
          progress.daily.isExceeded && "border-destructive",
          progress.daily.isWarning && !progress.daily.isExceeded && "border-category-warning"
        )}>
          <CardContent className="p-4">
            <div className="flex items-center justify-between mb-2">
              <p className="text-xs text-muted-foreground">Daily Goal</p>
              {progress.daily.isExceeded ? (
                <Badge variant="destructive" className="text-[9px]">Exceeded</Badge>
              ) : progress.daily.isWarning ? (
                <Badge className="text-[9px] bg-category-warning">Warning</Badge>
              ) : (
                <CheckCircle2 className="h-4 w-4 text-category-safe" />
              )}
            </div>
            <div className="flex items-end gap-1">
              <span className="text-2xl font-bold">{progress.daily.current}</span>
              <span className="text-sm text-muted-foreground mb-0.5">/ {progress.daily.limit}</span>
            </div>
            <Progress 
              value={progress.daily.percentage} 
              className={cn(
                "mt-2 h-2",
                progress.daily.isExceeded && "[&>div]:bg-destructive",
                progress.daily.isWarning && !progress.daily.isExceeded && "[&>div]:bg-category-warning"
              )}
            />
          </CardContent>
        </Card>

        <Card className={cn(
          progress.weekly.isExceeded && "border-destructive",
          progress.weekly.isWarning && !progress.weekly.isExceeded && "border-category-warning"
        )}>
          <CardContent className="p-4">
            <div className="flex items-center justify-between mb-2">
              <p className="text-xs text-muted-foreground">Weekly Goal</p>
              {progress.weekly.isExceeded ? (
                <Badge variant="destructive" className="text-[9px]">Exceeded</Badge>
              ) : progress.weekly.isWarning ? (
                <Badge className="text-[9px] bg-category-warning">Warning</Badge>
              ) : (
                <CheckCircle2 className="h-4 w-4 text-category-safe" />
              )}
            </div>
            <div className="flex items-end gap-1">
              <span className="text-2xl font-bold">{progress.weekly.current.toFixed(1)}</span>
              <span className="text-sm text-muted-foreground mb-0.5">/ {progress.weekly.limit}</span>
            </div>
            <Progress 
              value={progress.weekly.percentage} 
              className={cn(
                "mt-2 h-2",
                progress.weekly.isExceeded && "[&>div]:bg-destructive",
                progress.weekly.isWarning && !progress.weekly.isExceeded && "[&>div]:bg-category-warning"
              )}
            />
          </CardContent>
        </Card>
      </div>

      {/* Additional Stats */}
      <div className="grid grid-cols-4 gap-2">
        <Card>
          <CardContent className="p-3">
            <p className="text-[10px] text-muted-foreground">Today</p>
            <p className="text-lg font-bold">{todaysExposure.entries.length}</p>
            <p className="text-[9px] text-muted-foreground">entries</p>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-3">
            <p className="text-[10px] text-muted-foreground">7-Day Avg</p>
            <p className="text-lg font-bold">{weeklySummary.averageDaily.toFixed(1)}</p>
            <p className="text-[9px] text-muted-foreground">per day</p>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-3">
            <p className="text-[10px] text-muted-foreground">Strong</p>
            <p className="text-lg font-bold text-destructive">{weeklySummary.strongCount}</p>
            <p className="text-[9px] text-muted-foreground">this week</p>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-3">
            <p className="text-[10px] text-muted-foreground">Total</p>
            <p className="text-lg font-bold">{exposureLogs?.length || 0}</p>
            <p className="text-[9px] text-muted-foreground">all time</p>
          </CardContent>
        </Card>
      </div>

      {/* Chart */}
      <Card>
        <CardHeader className="pb-2">
          <div className="flex items-center justify-between">
            <CardTitle className="text-base flex items-center gap-2">
              <BarChart3 className="h-4 w-4" />
              Exposure Trend
            </CardTitle>
            <Select value={timeRange} onValueChange={(v) => setTimeRange(v as "7" | "14" | "30")}>
              <SelectTrigger className="w-[100px] h-8">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="7">7 days</SelectItem>
                <SelectItem value="14">14 days</SelectItem>
                <SelectItem value="30">30 days</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardHeader>
        <CardContent>
          <div className="h-[200px]">
            <ResponsiveContainer width="100%" height="100%">
              <BarChart data={chartData}>
                <CartesianGrid strokeDasharray="3 3" className="stroke-muted" />
                <XAxis 
                  dataKey="date" 
                  tick={{ fontSize: 10 }} 
                  tickLine={false}
                  axisLine={false}
                />
                <YAxis 
                  tick={{ fontSize: 10 }} 
                  tickLine={false}
                  axisLine={false}
                />
                <Tooltip 
                  contentStyle={{ 
                    backgroundColor: "hsl(var(--card))",
                    border: "1px solid hsl(var(--border))",
                    borderRadius: "8px",
                    fontSize: "12px"
                  }}
                />
                <ReferenceLine 
                  y={goals.daily_limit} 
                  stroke="hsl(var(--destructive))" 
                  strokeDasharray="5 5"
                  label={{ value: 'Limit', position: 'right', fontSize: 10, fill: 'hsl(var(--destructive))' }}
                />
                <Bar dataKey="total" fill="hsl(var(--destructive))" radius={[4, 4, 0, 0]} />
              </BarChart>
            </ResponsiveContainer>
          </div>
        </CardContent>
      </Card>

      {/* Top Sources & Recent Entries */}
      <div className="grid md:grid-cols-2 gap-4">
        {/* Top Sources */}
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-base">Top Exposure Sources</CardTitle>
            <CardDescription className="text-xs">Most frequently logged compounds</CardDescription>
          </CardHeader>
          <CardContent>
            {topSources.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-4">
                No exposure logged yet
              </p>
            ) : (
              <div className="space-y-3">
                {topSources.map((source, i) => (
                  <div key={source.name} className="flex items-center gap-3">
                    <span className="text-xs font-medium text-muted-foreground w-4">{i + 1}</span>
                    <div className="flex-1">
                      <p className="text-sm font-medium">{source.name}</p>
                      <p className="text-xs text-muted-foreground">
                        {source.count} servings · Score: {source.score}
                      </p>
                    </div>
                    <Progress value={(source.score / topSources[0].score) * 100} className="w-16 h-2" />
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Recent Entries */}
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-base">Recent Entries</CardTitle>
            <CardDescription className="text-xs">Latest logged exposures</CardDescription>
          </CardHeader>
          <CardContent>
            <ScrollArea className="h-[200px]">
              {!exposureLogs || exposureLogs.length === 0 ? (
                <p className="text-sm text-muted-foreground text-center py-4">
                  No entries yet
                </p>
              ) : (
                <div className="space-y-2">
                  {exposureLogs.slice(0, 10).map((entry) => (
                    <div 
                      key={entry.id} 
                      className="flex items-center justify-between p-2 rounded-lg bg-muted/50 group"
                    >
                      <div className="flex-1 min-w-0">
                        <div className="flex items-center gap-2">
                          <p className="text-sm font-medium truncate">{entry.phytoestrogen_name}</p>
                          <Badge variant="outline" className={cn("text-[9px]", potencyBadgeStyles[entry.potency])}>
                            {entry.potency}
                          </Badge>
                        </div>
                        <p className="text-xs text-muted-foreground truncate">
                          {entry.food_source} · {entry.servings} serving{entry.servings !== 1 ? "s" : ""}
                        </p>
                        <p className="text-[10px] text-muted-foreground">
                          {format(new Date(entry.logged_at), "MMM d, h:mm a")}
                        </p>
                      </div>
                      <Button
                        variant="ghost"
                        size="icon"
                        className="h-7 w-7 opacity-0 group-hover:opacity-100 transition-opacity"
                        onClick={() => deleteExposure(entry.id)}
                        disabled={isDeleting}
                      >
                        <Trash2 className="h-3.5 w-3.5 text-destructive" />
                      </Button>
                    </div>
                  ))}
                </div>
              )}
            </ScrollArea>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
