import { useState, useRef, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Label } from '@/components/ui/label';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Camera, Loader2, Save, User } from 'lucide-react';
import { userApi, API_BASE } from '@/lib/apiServices';
import { useAuth } from '@/hooks/useAuth';
import { useToast } from '@/hooks/use-toast';

interface ProfileData {
  display_name: string | null;
  bio: string | null;
  avatar_url: string | null;
}

export const ProfileCustomization = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const fileInputRef = useRef<HTMLInputElement>(null);
  
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [uploading, setUploading] = useState(false);
  const [profile, setProfile] = useState<ProfileData>({
    display_name: '',
    bio: '',
    avatar_url: null,
  });

  useEffect(() => {
    if (user) fetchProfile();
  }, [user?.id]);

  const fetchProfile = async () => {
    if (!user) return;
    try {
      const data = await userApi.getProfile();
      setProfile({
        display_name: data.display_name || '',
        bio: data.bio || '',
        avatar_url: data.avatar_url,
      });
    } catch {
      setProfile({
        display_name: user.email?.split('@')[0] || '',
        bio: '',
        avatar_url: null,
      });
    }
    setLoading(false);
  };

  const handleAvatarClick = () => {
    fileInputRef.current?.click();
  };

  const handleFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file || !user) return;

    // Validate file type
    if (!file.type.startsWith('image/')) {
      toast({
        title: 'Invalid file type',
        description: 'Please upload an image file',
        variant: 'destructive',
      });
      return;
    }

    // Validate file size (max 2MB)
    if (file.size > 2 * 1024 * 1024) {
      toast({
        title: 'File too large',
        description: 'Please upload an image under 2MB',
        variant: 'destructive',
      });
      return;
    }

    setUploading(true);
    const reader = new FileReader();
    reader.onload = async () => {
      try {
        const dataUrl = reader.result as string;
        const { avatar_url } = await userApi.uploadAvatar(dataUrl);
        setProfile(prev => ({ ...prev, avatar_url: `${avatar_url}?t=${Date.now()}` }));
        toast({
          title: 'Avatar updated!',
          description: 'Your profile picture has been changed',
        });
      } catch (err) {
        console.error('Error uploading avatar:', err);
        toast({
          title: 'Upload failed',
          description: 'Failed to upload avatar. Please try again.',
          variant: 'destructive',
        });
      } finally {
        setUploading(false);
      }
    };
    reader.readAsDataURL(file);
  };

  const handleSave = async () => {
    if (!user) return;
    setSaving(true);
    try {
      await userApi.updateProfile({
        display_name: profile.display_name,
        bio: profile.bio,
        avatar_url: profile.avatar_url,
      });
      toast({
        title: 'Profile saved!',
        description: 'Your profile has been updated',
      });
    } catch (error) {
      console.error('Error saving profile:', error);
      toast({
        title: 'Save failed',
        description: 'Failed to save profile. Please try again.',
        variant: 'destructive',
      });
    } finally {
      setSaving(false);
    }
  };

  if (loading) {
    return (
      <Card>
        <CardContent className="p-8 flex justify-center">
          <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
        </CardContent>
      </Card>
    );
  }

  const initials = (profile.display_name || user?.email?.split('@')[0] || 'U')
    .substring(0, 2)
    .toUpperCase();

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <User className="h-5 w-5" />
          Profile Customization
        </CardTitle>
        <CardDescription>
          Personalize your profile with a photo and bio
        </CardDescription>
      </CardHeader>
      <CardContent className="space-y-6">
        {/* Avatar Upload */}
        <div className="flex items-center gap-6">
          <div className="relative">
            <Avatar className="h-20 w-20 cursor-pointer" onClick={handleAvatarClick}>
              <AvatarImage src={profile.avatar_url?.startsWith('/') ? `${API_BASE}${profile.avatar_url}` : (profile.avatar_url || undefined)} alt="Profile" />
              <AvatarFallback className="text-xl bg-primary/10 text-primary">
                {initials}
              </AvatarFallback>
            </Avatar>
            <button
              onClick={handleAvatarClick}
              disabled={uploading}
              className="absolute bottom-0 right-0 h-7 w-7 rounded-full bg-primary text-primary-foreground flex items-center justify-center shadow-md hover:bg-primary/90 transition-colors disabled:opacity-50"
            >
              {uploading ? (
                <Loader2 className="h-3.5 w-3.5 animate-spin" />
              ) : (
                <Camera className="h-3.5 w-3.5" />
              )}
            </button>
            <input
              ref={fileInputRef}
              type="file"
              accept="image/*"
              onChange={handleFileChange}
              className="hidden"
            />
          </div>
          
          <div className="flex-1">
            <p className="font-medium">Profile Photo</p>
            <p className="text-sm text-muted-foreground">
              Click to upload a new photo (max 2MB)
            </p>
          </div>
        </div>

        {/* Display Name */}
        <div className="space-y-2">
          <Label htmlFor="displayName">Display Name</Label>
          <Input
            id="displayName"
            value={profile.display_name || ''}
            onChange={(e) => setProfile(prev => ({ ...prev, display_name: e.target.value }))}
            placeholder="Your display name"
            maxLength={50}
          />
          <p className="text-xs text-muted-foreground">
            This name will be shown on your posts and profile
          </p>
        </div>

        {/* Bio */}
        <div className="space-y-2">
          <Label htmlFor="bio">Bio</Label>
          <Textarea
            id="bio"
            value={profile.bio || ''}
            onChange={(e) => setProfile(prev => ({ ...prev, bio: e.target.value }))}
            placeholder="Tell the community about yourself..."
            rows={4}
            maxLength={300}
          />
          <p className="text-xs text-muted-foreground">
            {(profile.bio?.length || 0)}/300 characters
          </p>
        </div>

        {/* Save Button */}
        <Button onClick={handleSave} disabled={saving} className="w-full gap-2">
          {saving ? (
            <Loader2 className="h-4 w-4 animate-spin" />
          ) : (
            <Save className="h-4 w-4" />
          )}
          Save Profile
        </Button>
      </CardContent>
    </Card>
  );
};
