import { CheckCircle2 } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { useShoppingList } from '@/contexts/ShoppingListContext';
import { cn } from '@/lib/utils';

interface PurchasedButtonProps {
  itemId: string;
  itemType: 'food' | 'cosmetic';
  className?: string;
  size?: 'sm' | 'default';
}

export const PurchasedButton = ({ itemId, itemType, className, size = 'default' }: PurchasedButtonProps) => {
  const { items, togglePurchased, isInList } = useShoppingList();
  
  // Find the shopping list item for this product
  const shoppingItem = items.find(i => i.item_id === itemId && i.item_type === itemType);
  const isPurchased = shoppingItem?.is_purchased ?? false;
  const inList = isInList(itemId, itemType);

  // Only show button if item is in the shopping list
  if (!inList) {
    return null;
  }

  const handleClick = (e: React.MouseEvent) => {
    e.stopPropagation();
    if (shoppingItem) {
      togglePurchased(shoppingItem.id, !isPurchased);
    }
  };

  return (
    <Button
      variant="ghost"
      size={size === 'sm' ? 'icon' : 'default'}
      className={cn(
        'transition-colors',
        size === 'sm' ? 'h-8 w-8' : 'h-9 w-9',
        className
      )}
      onClick={handleClick}
      title={isPurchased ? 'Mark as not purchased' : 'Mark as purchased'}
    >
      <CheckCircle2 
        className={cn(
          'transition-colors',
          size === 'sm' ? 'h-4 w-4' : 'h-5 w-5',
          isPurchased ? 'fill-category-safe text-category-safe' : 'text-muted-foreground hover:text-category-safe'
        )}
      />
    </Button>
  );
};