import { useState, useMemo } from "react";
import { Book, Trash2, ChevronDown, ChevronUp, Clock, Users, ShoppingCart, Search, X, Filter, CalendarDays } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useSavedRecipes } from "@/hooks/useSavedRecipes";
import { useAuth } from "@/hooks/useAuth";
import { GroceryListGenerator } from "@/components/GroceryListGenerator";
import { RecipeShareDialog } from "@/components/RecipeShareDialog";
import { RecipeNutritionDisplay } from "@/components/RecipeNutritionDisplay";
import { MealPlannerCalendar } from "@/components/MealPlannerCalendar";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog";
import { Badge } from "@/components/ui/badge";

const mealTypeLabels: Record<string, string> = {
  daily: "Full Day",
  breakfast: "Breakfast",
  lunch: "Lunch",
  dinner: "Dinner",
  snacks: "Snacks",
  weekly: "Weekly",
};

const mealTypeOptions = [
  { value: "all", label: "All Meal Types" },
  { value: "daily", label: "Full Day" },
  { value: "breakfast", label: "Breakfast" },
  { value: "lunch", label: "Lunch" },
  { value: "dinner", label: "Dinner" },
  { value: "snacks", label: "Snacks" },
  { value: "weekly", label: "Weekly" },
];

export function RecipeCookbook() {
  const { user } = useAuth();
  const { recipes, loading, deleteRecipe, generateShareLink, disableSharing, analyzeNutrition } = useSavedRecipes();
  const [expandedId, setExpandedId] = useState<string | null>(null);
  const [selectedRecipeIds, setSelectedRecipeIds] = useState<string[]>([]);
  const [activeTab, setActiveTab] = useState("recipes");
  const [searchQuery, setSearchQuery] = useState("");
  const [mealTypeFilter, setMealTypeFilter] = useState("all");

  const filteredRecipes = useMemo(() => {
    return recipes.filter(recipe => {
      const matchesSearch = searchQuery === "" || 
        recipe.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
        recipe.content.toLowerCase().includes(searchQuery.toLowerCase());
      
      const matchesMealType = mealTypeFilter === "all" || recipe.meal_type === mealTypeFilter;
      
      return matchesSearch && matchesMealType;
    });
  }, [recipes, searchQuery, mealTypeFilter]);

  const hasActiveFilters = searchQuery !== "" || mealTypeFilter !== "all";

  const clearFilters = () => {
    setSearchQuery("");
    setMealTypeFilter("all");
  };

  if (!user) {
    return (
      <Card className="bg-muted/30">
        <CardContent className="p-4 sm:p-6 text-center">
          <Book className="h-10 w-10 mx-auto text-muted-foreground mb-3" />
          <h3 className="font-semibold text-sm sm:text-base mb-2">Your Cookbook</h3>
          <p className="text-xs sm:text-sm text-muted-foreground">
            Sign in to save recipes and build your personal cookbook
          </p>
        </CardContent>
      </Card>
    );
  }

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6 text-center">
          <div className="animate-pulse space-y-3">
            <div className="h-4 bg-muted rounded w-1/2 mx-auto" />
            <div className="h-20 bg-muted rounded" />
          </div>
        </CardContent>
      </Card>
    );
  }

  if (recipes.length === 0) {
    return (
      <Card className="bg-muted/30">
        <CardContent className="p-4 sm:p-6 text-center">
          <Book className="h-10 w-10 mx-auto text-muted-foreground mb-3" />
          <h3 className="font-semibold text-sm sm:text-base mb-2">Your Cookbook is Empty</h3>
          <p className="text-xs sm:text-sm text-muted-foreground">
            Generate a meal plan above and save it to start building your cookbook
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader className="p-4 sm:p-6 pb-2">
        <CardTitle className="text-base sm:text-lg flex items-center gap-2">
          <Book className="h-5 w-5 text-primary" />
          Your Cookbook
          <Badge variant="secondary" className="ml-auto">
            {recipes.length} {recipes.length === 1 ? 'recipe' : 'recipes'}
          </Badge>
        </CardTitle>
      </CardHeader>
      <CardContent className="p-4 sm:p-6 pt-2">
        <Tabs value={activeTab} onValueChange={setActiveTab}>
          <TabsList className="grid w-full grid-cols-3 mb-4">
            <TabsTrigger value="recipes" className="gap-1.5 text-xs sm:text-sm">
              <Book className="h-3.5 w-3.5" />
              <span className="hidden sm:inline">Recipes</span>
            </TabsTrigger>
            <TabsTrigger value="planner" className="gap-1.5 text-xs sm:text-sm">
              <CalendarDays className="h-3.5 w-3.5" />
              <span className="hidden sm:inline">Planner</span>
            </TabsTrigger>
            <TabsTrigger value="grocery" className="gap-1.5 text-xs sm:text-sm">
              <ShoppingCart className="h-3.5 w-3.5" />
              <span className="hidden sm:inline">Grocery</span>
            </TabsTrigger>
          </TabsList>

          <TabsContent value="recipes" className="mt-0 space-y-4">
            {/* Search and Filter */}
            <div className="flex flex-col sm:flex-row gap-2">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Search recipes or ingredients..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-9 pr-9 text-sm"
                />
                {searchQuery && (
                  <button
                    onClick={() => setSearchQuery("")}
                    className="absolute right-3 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground"
                  >
                    <X className="h-4 w-4" />
                  </button>
                )}
              </div>
              <Select value={mealTypeFilter} onValueChange={setMealTypeFilter}>
                <SelectTrigger className="w-full sm:w-[160px] text-sm">
                  <Filter className="h-3.5 w-3.5 mr-2 text-muted-foreground" />
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {mealTypeOptions.map(option => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {/* Active Filters Info */}
            {hasActiveFilters && (
              <div className="flex items-center justify-between text-xs text-muted-foreground">
                <span>
                  Showing {filteredRecipes.length} of {recipes.length} recipes
                </span>
                <Button variant="ghost" size="sm" onClick={clearFilters} className="h-6 text-xs">
                  Clear filters
                </Button>
              </div>
            )}

            {/* Recipe List */}
            {filteredRecipes.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                <Search className="h-8 w-8 mx-auto mb-2 opacity-50" />
                <p className="text-sm">No recipes match your search</p>
                {hasActiveFilters && (
                  <Button variant="link" size="sm" onClick={clearFilters} className="mt-1">
                    Clear filters
                  </Button>
                )}
              </div>
            ) : (
              <div className="space-y-3">
                {filteredRecipes.map((recipe) => (
                  <div
                    key={recipe.id}
                    className="border rounded-lg overflow-hidden bg-card"
                  >
                    <div
                      className="p-3 sm:p-4 flex items-center justify-between cursor-pointer hover:bg-muted/50 transition-colors"
                      onClick={() => setExpandedId(expandedId === recipe.id ? null : recipe.id)}
                    >
                      <div className="flex-1 min-w-0">
                        <h4 className="font-medium text-sm sm:text-base truncate">
                          {recipe.title}
                        </h4>
                        <div className="flex items-center gap-3 mt-1 text-xs text-muted-foreground">
                          <span className="flex items-center gap-1">
                            <Clock className="h-3 w-3" />
                            {mealTypeLabels[recipe.meal_type] || recipe.meal_type}
                          </span>
                          <span className="flex items-center gap-1">
                            <Users className="h-3 w-3" />
                            {recipe.servings} servings
                          </span>
                        </div>
                      </div>
                      <div className="flex items-center gap-1 ml-2">
                        <RecipeShareDialog
                          recipeId={recipe.id}
                          recipeTitle={recipe.title}
                          shareToken={recipe.share_token}
                          isShared={recipe.is_shared}
                          onGenerateLink={generateShareLink}
                          onDisableSharing={disableSharing}
                        />
                        <AlertDialog>
                          <AlertDialogTrigger asChild>
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-8 w-8 text-muted-foreground hover:text-destructive"
                              onClick={(e) => e.stopPropagation()}
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </AlertDialogTrigger>
                          <AlertDialogContent>
                            <AlertDialogHeader>
                              <AlertDialogTitle>Delete Recipe?</AlertDialogTitle>
                              <AlertDialogDescription>
                                This will permanently remove "{recipe.title}" from your cookbook.
                              </AlertDialogDescription>
                            </AlertDialogHeader>
                            <AlertDialogFooter>
                              <AlertDialogCancel>Cancel</AlertDialogCancel>
                              <AlertDialogAction
                                onClick={() => deleteRecipe(recipe.id)}
                                className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
                              >
                                Delete
                              </AlertDialogAction>
                            </AlertDialogFooter>
                          </AlertDialogContent>
                        </AlertDialog>
                        {expandedId === recipe.id ? (
                          <ChevronUp className="h-4 w-4 text-muted-foreground" />
                        ) : (
                          <ChevronDown className="h-4 w-4 text-muted-foreground" />
                        )}
                      </div>
                    </div>
                    
                    {expandedId === recipe.id && (
                      <div className="border-t bg-muted/20">
                        <div className="p-4 space-y-4">
                          {/* Nutrition Display */}
                          <RecipeNutritionDisplay
                            nutritionalInfo={recipe.nutritional_info}
                            onAnalyze={() => analyzeNutrition(recipe.id)}
                          />
                          
                          {/* Recipe Content */}
                          <ScrollArea className="h-[250px] sm:h-[350px]">
                            <div className="prose prose-sm dark:prose-invert max-w-none pr-4">
                              <div className="whitespace-pre-wrap text-sm leading-relaxed">
                                {recipe.content}
                              </div>
                            </div>
                          </ScrollArea>
                        </div>
                      </div>
                    )}
                  </div>
                ))}
              </div>
            )}
          </TabsContent>

          <TabsContent value="planner" className="mt-0">
            <MealPlannerCalendar />
          </TabsContent>

          <TabsContent value="grocery" className="mt-0">
            <GroceryListGenerator
              recipes={recipes}
              selectedRecipeIds={selectedRecipeIds}
              onSelectionChange={setSelectedRecipeIds}
            />
          </TabsContent>
        </Tabs>
      </CardContent>
    </Card>
  );
}
