import { useState } from "react";
import { Leaf, Flame, Beef, Wheat, Droplets, Loader2, Sparkles } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import type { NutritionalInfo } from "@/hooks/useSavedRecipes";

interface RecipeNutritionDisplayProps {
  nutritionalInfo: NutritionalInfo | null;
  onAnalyze: () => Promise<NutritionalInfo | null>;
}

export function RecipeNutritionDisplay({ 
  nutritionalInfo, 
  onAnalyze 
}: RecipeNutritionDisplayProps) {
  const [isAnalyzing, setIsAnalyzing] = useState(false);

  const handleAnalyze = async () => {
    setIsAnalyzing(true);
    await onAnalyze();
    setIsAnalyzing(false);
  };

  if (!nutritionalInfo) {
    return (
      <div className="p-3 bg-muted/30 rounded-lg border border-dashed">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2 text-sm text-muted-foreground">
            <Leaf className="h-4 w-4" />
            <span>Nutritional info not analyzed</span>
          </div>
          <Button
            variant="outline"
            size="sm"
            onClick={handleAnalyze}
            disabled={isAnalyzing}
            className="gap-1.5 text-xs"
          >
            {isAnalyzing ? (
              <>
                <Loader2 className="h-3.5 w-3.5 animate-spin" />
                Analyzing...
              </>
            ) : (
              <>
                <Sparkles className="h-3.5 w-3.5" />
                Analyze
              </>
            )}
          </Button>
        </div>
      </div>
    );
  }

  return (
    <div className="p-3 bg-gradient-to-br from-category-therapeutic/10 to-category-safe/10 rounded-lg border border-category-therapeutic/20">
      <div className="flex items-center gap-2 mb-3">
        <Leaf className="h-4 w-4 text-category-therapeutic" />
        <span className="text-sm font-medium">Nutrition per serving</span>
      </div>
      
      <div className="grid grid-cols-4 gap-2 mb-3">
        <div className="text-center p-2 bg-background/50 rounded">
          <Flame className="h-4 w-4 mx-auto text-orange-500 mb-0.5" />
          <p className="text-sm font-bold">{nutritionalInfo.calories}</p>
          <p className="text-[10px] text-muted-foreground">kcal</p>
        </div>
        <div className="text-center p-2 bg-background/50 rounded">
          <Beef className="h-4 w-4 mx-auto text-red-500 mb-0.5" />
          <p className="text-sm font-bold">{nutritionalInfo.protein}g</p>
          <p className="text-[10px] text-muted-foreground">protein</p>
        </div>
        <div className="text-center p-2 bg-background/50 rounded">
          <Wheat className="h-4 w-4 mx-auto text-amber-500 mb-0.5" />
          <p className="text-sm font-bold">{nutritionalInfo.carbs}g</p>
          <p className="text-[10px] text-muted-foreground">carbs</p>
        </div>
        <div className="text-center p-2 bg-background/50 rounded">
          <Droplets className="h-4 w-4 mx-auto text-yellow-500 mb-0.5" />
          <p className="text-sm font-bold">{nutritionalInfo.fat}g</p>
          <p className="text-[10px] text-muted-foreground">fat</p>
        </div>
      </div>

      {nutritionalInfo.highlights.length > 0 && (
        <div className="flex flex-wrap gap-1.5">
          {nutritionalInfo.highlights.slice(0, 3).map((highlight, i) => (
            <Badge key={i} variant="outline" className="text-[10px] bg-category-therapeutic/10 py-0">
              ✓ {highlight}
            </Badge>
          ))}
        </div>
      )}
    </div>
  );
}
