import { useState } from "react";
import { Share2, Link2, Check, Loader2, XCircle } from "lucide-react";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { toast } from "sonner";

interface RecipeShareDialogProps {
  recipeId: string;
  recipeTitle: string;
  shareToken: string | null;
  isShared: boolean;
  onGenerateLink: (id: string) => Promise<string | null>;
  onDisableSharing: (id: string) => Promise<boolean>;
}

export function RecipeShareDialog({
  recipeId,
  recipeTitle,
  shareToken,
  isShared,
  onGenerateLink,
  onDisableSharing,
}: RecipeShareDialogProps) {
  const [isOpen, setIsOpen] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [shareUrl, setShareUrl] = useState<string | null>(
    shareToken ? `${window.location.origin}/shared-recipe/${shareToken}` : null
  );
  const [copied, setCopied] = useState(false);

  const handleGenerateLink = async () => {
    setIsLoading(true);
    const url = await onGenerateLink(recipeId);
    if (url) {
      setShareUrl(url);
      toast.success("Share link generated!");
    }
    setIsLoading(false);
  };

  const handleCopyLink = async () => {
    if (shareUrl) {
      await navigator.clipboard.writeText(shareUrl);
      setCopied(true);
      toast.success("Link copied to clipboard!");
      setTimeout(() => setCopied(false), 2000);
    }
  };

  const handleDisableSharing = async () => {
    setIsLoading(true);
    const success = await onDisableSharing(recipeId);
    if (success) {
      setShareUrl(null);
      setIsOpen(false);
    }
    setIsLoading(false);
  };

  return (
    <Dialog open={isOpen} onOpenChange={setIsOpen}>
      <DialogTrigger asChild>
        <Button
          variant="ghost"
          size="icon"
          className="h-8 w-8 text-muted-foreground hover:text-primary"
          onClick={(e) => e.stopPropagation()}
        >
          <Share2 className="h-4 w-4" />
        </Button>
      </DialogTrigger>
      <DialogContent className="sm:max-w-md" onClick={(e) => e.stopPropagation()}>
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Share2 className="h-5 w-5 text-primary" />
            Share Recipe
          </DialogTitle>
          <DialogDescription>
            Share "{recipeTitle}" with anyone via a unique link
          </DialogDescription>
        </DialogHeader>
        <div className="space-y-4 pt-4">
          {!shareUrl ? (
            <div className="text-center py-4">
              <p className="text-sm text-muted-foreground mb-4">
                Generate a shareable link that anyone can use to view this recipe
              </p>
              <Button onClick={handleGenerateLink} disabled={isLoading} className="gap-2">
                {isLoading ? (
                  <Loader2 className="h-4 w-4 animate-spin" />
                ) : (
                  <Link2 className="h-4 w-4" />
                )}
                Generate Share Link
              </Button>
            </div>
          ) : (
            <>
              <div className="flex items-center gap-2">
                <Input
                  value={shareUrl}
                  readOnly
                  className="text-sm"
                />
                <Button
                  variant="outline"
                  size="icon"
                  onClick={handleCopyLink}
                  className="shrink-0"
                >
                  {copied ? (
                    <Check className="h-4 w-4 text-green-500" />
                  ) : (
                    <Link2 className="h-4 w-4" />
                  )}
                </Button>
              </div>
              <p className="text-xs text-muted-foreground">
                Anyone with this link can view your recipe
              </p>
              <div className="flex justify-between items-center pt-2 border-t">
                <span className="text-sm text-muted-foreground">
                  Sharing is enabled
                </span>
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={handleDisableSharing}
                  disabled={isLoading}
                  className="text-destructive hover:text-destructive gap-1.5"
                >
                  {isLoading ? (
                    <Loader2 className="h-3.5 w-3.5 animate-spin" />
                  ) : (
                    <XCircle className="h-3.5 w-3.5" />
                  )}
                  Disable Sharing
                </Button>
              </div>
            </>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
}
