import { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { ShoppingCart, Globe, Database, Loader2, Heart, Check, Info, Sparkles } from 'lucide-react';
import { OpenFoodFactsProduct, AIAnalysis } from '@/hooks/useBarcodeLookup';
import { PhytoestrogenAnalysisCard } from './PhytoestrogenAnalysisCard';
import { CosmeticDetailModal } from './CosmeticDetailModal';
import { AIAnalysisCard } from './AIAnalysisCard';
import { FavoriteButton } from './FavoriteButton';
import { ShoppingListButton } from './ShoppingListButton';
import { PurchasedButton } from './PurchasedButton';
import { useFavorites } from '@/contexts/FavoritesContext';
import { useShoppingList } from '@/contexts/ShoppingListContext';
import { cn } from '@/lib/utils';
import { estimateGlycogenicIndex, estimateLipogenicIndex, estimateTobinFromIndices, hasEnoughNutritionForIndices } from '@/utils/nutritionIndices';
import { getTobinRating, getIndexRating, getHalalStatusColor, getKosherStatusColor, getOrganicStatusColor } from '@/data/foodDatabase';
import foodPlaceholder from '@/assets/food-placeholder.png';
import cosmeticPlaceholder from '@/assets/cosmetic-placeholder.png';

interface LocalProduct {
  id: string;
  type: 'food' | 'cosmetic';
  name: string;
  barcode: string;
  details?: any;
}

interface ScannedProductCardProps {
  localProduct: LocalProduct | null;
  openFoodFactsProduct: OpenFoodFactsProduct | null;
  source: 'local' | 'openfoodfacts' | 'ai' | null;
  apiSourceString?: string | null;
  isLoading: boolean;
  user: any;
  onAddToList: () => void;
  currentBarcode?: string;
  isAiGenerated?: boolean;
  onSaveAiProduct?: () => void;
  aiAnalysis?: AIAnalysis | null;
  productType?: 'food' | 'cosmetic';
}

export const ScannedProductCard = ({
  localProduct,
  openFoodFactsProduct,
  source,
  apiSourceString,
  isLoading,
  user,
  onAddToList,
  currentBarcode,
  isAiGenerated,
  onSaveAiProduct,
  aiAnalysis,
  productType = 'food',
}: ScannedProductCardProps) => {
  const { isFavorite, toggleFavorite } = useFavorites();
  const { isInList, toggleItem } = useShoppingList();
  const [showCosmeticDetail, setShowCosmeticDetail] = useState(false);
  // For Open Food Facts products, use barcode as ID
  const getItemId = () => {
    if (localProduct) return localProduct.id;
    if (openFoodFactsProduct && currentBarcode) return currentBarcode;
    return '';
  };

  const getItemType = (): 'food' | 'cosmetic' => {
    if (localProduct) return localProduct.type;
    return 'food'; // Open Food Facts is always food
  };

  const itemId = getItemId();
  const itemType = getItemType();
  const isFav = itemId ? isFavorite(itemId, itemType) : false;
  const inShoppingList = itemId ? isInList(itemId, itemType) : false;

  const handleFavorite = (e: React.MouseEvent) => {
    e.stopPropagation();
    if (itemId) {
      toggleFavorite(itemId, itemType);
    }
  };

  const handleShoppingList = (e: React.MouseEvent) => {
    e.stopPropagation();
    if (itemId) {
      toggleItem(itemId, itemType);
    }
  };

  if (isLoading) {
    return (
      <Card className="border-primary">
        <CardContent className="flex items-center justify-center py-8">
          <Loader2 className="h-6 w-6 animate-spin text-primary mr-2" />
          <span>Looking up product...</span>
        </CardContent>
      </Card>
    );
  }

  // Local product display
  if (localProduct && source === 'local') {
    const isCosmetic = localProduct.type === 'cosmetic';
    
    return (
      <>
        <Card className="border-primary">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              Product Found
              <Badge variant="secondary" className="capitalize">
                {localProduct.type}
              </Badge>
              <Badge variant="outline" className="gap-1">
                <Database className="h-3 w-3" />
                Local
              </Badge>
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex items-center gap-4">
              {localProduct.details && (
                <span className="text-4xl">{localProduct.details.emoji}</span>
              )}
              <div className="flex-1">
                <p className="font-semibold text-lg">{localProduct.name}</p>
                {localProduct.details && (
                  <p className="text-sm text-muted-foreground">
                    {localProduct.type === 'food' 
                      ? localProduct.details.categoryName 
                      : localProduct.details.category?.replace('_', ' ')}
                  </p>
                )}
              </div>
              {isCosmetic && localProduct.details && (
                <Button 
                  variant="outline" 
                  size="sm"
                  onClick={() => setShowCosmeticDetail(true)}
                  className="gap-1"
                >
                  <Info className="h-4 w-4" />
                  View Details
                </Button>
              )}
            </div>

            {/* Quick cosmetic risk overview */}
            {isCosmetic && localProduct.details && (
              <div className="bg-muted/50 rounded-lg p-3 space-y-2">
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Risk Level</span>
                  <Badge className={cn(
                    localProduct.details.riskLevel === 1 ? "bg-red-500" :
                    localProduct.details.riskLevel === 2 ? "bg-orange-500" :
                    localProduct.details.riskLevel === 3 ? "bg-yellow-500" :
                    "bg-green-500"
                  )}>
                    {localProduct.details.riskLevel === 1 ? "High Risk" :
                     localProduct.details.riskLevel === 2 ? "Medium Risk" :
                     localProduct.details.riskLevel === 3 ? "Low Risk" :
                     "Minimal Risk"}
                  </Badge>
                </div>
                {localProduct.details.estrogenicProfile && (
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium">Estrogenic Potency</span>
                    <Badge variant="outline" className="capitalize">
                      {localProduct.details.estrogenicProfile.estrogenicPotency}
                    </Badge>
                  </div>
                )}
                {localProduct.details.tobinScore && (
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium">TOBIN Score</span>
                    <Badge variant="outline">
                      {localProduct.details.tobinScore.total}/100
                    </Badge>
                  </div>
                )}
                <Button 
                  variant="link" 
                  size="sm" 
                  className="p-0 h-auto text-primary"
                  onClick={() => setShowCosmeticDetail(true)}
                >
                  See full analysis →
                </Button>
              </div>
            )}
            
            {user ? (
              <div className="flex gap-2">
                <Button 
                  variant="outline" 
                  size="icon"
                  onClick={handleFavorite}
                  title={isFav ? 'Remove from favorites' : 'Add to favorites'}
                >
                  <Heart className={cn(
                    "h-4 w-4 transition-colors",
                    isFav ? "fill-destructive text-destructive" : "text-muted-foreground"
                  )} />
                </Button>
                <Button 
                  className="flex-1 gap-2" 
                  variant={inShoppingList ? "secondary" : "default"}
                  onClick={handleShoppingList}
                >
                  {inShoppingList ? (
                    <>
                      <Check className="h-4 w-4" />
                      In Shopping List
                    </>
                  ) : (
                    <>
                      <ShoppingCart className="h-4 w-4" />
                      Add to Shopping List
                    </>
                  )}
                </Button>
              </div>
            ) : (
              <div className="text-center">
                <p className="text-sm text-muted-foreground mb-2">
                  Sign in to save favorites and shopping list
                </p>
                <Link to="/auth">
                  <Button variant="outline">Sign In</Button>
                </Link>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Cosmetic Detail Modal */}
        {isCosmetic && localProduct.details && (
          <CosmeticDetailModal
            cosmetic={localProduct.details}
            open={showCosmeticDetail}
            onClose={() => setShowCosmeticDetail(false)}
          />
        )}
      </>
    );
  }

  // Open Food Facts or AI-identified product display (same layout as reference FoodCard)
  if (openFoodFactsProduct && (source === 'openfoodfacts' || source === 'ai')) {
    const isAi = source === 'ai';
    const isCosmetic = productType === 'cosmetic';
    const placeholderImage = isCosmetic ? cosmeticPlaceholder : foodPlaceholder;
    const nut = openFoodFactsProduct.nutrition;
    const hasNut = !isCosmetic && !!nut && hasEnoughNutritionForIndices(nut);
    const aiGi = aiAnalysis?.glycogenicIndex;
    const aiLi = aiAnalysis?.lipogenicIndex;
    const aiTobin = aiAnalysis?.tobinScore;
    const clamp = (n: number, min: number, max: number) => Math.max(min, Math.min(max, n));
    const giFromNut = nut ? estimateGlycogenicIndex(nut) : null;
    const liFromNut = nut ? estimateLipogenicIndex(nut) : null;
    const gi = (hasNut && giFromNut != null) ? giFromNut : (typeof aiGi === 'number' && Number.isFinite(aiGi) ? clamp(aiGi, 0, 3) : giFromNut);
    const li = (hasNut && liFromNut != null) ? liFromNut : (typeof aiLi === 'number' && Number.isFinite(aiLi) ? clamp(aiLi, 0, 3) : liFromNut);
    const tobin = (hasNut && (gi != null || li != null)) ? estimateTobinFromIndices(gi, li) : (typeof aiTobin === 'number' && Number.isFinite(aiTobin) ? clamp(aiTobin, 0, 50) : (!isCosmetic ? estimateTobinFromIndices(gi, li) : aiTobin ?? null));
    const tobinRating = tobin != null ? getTobinRating(tobin) : null;
    const nutriscoreGrade = openFoodFactsProduct.nutriscoreGrade ?? aiAnalysis?.nutriscoreGrade ?? null;
    const novaGroup = openFoodFactsProduct.novaGroup ?? aiAnalysis?.novaGroup ?? null;
    const giRating = gi != null ? getIndexRating(gi, 'gi') : null;
    const liRating = li != null ? getIndexRating(li, 'li') : null;
    const nutNum = (v: number | null | undefined) => (v != null && Number.isFinite(Number(v)) ? Number(v) : null);
    const carbs = nutNum(nut?.carbs);
    const fiber = nutNum(nut?.fiber);
    const protein = nutNum(nut?.protein);
    const netCarbs = carbs != null && fiber != null ? Math.max(0, carbs - fiber) : carbs;

    return (
      <div className="space-y-4">
        <Card
          className={cn(
            'cursor-pointer transition-all duration-300 hover:shadow-card hover:-translate-y-1',
            (isAi || aiAnalysis) && 'border-amber-500'
          )}
        >
          <CardHeader className="pb-2 p-3 sm:p-4 sm:pb-2">
            <div className="flex items-start justify-between gap-2">
              <div className="flex items-center gap-2 sm:gap-3 min-w-0">
                {openFoodFactsProduct.imageUrl ? (
                  <img src={openFoodFactsProduct.imageUrl} alt="" className="h-10 w-10 sm:h-12 sm:w-12 flex-shrink-0 object-cover rounded-lg" />
                ) : (
                  <span className="text-2xl sm:text-3xl flex-shrink-0">{isCosmetic ? '💄' : '🍽️'}</span>
                )}
                <div className="min-w-0">
                  <h3 className="font-display font-semibold text-foreground text-sm sm:text-base truncate">{openFoodFactsProduct.name}</h3>
                  {openFoodFactsProduct.brand && (
                    <p className="text-xs text-muted-foreground truncate">{openFoodFactsProduct.brand}</p>
                  )}
                  <div className="flex flex-wrap gap-1 mt-0.5 sm:mt-1">
                    {nutriscoreGrade && (
                      <Badge className={`uppercase text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5 ${nutriscoreGrade === 'a' ? 'bg-green-500 text-white' : nutriscoreGrade === 'b' ? 'bg-lime-500 text-white' : nutriscoreGrade === 'c' ? 'bg-yellow-500 text-black' : nutriscoreGrade === 'd' ? 'bg-orange-500 text-white' : 'bg-red-500 text-white'}`}>
                        Nutri {nutriscoreGrade}
                      </Badge>
                    )}
                    {novaGroup != null && (
                      <Badge variant="outline" className="text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5">
                        NOVA {novaGroup}
                      </Badge>
                    )}
                    {tobin != null && tobinRating && (
                      <Badge variant="outline" className={`${tobinRating.color} ${tobinRating.bgColor} border-0 text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5`}>
                        TOBIN: {tobin}/50
                      </Badge>
                    )}
                    {!isCosmetic && aiAnalysis?.halal && (
                      <Badge className={`text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5 ${getHalalStatusColor(aiAnalysis.halal as 'halal' | 'haram' | 'mashbooh')}`}>
                        {aiAnalysis.halal === 'mashbooh' ? 'Mashbooh' : aiAnalysis.halal}
                      </Badge>
                    )}
                    {!isCosmetic && aiAnalysis?.kosher && (
                      <Badge className={`text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5 ${getKosherStatusColor(aiAnalysis.kosher as 'kosher' | 'not_kosher' | 'pareve' | 'dairy' | 'meat')}`}>
                        {aiAnalysis.kosher.replace('_', ' ')}
                      </Badge>
                    )}
                    {!isCosmetic && aiAnalysis?.organic && (
                      <Badge className={`text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5 ${getOrganicStatusColor(aiAnalysis.organic as 'organic' | 'non_organic' | 'varies')}`}>
                        {aiAnalysis.organic.replace('_', ' ')}
                      </Badge>
                    )}
                    {isAi || aiAnalysis ? (
                      <Badge variant="outline" className="gap-0.5 border-amber-500 text-amber-600 text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5">
                        <Sparkles className="h-3 w-3" />
                        {apiSourceString || 'AI'}
                      </Badge>
                    ) : (
                      <Badge variant="outline" className="gap-0.5 text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5">
                        <Globe className="h-3 w-3" />
                        {isCosmetic ? 'Database' : 'Open Food Facts'}
                      </Badge>
                    )}
                  </div>
                </div>
              </div>
              {user && itemId && (
                <div className="flex items-center gap-0.5 sm:gap-1 flex-shrink-0">
                  <PurchasedButton itemId={itemId} itemType={itemType} size="sm" />
                  <ShoppingListButton itemId={itemId} itemType={itemType} size="sm" />
                  <FavoriteButton itemId={itemId} itemType={itemType} size="sm" />
                </div>
              )}
            </div>
          </CardHeader>

          <CardContent className="space-y-3 sm:space-y-4 p-3 sm:p-4 pt-0 sm:pt-0">
            {isAi && (
              <div className="bg-amber-50 border border-amber-200 rounded-lg p-2 text-xs text-amber-800">
                AI-identified product. Verify details before using.
              </div>
            )}

            {/* Stats: GI*, LI, TOBIN - show for both food and cosmetics (cosmetics use AI analysis) */}
            <div className="space-y-3 sm:space-y-4">
                {/* Index Meters - GI* and LI (same as FoodCard) */}
                <div className="grid grid-cols-2 gap-2 sm:gap-4">
                  <div className="space-y-1 sm:space-y-1.5">
                    <div className="flex items-center justify-between text-xs sm:text-sm">
                      <span className="text-muted-foreground">GI*</span>
                      <span className={cn('font-semibold', gi != null ? giRating?.color : 'text-muted-foreground')}>
                        {gi != null ? gi.toFixed(2) : '—'}
                      </span>
                    </div>
                    <div className="h-1.5 sm:h-2 bg-muted rounded-full overflow-hidden">
                      <div
                        className={cn(
                          'h-full rounded-full transition-all duration-500',
                          gi == null ? 'bg-muted' : gi < 0.6 ? 'bg-category-safe' : gi < 1.2 ? 'bg-category-warning' : 'bg-category-danger'
                        )}
                        style={{ width: gi != null ? `${Math.min(gi * 30, 100)}%` : '0%' }}
                      />
                    </div>
                    <p className={cn('text-[10px] sm:text-xs', gi != null && giRating ? giRating.color : 'text-muted-foreground')}>
                      {gi != null && giRating ? giRating.label : '—'}
                    </p>
                  </div>
                  <div className="space-y-1 sm:space-y-1.5">
                    <div className="flex items-center justify-between text-xs sm:text-sm">
                      <span className="text-muted-foreground">LI</span>
                      <span className={cn('font-semibold', li != null ? liRating?.color : 'text-muted-foreground')}>
                        {li != null ? li.toFixed(2) : '—'}
                      </span>
                    </div>
                    <div className="h-1.5 sm:h-2 bg-muted rounded-full overflow-hidden">
                      <div
                        className={cn(
                          'h-full rounded-full transition-all duration-500',
                          li == null ? 'bg-muted' : li < 0.5 ? 'bg-category-safe' : li < 1.0 ? 'bg-category-warning' : 'bg-category-danger'
                        )}
                        style={{ width: li != null ? `${Math.min(li * 40, 100)}%` : '0%' }}
                      />
                    </div>
                    <p className={cn('text-[10px] sm:text-xs', li != null && liRating ? liRating.color : 'text-muted-foreground')}>
                      {li != null && liRating ? liRating.label : '—'}
                    </p>
                  </div>
                </div>

                {/* TOBIN (show for both food and cosmetics) */}
                <div className="flex justify-between text-[10px] sm:text-xs border-t pt-1.5 sm:pt-2">
                  <span className={cn(tobin != null && tobinRating ? tobinRating.color : 'text-muted-foreground')}>
                    TOBIN: {tobin != null ? `${tobin}/50` : '—/50'}
                  </span>
                  {tobin != null && tobinRating && <span className={tobinRating.color}>{tobinRating.label}</span>}
                </div>

                {/* Quick Stats (Carbs/Fiber/Protein - food only) */}
                {!isCosmetic && (
                <div className="flex flex-wrap gap-1 sm:gap-2 text-[10px] sm:text-xs">
                  <span className="px-1.5 sm:px-2 py-0.5 sm:py-1 bg-secondary rounded-full">
                    Carbs: {netCarbs != null ? `${netCarbs}g` : '—'}
                  </span>
                  <span className="px-1.5 sm:px-2 py-0.5 sm:py-1 bg-secondary rounded-full">
                    Fiber: {fiber != null ? `${fiber}g` : '—'}
                  </span>
                  <span className="px-1.5 sm:px-2 py-0.5 sm:py-1 bg-secondary rounded-full">
                    Protein: {protein != null ? `${protein}g` : '—'}
                  </span>
                </div>
                )}
            </div>

            {!user && (
              <div className="text-center pt-2">
                <p className="text-xs text-muted-foreground mb-2">Sign in to save favorites and shopping list</p>
                <Link to="/auth"><Button variant="outline" size="sm">Sign In</Button></Link>
              </div>
            )}

            {user && isAi && onSaveAiProduct && (
              <Button variant="outline" size="sm" className="w-full gap-2 border-amber-500 text-amber-600 hover:bg-amber-50" onClick={onSaveAiProduct}>
                <Database className="h-4 w-4" />
                Save to Database
              </Button>
            )}
          </CardContent>
        </Card>

        {aiAnalysis && (
          <AIAnalysisCard analysis={aiAnalysis} hasWebSearch={apiSourceString?.includes('AI') || isAi} />
        )}
        <PhytoestrogenAnalysisCard ingredients={openFoodFactsProduct.ingredients} productName={openFoodFactsProduct.name} />
      </div>
    );
  }

  return null;
};