import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { 
  Sparkles, AlertTriangle, CheckCircle, Info, 
  ShoppingCart, Heart, ExternalLink, Package,
  Flame, Droplets, Wheat, Dumbbell, Bot, Star
} from 'lucide-react';
import { ScanHistoryItem } from '@/hooks/useScanHistory';
import { useFavorites } from '@/contexts/FavoritesContext';
import { useShoppingList } from '@/contexts/ShoppingListContext';
import { formatDistanceToNow } from 'date-fns';
import { estimateGlycogenicIndex, estimateLipogenicIndex, estimateTobinFromIndices, hasEnoughNutritionForIndices } from '@/utils/nutritionIndices';
import { getTobinRating, getHalalStatusColor, getKosherStatusColor, getOrganicStatusColor } from '@/data/foodDatabase';
import foodPlaceholder from '@/assets/food-placeholder.png';
import cosmeticPlaceholder from '@/assets/cosmetic-placeholder.png';

interface ScannedProductDetailModalProps {
  item: ScanHistoryItem | null;
  open: boolean;
  onClose: () => void;
}

export function ScannedProductDetailModal({ item, open, onClose }: ScannedProductDetailModalProps) {
  const { isFavorite, toggleFavorite } = useFavorites();
  const { isInList, toggleItem } = useShoppingList();

  if (!item) return null;

  const details = item.product_details;
  const isCosmetic = item.product_type === 'cosmetic';
  const placeholderImage = isCosmetic ? cosmeticPlaceholder : foodPlaceholder;
  const isAiGenerated = details?.isAiGenerated === true || details?.source === 'ai';

  const getSafetyColor = (level?: string) => {
    switch (level) {
      case 'safe': return 'bg-category-safe text-white';
      case 'low_concern': return 'bg-emerald-500 text-white';
      case 'moderate_concern': return 'bg-amber-500 text-white';
      case 'high_concern': return 'bg-category-danger text-white';
      default: return 'bg-muted text-muted-foreground';
    }
  };

  const getSafetyLabel = (level?: string) => {
    switch (level) {
      case 'safe': return 'Safe';
      case 'low_concern': return 'Low Concern';
      case 'moderate_concern': return 'Moderate Concern';
      case 'high_concern': return 'High Concern';
      default: return 'Unknown';
    }
  };

  // AI may return array items as strings or objects (e.g. { concern, explanation } or { name, note }) – ensure we render a string
  const toDisplayText = (v: unknown): string => {
    if (v == null) return '';
    if (typeof v === 'string') return v;
    if (typeof v === 'object' && v !== null) {
      const o = v as Record<string, unknown>;
      const concern = o.concern != null ? String(o.concern) : '';
      const explanation = o.explanation != null ? String(o.explanation) : '';
      if (concern && explanation) return `${concern}: ${explanation}`;
      if (concern) return concern;
      if (explanation) return explanation;
      const name = o.name != null ? String(o.name) : '';
      const note = o.note != null ? String(o.note) : '';
      if (name && note) return `${name}. ${note}`;
      if (name) return name;
      if (note) return note;
    }
    return String(v);
  };

  const getNutriScoreColor = (grade?: string) => {
    switch (grade?.toLowerCase()) {
      case 'a': return 'bg-emerald-500 text-white';
      case 'b': return 'bg-lime-500 text-white';
      case 'c': return 'bg-yellow-500 text-black';
      case 'd': return 'bg-orange-500 text-white';
      case 'e': return 'bg-red-500 text-white';
      default: return 'bg-muted text-muted-foreground';
    }
  };

  const isFav = isFavorite(item.barcode, item.product_type);
  const inList = isInList(item.barcode, item.product_type);

  return (
    <Dialog open={open} onOpenChange={onClose}>
      <DialogContent className="flex max-h-[90dvh] w-[calc(100vw-2rem)] max-w-2xl flex-col gap-0 p-0 overflow-hidden sm:w-full">
        <div className="flex min-h-0 flex-1 flex-col overflow-hidden">
          <div className="overflow-y-auto overscroll-contain max-h-[calc(90dvh-2rem)]">
            <div className="p-4 sm:p-6">
              <DialogHeader className="mb-4">
                <div className="flex items-start gap-3 sm:gap-4">
                  <img 
                    src={item.image_url || details?.imageUrl || placeholderImage}
                    alt={item.product_name}
                    className="h-20 w-20 shrink-0 object-cover rounded-lg bg-muted sm:h-24 sm:w-24"
                  />
                  <div className="min-w-0 flex-1">
                    <DialogTitle className="text-lg font-bold mb-2 break-words sm:text-xl">
                      {item.product_name}
                    </DialogTitle>
                  <div className="flex flex-wrap gap-2 mb-2">
                    <Badge variant="outline" className="capitalize">
                      {item.product_type}
                    </Badge>
                    {details?.source && (
                      <Badge variant="secondary" className="gap-1">
                        {details.isAiGenerated && <Sparkles className="h-3 w-3" />}
                        {details.apiSourceString || details.source}
                      </Badge>
                    )}
                  </div>
                  {details?.brand && (
                    <p className="text-sm text-muted-foreground">Brand: {details.brand}</p>
                  )}
                  <p className="text-xs text-muted-foreground mt-1">
                    Scanned {formatDistanceToNow(new Date(item.scanned_at), { addSuffix: true })}
                  </p>
                </div>
              </div>
            </DialogHeader>

            {/* AI-Generated Data Disclaimer */}
            {isAiGenerated && (
              <Alert className="mb-4 border-warning bg-warning/10">
                <Bot className="h-4 w-4" />
                <AlertTitle className="text-sm font-semibold">AI-Generated Product Data</AlertTitle>
                <AlertDescription className="text-xs text-muted-foreground">
                  This product was not found in our verified databases. The information shown was generated by AI based on the product name and barcode, and may contain inaccuracies. Please verify important details independently before making health or purchase decisions.
                </AlertDescription>
              </Alert>
            )}

            {/* AI Analysis Section */}
            {details?.aiAnalysis && (
              <Card className="mb-4 border-amber-200 bg-amber-50/50">
                <CardContent className="pt-4">
                  <div className="flex items-center gap-2 mb-3">
                    <Sparkles className="h-5 w-5 text-amber-500" />
                    <h3 className="font-semibold">AI Safety Analysis</h3>
                    <Badge className={getSafetyColor(details.aiAnalysis.safetyLevel)}>
                      {getSafetyLabel(details.aiAnalysis.safetyLevel)}
                    </Badge>
                  </div>
                  
                  <p className="text-sm text-muted-foreground mb-4 break-words">
                    {details.aiAnalysis.summary}
                  </p>

                  {details.aiAnalysis.concerns.length > 0 && (
                    <div className="mb-4">
                      <h4 className="text-sm font-medium flex items-center gap-1 mb-2">
                        <AlertTriangle className="h-4 w-4 text-amber-500" />
                        Concerns
                      </h4>
                      <ul className="text-sm space-y-1">
                        {details.aiAnalysis.concerns.map((concern, i) => (
                          <li key={i} className="flex items-start gap-2">
                            <span className="text-amber-500 mt-1">•</span>
                            <span>{toDisplayText(concern)}</span>
                          </li>
                        ))}
                      </ul>
                    </div>
                  )}

                  {details.aiAnalysis.saferAlternatives.length > 0 && (
                    <div className="mb-4">
                      <h4 className="text-sm font-medium flex items-center gap-1 mb-2">
                        <CheckCircle className="h-4 w-4 text-category-safe" />
                        Safer Alternatives
                      </h4>
                      <ul className="text-sm space-y-1">
                        {details.aiAnalysis.saferAlternatives.map((alt, i) => (
                          <li key={i} className="flex items-start gap-2">
                            <span className="text-category-safe mt-1">•</span>
                            <span>{toDisplayText(alt)}</span>
                          </li>
                        ))}
                      </ul>
                    </div>
                  )}

                  {details.aiAnalysis.ingredients.length > 0 && (
                    <div>
                      <h4 className="text-sm font-medium mb-2">Key Ingredients</h4>
                      <div className="flex flex-wrap gap-1">
                        {details.aiAnalysis.ingredients.slice(0, 10).map((ing, i) => (
                          <Badge key={i} variant="outline" className="text-xs">
                            {toDisplayText(ing)}
                          </Badge>
                        ))}
                        {details.aiAnalysis.ingredients.length > 10 && (
                          <Badge variant="secondary" className="text-xs">
                            +{details.aiAnalysis.ingredients.length - 10} more
                          </Badge>
                        )}
                      </div>
                    </div>
                  )}
                </CardContent>
              </Card>
            )}

            {/* Key stats: Nutri-Score, NOVA (food), GI*, LI, TOBIN - show for food and cosmetics (cosmetics use AI) */}
            {(() => {
              const nut = details?.nutrition;
              const hasNut = !isCosmetic && !!nut && hasEnoughNutritionForIndices(nut);
              const ai = details?.aiAnalysis;
              const clamp = (n: number, min: number, max: number) => Math.max(min, Math.min(max, n));
              const giFromNut = nut ? estimateGlycogenicIndex(nut) : null;
              const liFromNut = nut ? estimateLipogenicIndex(nut) : null;
              const gi = (hasNut && giFromNut != null) ? giFromNut : (typeof ai?.glycogenicIndex === 'number' && Number.isFinite(ai?.glycogenicIndex) ? clamp(ai.glycogenicIndex, 0, 3) : giFromNut);
              const li = (hasNut && liFromNut != null) ? liFromNut : (typeof ai?.lipogenicIndex === 'number' && Number.isFinite(ai?.lipogenicIndex) ? clamp(ai.lipogenicIndex, 0, 3) : liFromNut);
              const tobin = (hasNut && (gi != null || li != null)) ? estimateTobinFromIndices(gi, li) : (typeof ai?.tobinScore === 'number' && Number.isFinite(ai?.tobinScore) ? clamp(ai.tobinScore, 0, 50) : (!isCosmetic ? estimateTobinFromIndices(gi, li) : null));
              const tobinRating = tobin != null ? getTobinRating(tobin) : null;
              const showStats = !isCosmetic || tobin != null || gi != null || li != null;
              if (!showStats) return null;
              return (
                <Card className="mb-4">
                  <CardContent className="pt-4">
                    <h3 className="font-semibold mb-3">Key statistics</h3>
                    <div className="flex flex-wrap items-center gap-x-4 gap-y-2 text-sm">
                        {!isCosmetic && (
                          <>
                        <span className="text-muted-foreground">Nutri-Score:</span>
                        {(details?.nutriscoreGrade ?? details?.aiAnalysis?.nutriscoreGrade) ? (
                          <Badge className={getNutriScoreColor(details?.nutriscoreGrade ?? details?.aiAnalysis?.nutriscoreGrade)}>{(details?.nutriscoreGrade ?? details?.aiAnalysis?.nutriscoreGrade).toUpperCase()}</Badge>
                        ) : (
                          <span className="text-muted-foreground">—</span>
                        )}
                        <span className="text-muted-foreground">NOVA:</span>
                        {(details?.novaGroup ?? details?.aiAnalysis?.novaGroup) != null ? (
                          <Badge variant="outline">Group {details?.novaGroup ?? details?.aiAnalysis?.novaGroup}</Badge>
                        ) : (
                          <span className="text-muted-foreground">—</span>
                        )}
                          </>
                        )}
                      <span className="text-muted-foreground">GI*:</span>
                      {gi != null ? (
                        <Badge variant="outline" className={gi < 0.6 ? 'border-green-500 text-green-700' : gi < 1 ? 'border-amber-500 text-amber-700' : 'border-red-500 text-red-700'}>
                          {gi.toFixed(2)}
                        </Badge>
                      ) : (
                        <span className="text-muted-foreground">—</span>
                      )}
                      <span className="text-muted-foreground">LI:</span>
                      {li != null ? (
                        <Badge variant="outline" className={li < 0.5 ? 'border-green-500 text-green-700' : li < 1 ? 'border-amber-500 text-amber-700' : 'border-red-500 text-red-700'}>
                          {li.toFixed(2)}
                        </Badge>
                      ) : (
                        <span className="text-muted-foreground">—</span>
                      )}
                      <span className="text-muted-foreground">TOBIN:</span>
                      {tobin != null && tobinRating ? (
                        <Badge variant="outline" className={tobinRating.bgColor}>{tobin}/50</Badge>
                      ) : (
                        <span className="text-muted-foreground">—</span>
                      )}
                    </div>
                  </CardContent>
                </Card>
              );
            })()}

            {/* Dietary Classifications (from AI / product_details) */}
            {!isCosmetic && (() => {
              const dietary = details?.dietaryClassification || (details?.aiAnalysis && (details.aiAnalysis.halal || details.aiAnalysis.kosher || details.aiAnalysis.organic) ? {
                halal: details.aiAnalysis.halal ?? undefined,
                kosher: details.aiAnalysis.kosher ?? undefined,
                organic: details.aiAnalysis.organic ?? undefined,
                certificationNotes: details.aiAnalysis.certificationNotes ?? undefined,
              } : null);
              if (!dietary || (!dietary.halal && !dietary.kosher && !dietary.organic)) return null;
              return (
                <Card className="mb-4">
                  <CardContent className="pt-4">
                    <h4 className="font-semibold mb-3 flex items-center gap-2">
                      <Star className="h-5 w-5 text-slate-600" />
                      Dietary status
                    </h4>
                    <div className="grid grid-cols-3 gap-3">
                      {dietary.halal && (
                        <div className="text-center p-2 rounded-lg bg-background border">
                          <p className="text-xs text-muted-foreground mb-1">Halal</p>
                          <Badge className={getHalalStatusColor(dietary.halal as 'halal' | 'haram' | 'mashbooh')}>
                            {dietary.halal === 'mashbooh' ? 'Doubtful' : dietary.halal.replace('_', ' ')}
                          </Badge>
                        </div>
                      )}
                      {dietary.kosher && (
                        <div className="text-center p-2 rounded-lg bg-background border">
                          <p className="text-xs text-muted-foreground mb-1">Kosher</p>
                          <Badge className={getKosherStatusColor(dietary.kosher as 'kosher' | 'not_kosher' | 'pareve' | 'dairy' | 'meat')}>
                            {dietary.kosher.replace('_', ' ')}
                          </Badge>
                        </div>
                      )}
                      {dietary.organic && (
                        <div className="text-center p-2 rounded-lg bg-background border">
                          <p className="text-xs text-muted-foreground mb-1">Organic</p>
                          <Badge className={getOrganicStatusColor(dietary.organic as 'organic' | 'non_organic' | 'varies')}>
                            {dietary.organic.replace('_', ' ')}
                          </Badge>
                        </div>
                      )}
                    </div>
                    {dietary.certificationNotes && (
                      <p className="text-sm text-muted-foreground mt-2 italic">{dietary.certificationNotes}</p>
                    )}
                  </CardContent>
                </Card>
              );
            })()}

            {/* Nutrition Section (for food) */}
            {!isCosmetic && details?.nutrition && (
              <Card className="mb-4">
                <CardContent className="pt-4">
                  <h3 className="font-semibold mb-3 flex items-center gap-2">
                    <Package className="h-5 w-5 text-primary" />
                    Nutritional Information
                  </h3>
                  <div className="grid grid-cols-4 gap-3 text-center">
                    {details.nutrition.calories !== null && (
                      <div className="bg-muted rounded-lg p-3">
                        <Flame className="h-4 w-4 mx-auto mb-1 text-orange-500" />
                        <p className="text-lg font-bold">{details.nutrition.calories}</p>
                        <p className="text-xs text-muted-foreground">kcal</p>
                      </div>
                    )}
                    {details.nutrition.protein !== null && (
                      <div className="bg-muted rounded-lg p-3">
                        <Dumbbell className="h-4 w-4 mx-auto mb-1 text-red-500" />
                        <p className="text-lg font-bold">{details.nutrition.protein}g</p>
                        <p className="text-xs text-muted-foreground">Protein</p>
                      </div>
                    )}
                    {details.nutrition.carbs !== null && (
                      <div className="bg-muted rounded-lg p-3">
                        <Wheat className="h-4 w-4 mx-auto mb-1 text-amber-500" />
                        <p className="text-lg font-bold">{details.nutrition.carbs}g</p>
                        <p className="text-xs text-muted-foreground">Carbs</p>
                      </div>
                    )}
                    {details.nutrition.fat !== null && (
                      <div className="bg-muted rounded-lg p-3">
                        <Droplets className="h-4 w-4 mx-auto mb-1 text-yellow-500" />
                        <p className="text-lg font-bold">{details.nutrition.fat}g</p>
                        <p className="text-xs text-muted-foreground">Fat</p>
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Ingredients Section */}
            {details?.ingredients && (
              <Card className="mb-4">
                <CardContent className="pt-4">
                  <h3 className="font-semibold mb-2 flex items-center gap-2">
                    <Info className="h-5 w-5 text-muted-foreground" />
                    Ingredients
                  </h3>
                  <p className="text-sm text-muted-foreground whitespace-pre-wrap break-words overflow-hidden">
                    {details.ingredients}
                  </p>
                </CardContent>
              </Card>
            )}

            {/* Categories */}
            {details?.categories && (
              <div className="mb-4">
                <h4 className="text-sm font-medium mb-2">Categories</h4>
                <p className="text-sm text-muted-foreground break-words">{details.categories}</p>
              </div>
            )}

            {/* Barcode */}
            <div className="mb-4">
              <h4 className="text-sm font-medium mb-1">Barcode</h4>
              <code className="text-sm bg-muted px-2 py-1 rounded">{item.barcode}</code>
            </div>

            <Separator className="my-4" />

            {/* Actions */}
            <div className="flex gap-2">
              <Button 
                variant={isFav ? "default" : "outline"}
                className="flex-1 gap-2"
                onClick={() => toggleFavorite(item.barcode, item.product_type)}
              >
                <Heart className={`h-4 w-4 ${isFav ? 'fill-current' : ''}`} />
                {isFav ? 'Favorited' : 'Add to Favorites'}
              </Button>
              <Button 
                variant={inList ? "default" : "outline"}
                className="flex-1 gap-2"
                onClick={() => toggleItem(item.barcode, item.product_type)}
              >
                <ShoppingCart className="h-4 w-4" />
                {inList ? 'In Shopping List' : 'Add to List'}
              </Button>
            </div>

            {/* Sources */}
            {details?.aiAnalysis?.sources && details.aiAnalysis.sources.length > 0 && (
              <div className="mt-4 pt-4 border-t">
                <h4 className="text-sm font-medium mb-2 flex items-center gap-1">
                  <ExternalLink className="h-4 w-4" />
                  Sources
                </h4>
                <div className="space-y-1 overflow-hidden">
                  {details.aiAnalysis.sources.map((source, i) => (
                    <a 
                      key={i}
                      href={source}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="text-xs text-primary hover:underline block break-all"
                    >
                      {source}
                    </a>
                  ))}
                </div>
              </div>
            )}
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}
