import { Card, CardContent, CardHeader } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Sparkles, AlertTriangle, ShieldCheck, Shield, ShieldX, Bot } from 'lucide-react';
import { FavoriteButton } from './FavoriteButton';
import { ShoppingListButton } from './ShoppingListButton';
import { PurchasedButton } from './PurchasedButton';
import { useFavorites } from '@/contexts/FavoritesContext';
import { useShoppingList } from '@/contexts/ShoppingListContext';
import { ScanHistoryItem } from '@/hooks/useScanHistory';
import { estimateGlycogenicIndex, estimateLipogenicIndex, estimateTobinFromIndices, hasEnoughNutritionForIndices } from '@/utils/nutritionIndices';
import { getTobinRating, getIndexRating } from '@/data/foodDatabase';
import { cn } from '@/lib/utils';
import foodPlaceholder from '@/assets/food-placeholder.png';
import cosmeticPlaceholder from '@/assets/cosmetic-placeholder.png';

export interface ScannedProduct {
  id: string;
  product_name: string;
  product_type: 'food' | 'cosmetic';
  image_url?: string | null;
  barcode: string;
  isScanned: true;
  quantity?: number;
  scanHistoryItem?: ScanHistoryItem;
}

// Use barcode as the unique key for ScannedProduct since that's what's stored in favorites/shopping
export function getScannedProductKey(product: ScannedProduct): string {
  return product.barcode;
}

// Nutri-Score badge styling
const getNutriScoreStyle = (grade: string) => {
  const g = grade?.toUpperCase();
  switch (g) {
    case 'A': return 'bg-green-600 text-white';
    case 'B': return 'bg-lime-500 text-white';
    case 'C': return 'bg-yellow-400 text-black';
    case 'D': return 'bg-orange-500 text-white';
    case 'E': return 'bg-red-600 text-white';
    default: return 'bg-muted text-muted-foreground';
  }
};

// NOVA group styling
const getNovaStyle = (group: number | string) => {
  const g = typeof group === 'string' ? parseInt(group) : group;
  switch (g) {
    case 1: return { style: 'bg-green-600 text-white', label: 'Unprocessed' };
    case 2: return { style: 'bg-lime-500 text-white', label: 'Processed ingredients' };
    case 3: return { style: 'bg-yellow-500 text-black', label: 'Processed' };
    case 4: return { style: 'bg-red-600 text-white', label: 'Ultra-processed' };
    default: return { style: 'bg-muted text-muted-foreground', label: 'Unknown' };
  }
};

// Safety level styling for cosmetics
const getSafetyStyle = (level: string) => {
  switch (level) {
    case 'safe': return { style: 'bg-green-600 text-white', icon: ShieldCheck, label: 'Safe' };
    case 'low_concern': return { style: 'bg-lime-500 text-white', icon: Shield, label: 'Low Risk' };
    case 'moderate_concern': return { style: 'bg-orange-500 text-white', icon: AlertTriangle, label: 'Moderate' };
    case 'high_concern': return { style: 'bg-red-600 text-white', icon: ShieldX, label: 'High Risk' };
    default: return { style: 'bg-muted text-muted-foreground', icon: Shield, label: 'Unknown' };
  }
};

interface ScannedProductMiniCardProps {
  product: ScannedProduct;
  showQuantity?: boolean;
  onProductClick?: (item: ScanHistoryItem) => void;
}

export function ScannedProductMiniCard({ 
  product, 
  showQuantity = false,
  onProductClick 
}: ScannedProductMiniCardProps) {
  const placeholderImage = product.product_type === 'cosmetic' ? cosmeticPlaceholder : foodPlaceholder;
  
  // Use barcode as the consistent ID for favorites/shopping list
  const itemId = product.barcode;
  const itemType = product.product_type;
  const { isFavorite, toggleFavorite } = useFavorites();
  const { items: shoppingItems, isInList, toggleItem, togglePurchased } = useShoppingList();
  
  // Check if this barcode is favorited under either type (handles legacy data mismatches)
  const isFavFood = isFavorite(itemId, 'food');
  const isFavCosmetic = isFavorite(itemId, 'cosmetic');
  const isFav = isFavFood || isFavCosmetic;
  
  // Check if this barcode is in shopping list under either type
  const inListFood = isInList(itemId, 'food');
  const inListCosmetic = isInList(itemId, 'cosmetic');
  const inList = inListFood || inListCosmetic;
  
  // Find shopping item to check purchased status
  const shoppingItem = shoppingItems.find(i => 
    i.item_id === itemId && (i.item_type === 'food' || i.item_type === 'cosmetic')
  );
  const isPurchased = shoppingItem?.is_purchased ?? false;

  // Extract product details for indicators (prefer top-level, fallback to AI analysis)
  const details = product.scanHistoryItem?.product_details;
  const nutriScore = details?.nutriscoreGrade ?? details?.aiAnalysis?.nutriscoreGrade;
  const novaGroup = details?.novaGroup ?? details?.aiAnalysis?.novaGroup;
  const aiAnalysis = details?.aiAnalysis;
  const safetyLevel = aiAnalysis?.safetyLevel;
  const isAiGenerated = details?.isAiGenerated === true || details?.source === 'ai';

  // GI*, LI, TOBIN: for food prefer calculated from nutrition, fallback to AI; for cosmetics use AI analysis
  const nut = details?.nutrition;
  const hasNut = product.product_type === 'food' && !!nut && hasEnoughNutritionForIndices(nut);
  const clamp = (n: number, min: number, max: number) => Math.max(min, Math.min(max, n));
  const giFromNut = product.product_type === 'food' && nut ? estimateGlycogenicIndex(nut) : null;
  const liFromNut = product.product_type === 'food' && nut ? estimateLipogenicIndex(nut) : null;
  const gi = product.product_type === 'food'
    ? (hasNut && giFromNut != null ? giFromNut : (typeof aiAnalysis?.glycogenicIndex === 'number' && Number.isFinite(aiAnalysis.glycogenicIndex) ? clamp(aiAnalysis.glycogenicIndex, 0, 3) : giFromNut))
    : (typeof aiAnalysis?.glycogenicIndex === 'number' && Number.isFinite(aiAnalysis.glycogenicIndex) ? clamp(aiAnalysis.glycogenicIndex, 0, 3) : null);
  const li = product.product_type === 'food'
    ? (hasNut && liFromNut != null ? liFromNut : (typeof aiAnalysis?.lipogenicIndex === 'number' && Number.isFinite(aiAnalysis.lipogenicIndex) ? clamp(aiAnalysis.lipogenicIndex, 0, 3) : liFromNut))
    : (typeof aiAnalysis?.lipogenicIndex === 'number' && Number.isFinite(aiAnalysis.lipogenicIndex) ? clamp(aiAnalysis.lipogenicIndex, 0, 3) : null);
  const tobin = product.product_type === 'food'
    ? (hasNut && (gi != null || li != null) ? estimateTobinFromIndices(gi, li) : (typeof aiAnalysis?.tobinScore === 'number' && Number.isFinite(aiAnalysis.tobinScore) ? clamp(aiAnalysis.tobinScore, 0, 50) : estimateTobinFromIndices(gi, li)))
    : (typeof aiAnalysis?.tobinScore === 'number' && Number.isFinite(aiAnalysis.tobinScore) ? clamp(aiAnalysis.tobinScore, 0, 50) : null);
  const tobinRating = tobin != null ? getTobinRating(tobin) : null;
  const giRating = gi != null ? getIndexRating(gi, 'gi') : null;
  const liRating = li != null ? getIndexRating(li, 'li') : null;
  const nutNum = (v: number | null | undefined) => (v != null && Number.isFinite(Number(v)) ? Number(v) : null);
  const carbs = nutNum(nut?.carbs);
  const fiber = nutNum(nut?.fiber);
  const protein = nutNum(nut?.protein);
  const netCarbs = carbs != null && fiber != null ? Math.max(0, carbs - fiber) : carbs;

  const handleClick = () => {
    if (product.scanHistoryItem && onProductClick) {
      onProductClick(product.scanHistoryItem);
    }
  };
  
  return (
    <Card 
      className={cn(
        'cursor-pointer transition-all duration-300 hover:shadow-card hover:-translate-y-1',
        isAiGenerated && 'border-amber-500'
      )}
      onClick={handleClick}
    >
      <CardHeader className="pb-2 p-3 sm:p-4 sm:pb-2">
        <div className="flex items-start justify-between gap-2">
          <div className="flex items-center gap-2 sm:gap-3 min-w-0">
            {product.image_url ? (
              <img src={product.image_url} alt="" className="h-10 w-10 sm:h-12 sm:w-12 flex-shrink-0 object-cover rounded-lg" />
            ) : (
              <span className="text-2xl sm:text-3xl flex-shrink-0">{product.product_type === 'cosmetic' ? '💄' : '🍽️'}</span>
            )}
            <div className="min-w-0">
              <h3 className="font-display font-semibold text-foreground text-sm sm:text-base truncate">{product.product_name}</h3>
              <div className="flex flex-wrap gap-1 mt-0.5 sm:mt-1">
                {product.product_type === 'food' && nutriScore && (
                  <Badge className={`uppercase text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5 ${getNutriScoreStyle(nutriScore)}`}>
                    Nutri {nutriScore}
                  </Badge>
                )}
                {product.product_type === 'food' && novaGroup != null && (
                  <Badge variant="outline" className="text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5">
                    NOVA {novaGroup}
                  </Badge>
                )}
                {tobin != null && tobinRating && (
                  <Badge variant="outline" className={`${tobinRating.color} ${tobinRating.bgColor} border-0 text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5`}>
                    TOBIN: {tobin}/50
                  </Badge>
                )}
                {isAiGenerated && (
                  <Badge variant="outline" className="gap-0.5 border-amber-500 text-amber-600 text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5">
                    <Bot className="h-3 w-3" />
                    AI
                  </Badge>
                )}
                {product.product_type === 'cosmetic' && safetyLevel && (
                  <Badge className={`${getSafetyStyle(safetyLevel).style} text-[10px] sm:text-xs px-1.5 sm:px-2 py-0 sm:py-0.5 gap-0.5`}>
                    {(() => {
                      const IconComponent = getSafetyStyle(safetyLevel).icon;
                      return <IconComponent className="h-3 w-3" />;
                    })()}
                    {getSafetyStyle(safetyLevel).label}
                  </Badge>
                )}
              </div>
            </div>
          </div>
          <div className="flex items-center gap-0.5 sm:gap-1 flex-shrink-0">
            <PurchasedButton itemId={itemId} itemType={itemType} size="sm" />
            <ShoppingListButton itemId={itemId} itemType={itemType} size="sm" />
            <FavoriteButton itemId={itemId} itemType={itemType} size="sm" />
          </div>
        </div>
      </CardHeader>

      {/* Stats: GI*, LI, TOBIN - show for food and cosmetics (cosmetics use AI) */}
      {(product.product_type === 'food' || tobin != null || gi != null || li != null) && (
        <CardContent className="space-y-3 sm:space-y-4 p-3 sm:p-4 pt-0 sm:pt-0">
          {/* Index Meters - GI* and LI (same as FoodCard) */}
          <div className="grid grid-cols-2 gap-2 sm:gap-4">
            <div className="space-y-1 sm:space-y-1.5">
              <div className="flex items-center justify-between text-xs sm:text-sm">
                <span className="text-muted-foreground">GI*</span>
                <span className={cn('font-semibold', gi != null ? giRating?.color : 'text-muted-foreground')}>
                  {gi != null ? gi.toFixed(2) : '—'}
                </span>
              </div>
              <div className="h-1.5 sm:h-2 bg-muted rounded-full overflow-hidden">
                <div
                  className={cn(
                    'h-full rounded-full transition-all duration-500',
                    gi == null ? 'bg-muted' : gi < 0.6 ? 'bg-category-safe' : gi < 1.2 ? 'bg-category-warning' : 'bg-category-danger'
                  )}
                  style={{ width: gi != null ? `${Math.min(gi * 30, 100)}%` : '0%' }}
                />
              </div>
              <p className={cn('text-[10px] sm:text-xs', gi != null && giRating ? giRating.color : 'text-muted-foreground')}>
                {gi != null && giRating ? giRating.label : '—'}
              </p>
            </div>
            <div className="space-y-1 sm:space-y-1.5">
              <div className="flex items-center justify-between text-xs sm:text-sm">
                <span className="text-muted-foreground">LI</span>
                <span className={cn('font-semibold', li != null ? liRating?.color : 'text-muted-foreground')}>
                  {li != null ? li.toFixed(2) : '—'}
                </span>
              </div>
              <div className="h-1.5 sm:h-2 bg-muted rounded-full overflow-hidden">
                <div
                  className={cn(
                    'h-full rounded-full transition-all duration-500',
                    li == null ? 'bg-muted' : li < 0.5 ? 'bg-category-safe' : li < 1.0 ? 'bg-category-warning' : 'bg-category-danger'
                  )}
                  style={{ width: li != null ? `${Math.min(li * 40, 100)}%` : '0%' }}
                />
              </div>
              <p className={cn('text-[10px] sm:text-xs', li != null && liRating ? liRating.color : 'text-muted-foreground')}>
                {li != null && liRating ? liRating.label : '—'}
              </p>
            </div>
          </div>

          {/* TOBIN (same as FoodCard - always show row for food) */}
          <div className="flex justify-between text-[10px] sm:text-xs border-t pt-1.5 sm:pt-2">
            <span className={cn(tobin != null && tobinRating ? tobinRating.color : 'text-muted-foreground')}>
              TOBIN: {tobin != null ? `${tobin}/50` : '—/50'}
            </span>
            {tobin != null && tobinRating && <span className={tobinRating.color}>{tobinRating.label}</span>}
          </div>

          {/* Quick Stats (food only) */}
          {product.product_type === 'food' && (
            <div className="flex flex-wrap gap-1 sm:gap-2 text-[10px] sm:text-xs">
              <span className="px-1.5 sm:px-2 py-0.5 sm:py-1 bg-secondary rounded-full">
                Carbs: {netCarbs != null ? `${netCarbs}g` : '—'}
              </span>
              <span className="px-1.5 sm:px-2 py-0.5 sm:py-1 bg-secondary rounded-full">
                Fiber: {fiber != null ? `${fiber}g` : '—'}
              </span>
              <span className="px-1.5 sm:px-2 py-0.5 sm:py-1 bg-secondary rounded-full">
                Protein: {protein != null ? `${protein}g` : '—'}
              </span>
            </div>
          )}
        </CardContent>
      )}

      {/* Cosmetic with no stats: show quantity only */}
      {product.product_type === 'cosmetic' && tobin == null && gi == null && li == null && (
        <CardContent className="p-3 sm:p-4 pt-0 sm:pt-0">
          {showQuantity && product.quantity && product.quantity > 1 && (
            <Badge className="mb-2">Quantity: {product.quantity}</Badge>
          )}
        </CardContent>
      )}
    </Card>
  );
}
