import { foodDatabase } from "@/data/foodDatabase";
import { Card, CardContent } from "@/components/ui/card";
import { AlertTriangle, CheckCircle, AlertCircle, Sparkles, Heart, ShoppingCart, History, ScanLine } from "lucide-react";
import { useAuth } from "@/hooks/useAuth";
import { useFavorites } from "@/hooks/useFavorites";
import { useShoppingList } from "@/hooks/useShoppingList";
import { useScanHistory } from "@/hooks/useScanHistory";

export function StatsOverview() {
  const { user } = useAuth();
  const { favorites } = useFavorites();
  const { items: shoppingItems } = useShoppingList();
  const { history: scanHistory } = useScanHistory();
  
  // Show user stats if logged in
  if (user) {
    const userStats = [
      {
        label: "Favorites",
        count: favorites.length,
        icon: Heart,
        color: "text-primary",
        bg: "bg-primary/10",
        description: "Saved items"
      },
      {
        label: "To Buy",
        count: shoppingItems.filter(i => !i.is_purchased).length,
        icon: ShoppingCart,
        color: "text-category-safe",
        bg: "bg-category-safe/10",
        description: "Shopping list"
      },
      {
        label: "Purchased",
        count: shoppingItems.filter(i => i.is_purchased).length,
        icon: CheckCircle,
        color: "text-muted-foreground",
        bg: "bg-muted",
        description: "Completed items"
      },
      {
        label: "Scanned",
        count: scanHistory.length,
        icon: ScanLine,
        color: "text-category-warning",
        bg: "bg-category-warning/10",
        description: "Product scans"
      },
    ];
    
    return (
      <div className="grid grid-cols-2 lg:grid-cols-4 gap-2 sm:gap-3 md:gap-4 w-full">
        {userStats.map((stat) => {
          const Icon = stat.icon;
          return (
            <Card key={stat.label} className="shadow-soft hover:shadow-card transition-shadow">
              <CardContent className="p-3 sm:p-4 md:pt-6">
                <div className="flex items-center gap-2 sm:gap-3">
                  <div className={`p-1.5 sm:p-2 md:p-2.5 rounded-lg sm:rounded-xl ${stat.bg}`}>
                    <Icon className={`h-4 w-4 sm:h-5 sm:w-5 ${stat.color}`} />
                  </div>
                  <div className="min-w-0">
                    <p className={`text-lg sm:text-xl md:text-2xl font-display font-bold ${stat.color}`}>{stat.count}</p>
                    <p className="text-xs sm:text-sm font-medium truncate">{stat.label}</p>
                  </div>
                </div>
                <p className="text-[10px] sm:text-xs text-muted-foreground mt-1.5 sm:mt-2 line-clamp-1">{stat.description}</p>
              </CardContent>
            </Card>
          );
        })}
      </div>
    );
  }
  
  // Show database stats for non-logged in users
  const categoryCount = {
    1: foodDatabase.filter(f => f.category === 1).length,
    2: foodDatabase.filter(f => f.category === 2).length,
    3: foodDatabase.filter(f => f.category === 3).length,
    4: foodDatabase.filter(f => f.category === 4).length,
  };
  
  const stats = [
    {
      label: "Avoid",
      count: categoryCount[1],
      icon: AlertTriangle,
      color: "text-category-danger",
      bg: "bg-category-danger/10",
      description: "High GI* & LI"
    },
    {
      label: "Conditional",
      count: categoryCount[2],
      icon: AlertCircle,
      color: "text-category-warning",
      bg: "bg-category-warning/10",
      description: "Context-dependent"
    },
    {
      label: "Safe",
      count: categoryCount[3],
      icon: CheckCircle,
      color: "text-category-safe",
      bg: "bg-category-safe/10",
      description: "Low insulin impact"
    },
    {
      label: "Therapeutic",
      count: categoryCount[4],
      icon: Sparkles,
      color: "text-category-therapeutic",
      bg: "bg-category-therapeutic/10",
      description: "Active fat-burning"
    },
  ];
  
  return (
    <div className="grid grid-cols-2 lg:grid-cols-4 gap-2 sm:gap-3 md:gap-4 w-full">
      {stats.map((stat) => {
        const Icon = stat.icon;
        return (
          <Card key={stat.label} className="shadow-soft hover:shadow-card transition-shadow">
            <CardContent className="p-3 sm:p-4 md:pt-6">
              <div className="flex items-center gap-2 sm:gap-3">
                <div className={`p-1.5 sm:p-2 md:p-2.5 rounded-lg sm:rounded-xl ${stat.bg}`}>
                  <Icon className={`h-4 w-4 sm:h-5 sm:w-5 ${stat.color}`} />
                </div>
                <div className="min-w-0">
                  <p className={`text-lg sm:text-xl md:text-2xl font-display font-bold ${stat.color}`}>{stat.count}</p>
                  <p className="text-xs sm:text-sm font-medium truncate">{stat.label}</p>
                </div>
              </div>
              <p className="text-[10px] sm:text-xs text-muted-foreground mt-1.5 sm:mt-2 line-clamp-1">{stat.description}</p>
            </CardContent>
          </Card>
        );
      })}
    </div>
  );
}
