import { Button } from "@/components/ui/button";
import { Slider } from "@/components/ui/slider";
import { Label } from "@/components/ui/label";
import { 
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { 
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from "@/components/ui/collapsible";
import { 
  AlertTriangle, 
  Skull, 
  Flame, 
  HeartPulse, 
  Zap, 
  Pill,
  ChevronDown,
  SlidersHorizontal,
  ArrowUpDown
} from "lucide-react";
import { useState } from "react";

export interface TobinFilters {
  maxTotal: number;
  maxT: number;
  maxO: number;
  maxB: number;
  maxI: number;
  maxN: number;
}

export type SortOption = 'none' | 'tobin-asc' | 'tobin-desc' | 'name-asc' | 'name-desc';

interface TobinFilterProps {
  filters: TobinFilters;
  onChange: (filters: TobinFilters) => void;
  sortBy: SortOption;
  onSortChange: (sort: SortOption) => void;
}

const defaultFilters: TobinFilters = {
  maxTotal: 50,
  maxT: 10,
  maxO: 10,
  maxB: 10,
  maxI: 10,
  maxN: 10,
};

const categories = [
  { key: 'T', label: 'Toxins', icon: Skull, color: 'text-red-500', description: 'Heavy metals, carcinogens' },
  { key: 'O', label: 'Oxidative', icon: Flame, color: 'text-orange-500', description: 'Free radical damage' },
  { key: 'B', label: 'Biological', icon: HeartPulse, color: 'text-purple-500', description: 'Hormone disruption' },
  { key: 'I', label: 'Inflammatory', icon: Zap, color: 'text-yellow-500', description: 'Chronic inflammation' },
  { key: 'N', label: 'Nutrient', icon: Pill, color: 'text-blue-500', description: 'Nutrient depletion' },
] as const;

function getTobinLabel(value: number): string {
  if (value <= 10) return 'Excellent (0-10)';
  if (value <= 20) return 'Good (0-20)';
  if (value <= 30) return 'Moderate (0-30)';
  if (value <= 40) return 'Concern (0-40)';
  return 'All (0-50)';
}

function getCategoryLabel(value: number): string {
  if (value <= 2) return 'Very Low';
  if (value <= 4) return 'Low';
  if (value <= 6) return 'Moderate';
  if (value <= 8) return 'High';
  return 'All';
}

export function TobinFilter({ filters, onChange, sortBy, onSortChange }: TobinFilterProps) {
  const [isOpen, setIsOpen] = useState(false);
  
  const isFiltered = filters.maxTotal < 50 || 
    filters.maxT < 10 || filters.maxO < 10 || 
    filters.maxB < 10 || filters.maxI < 10 || filters.maxN < 10;
  
  const handleReset = () => {
    onChange(defaultFilters);
    onSortChange('none');
  };
  
  const handleQuickFilter = (maxTotal: number) => {
    onChange({ ...defaultFilters, maxTotal });
  };
  
  return (
    <div className="space-y-2 sm:space-y-3">
      {/* Quick Actions Row */}
      <div className="flex flex-col sm:flex-row flex-wrap items-start sm:items-center gap-2">
        <span className="text-xs sm:text-sm font-medium text-muted-foreground">TOBIN Score:</span>
        
        {/* Quick filter buttons */}
        <div className="flex flex-wrap gap-1">
          <Button
            variant={filters.maxTotal <= 10 ? "default" : "outline"}
            size="sm"
            onClick={() => handleQuickFilter(10)}
            className={`text-xs sm:text-sm h-7 sm:h-8 px-2 sm:px-3 ${filters.maxTotal <= 10 ? "bg-green-600 hover:bg-green-700" : ""}`}
          >
            Excellent
          </Button>
          <Button
            variant={filters.maxTotal > 10 && filters.maxTotal <= 20 ? "default" : "outline"}
            size="sm"
            onClick={() => handleQuickFilter(20)}
            className={`text-xs sm:text-sm h-7 sm:h-8 px-2 sm:px-3 ${filters.maxTotal > 10 && filters.maxTotal <= 20 ? "bg-emerald-600 hover:bg-emerald-700" : ""}`}
          >
            Good
          </Button>
          <Button
            variant={filters.maxTotal > 20 && filters.maxTotal <= 30 ? "default" : "outline"}
            size="sm"
            onClick={() => handleQuickFilter(30)}
            className={`text-xs sm:text-sm h-7 sm:h-8 px-2 sm:px-3 ${filters.maxTotal > 20 && filters.maxTotal <= 30 ? "bg-yellow-600 hover:bg-yellow-700" : ""}`}
          >
            Moderate
          </Button>
          <Button
            variant={filters.maxTotal >= 50 ? "default" : "outline"}
            size="sm"
            onClick={() => handleQuickFilter(50)}
            className="text-xs sm:text-sm h-7 sm:h-8 px-2 sm:px-3"
          >
            All
          </Button>
        </div>
        
        <div className="flex flex-wrap items-center gap-2 w-full sm:w-auto">
          {/* Sort dropdown */}
          <Select value={sortBy} onValueChange={(value) => onSortChange(value as SortOption)}>
            <SelectTrigger className="w-full sm:w-[160px] h-7 sm:h-8 text-xs sm:text-sm">
              <ArrowUpDown className="h-3 w-3 sm:h-3.5 sm:w-3.5 mr-1 sm:mr-1.5" />
              <SelectValue placeholder="Sort by..." />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="none">Default Order</SelectItem>
              <SelectItem value="tobin-asc">TOBIN: Low to High</SelectItem>
              <SelectItem value="tobin-desc">TOBIN: High to Low</SelectItem>
              <SelectItem value="name-asc">Name: A to Z</SelectItem>
              <SelectItem value="name-desc">Name: Z to A</SelectItem>
            </SelectContent>
          </Select>
          
          {/* Advanced toggle */}
          <Collapsible open={isOpen} onOpenChange={setIsOpen}>
            <CollapsibleTrigger asChild>
              <Button variant="ghost" size="sm" className="gap-1 sm:gap-1.5 h-7 sm:h-8 text-xs sm:text-sm px-2 sm:px-3">
                <SlidersHorizontal className="h-3 w-3 sm:h-3.5 sm:w-3.5" />
                <span className="hidden xs:inline">Advanced</span>
                <ChevronDown className={`h-3 w-3 sm:h-3.5 sm:w-3.5 transition-transform ${isOpen ? 'rotate-180' : ''}`} />
              </Button>
            </CollapsibleTrigger>
          </Collapsible>
          
          {isFiltered && (
            <Button variant="ghost" size="sm" onClick={handleReset} className="text-muted-foreground h-7 sm:h-8 text-xs sm:text-sm px-2 sm:px-3">
              Reset
            </Button>
          )}
        </div>
      </div>
      
      {/* Advanced Filters Panel */}
      <Collapsible open={isOpen} onOpenChange={setIsOpen}>
        <CollapsibleContent>
          <div className="bg-card border rounded-lg p-4 space-y-4">
            {/* Total TOBIN Score Slider */}
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <Label className="flex items-center gap-2">
                  <AlertTriangle className="h-4 w-4 text-primary" />
                  Max Total TOBIN Score
                </Label>
                <span className="text-sm font-medium">{getTobinLabel(filters.maxTotal)}</span>
              </div>
              <Slider
                value={[filters.maxTotal]}
                onValueChange={([value]) => onChange({ ...filters, maxTotal: value })}
                max={50}
                min={0}
                step={5}
                className="w-full"
              />
              <div className="flex justify-between text-xs text-muted-foreground">
                <span>0 (Best)</span>
                <span>50 (Worst)</span>
              </div>
            </div>
            
            {/* Individual Category Sliders */}
            <div className="grid sm:grid-cols-2 lg:grid-cols-5 gap-4 pt-2 border-t">
              {categories.map(({ key, label, icon: Icon, color, description }) => (
                <div key={key} className="space-y-2">
                  <div className="flex items-center justify-between">
                    <Label className="flex items-center gap-1.5 text-xs">
                      <Icon className={`h-3.5 w-3.5 ${color}`} />
                      {label}
                    </Label>
                    <span className="text-xs text-muted-foreground">≤{filters[`max${key}` as keyof TobinFilters]}</span>
                  </div>
                  <Slider
                    value={[filters[`max${key}` as keyof TobinFilters]]}
                    onValueChange={([value]) => onChange({ ...filters, [`max${key}`]: value })}
                    max={10}
                    min={0}
                    step={1}
                    className="w-full"
                  />
                  <p className="text-[10px] text-muted-foreground">{description}</p>
                </div>
              ))}
            </div>
          </div>
        </CollapsibleContent>
      </Collapsible>
    </div>
  );
}

export { defaultFilters as defaultTobinFilters };
