import { TobinScore, getTobinRating } from "@/data/foodDatabase";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { 
  Skull, 
  Flame, 
  HeartPulse, 
  Zap, 
  Pill,
  AlertTriangle,
  CheckCircle,
  Info,
  Lightbulb
} from "lucide-react";
import TobinAlert from "./TobinAlert";

interface TobinScoreDisplayProps {
  tobinScore: TobinScore;
  showRecommendations?: boolean;
}

const categories = [
  { 
    key: 'T' as const, 
    label: 'Toxins', 
    fullLabel: 'Toxins & Carcinogens',
    icon: Skull, 
    color: 'text-red-500',
    bgColor: 'bg-red-500',
    description: 'Heavy metals, pesticides, carcinogens, environmental pollutants',
    lowTip: 'Low toxin exposure - continue choosing clean sources',
    highTip: 'Consider organic alternatives to reduce toxin load'
  },
  { 
    key: 'O' as const, 
    label: 'Oxidative', 
    fullLabel: 'Oxidative Stress',
    icon: Flame, 
    color: 'text-orange-500',
    bgColor: 'bg-orange-500',
    description: 'Free radical generation, lipid peroxidation, AGE formation',
    lowTip: 'Good antioxidant profile - supports cellular health',
    highTip: 'Pair with antioxidant-rich foods like berries or green tea'
  },
  { 
    key: 'B' as const, 
    label: 'Biological', 
    fullLabel: 'Biological/Hormonal',
    icon: HeartPulse, 
    color: 'text-purple-500',
    bgColor: 'bg-purple-500',
    description: 'Endocrine disruption, xenoestrogens, hormone mimics',
    lowTip: 'Minimal hormonal impact - safe for regular use',
    highTip: 'Limit exposure, especially for children and during pregnancy'
  },
  { 
    key: 'I' as const, 
    label: 'Inflammatory', 
    fullLabel: 'Inflammatory Potential',
    icon: Zap, 
    color: 'text-yellow-600',
    bgColor: 'bg-yellow-500',
    description: 'Pro-inflammatory compounds, gut irritation, immune activation',
    lowTip: 'Anti-inflammatory or neutral - supports immune balance',
    highTip: 'Balance with omega-3 rich foods and anti-inflammatory herbs'
  },
  { 
    key: 'N' as const, 
    label: 'Nutrient', 
    fullLabel: 'Nutrient Depletion',
    icon: Pill, 
    color: 'text-blue-500',
    bgColor: 'bg-blue-500',
    description: 'Depletes vitamins, minerals, or interferes with nutrient absorption',
    lowTip: 'Nutrient-dense or neutral - supports nutritional status',
    highTip: 'Ensure adequate B vitamins, magnesium, and mineral intake'
  },
] as const;

function getScoreColor(score: number): string {
  if (score <= 2) return 'text-green-600';
  if (score <= 4) return 'text-emerald-600';
  if (score <= 6) return 'text-yellow-600';
  if (score <= 8) return 'text-orange-600';
  return 'text-red-600';
}

function getProgressColor(score: number): string {
  if (score <= 2) return 'bg-green-500';
  if (score <= 4) return 'bg-emerald-500';
  if (score <= 6) return 'bg-yellow-500';
  if (score <= 8) return 'bg-orange-500';
  return 'bg-red-500';
}

function getRecommendations(tobinScore: TobinScore): string[] {
  const recommendations: string[] = [];
  
  if (tobinScore.T >= 5) {
    recommendations.push("Choose organic versions when available to reduce toxin exposure");
  }
  if (tobinScore.O >= 5) {
    recommendations.push("Consume with antioxidants (vitamin C, E, polyphenols) to offset oxidative stress");
  }
  if (tobinScore.B >= 5) {
    recommendations.push("Limit frequency of use; consider hormone-free alternatives");
  }
  if (tobinScore.I >= 5) {
    recommendations.push("Balance with anti-inflammatory foods like fatty fish, turmeric, or ginger");
  }
  if (tobinScore.N >= 5) {
    recommendations.push("Supplement with depleted nutrients or consume nutrient-dense foods alongside");
  }
  
  if (tobinScore.total <= 10) {
    recommendations.push("Excellent choice! This item supports overall health with minimal concerns");
  } else if (tobinScore.total <= 20) {
    recommendations.push("Good choice for regular use with few precautions needed");
  } else if (tobinScore.total >= 35) {
    recommendations.push("Consider reducing frequency or finding safer alternatives");
  }
  
  return recommendations;
}

export function TobinScoreDisplay({ tobinScore, showRecommendations = true }: TobinScoreDisplayProps) {
  const rating = getTobinRating(tobinScore.total);
  const recommendations = getRecommendations(tobinScore);
  
  // Convert TobinScore to format for alerts
  const alertScore = {
    total: tobinScore.total,
    toxins: tobinScore.T,
    oxidative: tobinScore.O,
    biological: tobinScore.B,
    inflammatory: tobinScore.I,
    nutrient: tobinScore.N,
  };
  
  return (
    <div className="space-y-4">
      {/* Personalized Health Alert */}
      <TobinAlert tobinScore={alertScore} />
      
      {/* Overall Score Header */}
      <div className="flex items-center justify-between p-4 rounded-lg bg-gradient-to-r from-primary/10 to-primary/5 border">
        <div className="flex items-center gap-3">
          <AlertTriangle className="h-6 w-6 text-primary" />
          <div>
            <h4 className="font-semibold">TOBIN Health Score</h4>
            <p className="text-xs text-muted-foreground">
              Toxins • Oxidative • Biological • Inflammatory • Nutrient
            </p>
          </div>
        </div>
        <div className="text-right">
          <div className={`text-3xl font-display font-bold ${rating.color}`}>
            {tobinScore.total}/50
          </div>
          <Badge className={`${rating.bgColor} ${rating.color.replace('text-', 'text-')} border-0`}>
            {rating.label}
          </Badge>
        </div>
      </div>
      
      {/* Category Breakdown */}
      <div className="grid gap-3">
        {categories.map(({ key, label, fullLabel, icon: Icon, color, bgColor, description }) => {
          const score = tobinScore[key];
          const progressValue = (score / 10) * 100;
          
          return (
            <div key={key} className="p-3 rounded-lg bg-muted/50 border">
              <div className="flex items-center justify-between mb-2">
                <div className="flex items-center gap-2">
                  <Icon className={`h-4 w-4 ${color}`} />
                  <span className="font-medium text-sm">{fullLabel}</span>
                </div>
                <span className={`font-bold ${getScoreColor(score)}`}>{score}/10</span>
              </div>
              
              {/* Progress Bar */}
              <div className="h-2 bg-muted rounded-full overflow-hidden mb-2">
                <div 
                  className={`h-full rounded-full transition-all duration-500 ${getProgressColor(score)}`}
                  style={{ width: `${progressValue}%` }}
                />
              </div>
              
              <p className="text-xs text-muted-foreground">{description}</p>
            </div>
          );
        })}
      </div>
      
      {/* Notes */}
      {tobinScore.notes && (
        <div className="p-3 rounded-lg bg-blue-50 border border-blue-200 dark:bg-blue-950/20 dark:border-blue-800">
          <div className="flex items-start gap-2">
            <Info className="h-4 w-4 text-blue-600 flex-shrink-0 mt-0.5" />
            <p className="text-sm text-blue-800 dark:text-blue-200">{tobinScore.notes}</p>
          </div>
        </div>
      )}
      
      {/* Recommendations */}
      {showRecommendations && recommendations.length > 0 && (
        <div className="p-4 rounded-lg bg-primary/5 border border-primary/20">
          <div className="flex items-start gap-3">
            <Lightbulb className="h-5 w-5 text-primary flex-shrink-0 mt-0.5" />
            <div>
              <h4 className="font-semibold mb-2">Health Recommendations</h4>
              <ul className="space-y-2">
                {recommendations.map((rec, i) => (
                  <li key={i} className="flex items-start gap-2 text-sm">
                    <CheckCircle className="h-4 w-4 text-primary flex-shrink-0 mt-0.5" />
                    <span>{rec}</span>
                  </li>
                ))}
              </ul>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
