import { useState, useMemo } from 'react';
import { Link } from 'react-router-dom';
import { useAuth } from '@/hooks/useAuth';
import { useFavorites } from '@/contexts/FavoritesContext';
import { useShoppingList } from '@/contexts/ShoppingListContext';
import { useScanHistory, ScanHistoryItem } from '@/hooks/useScanHistory';
import { foodDatabase, FoodItem } from '@/data/foodDatabase';
import { FoodCard } from '@/components/FoodCard';
import { ScannedProductMiniCard, ScannedProduct, getScannedProductKey } from '@/components/ScannedProductMiniCard';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Heart, ShoppingCart, History, Star, ArrowRight, ScanLine, Sparkles, CheckCircle2 } from 'lucide-react';
import { FoodDetailModal } from '@/components/FoodDetailModal';
import { ScannedProductDetailModal } from '@/components/ScannedProductDetailModal';
import { DietaryFilter, type DietaryFilters } from '@/components/DietaryFilter';

const defaultDietaryFilters: DietaryFilters = {
  halal: 'all',
  kosher: 'all',
  organic: 'all',
  phytoestrogen: 'all',
};

export function UserDashboard() {
  const { user } = useAuth();
  const { favorites, loading: favoritesLoading } = useFavorites();
  const { items: shoppingItems, loading: shoppingLoading } = useShoppingList();
  const { history: scanHistory, loading: scanLoading } = useScanHistory();
  
  const [selectedFood, setSelectedFood] = useState<FoodItem | null>(null);
  const [selectedScannedProduct, setSelectedScannedProduct] = useState<ScanHistoryItem | null>(null);
  const [dietaryFilters, setDietaryFilters] = useState<DietaryFilters>(defaultDietaryFilters);

  // Filter for food products only
  const foodFavorites = useMemo(() => 
    favorites.filter(f => f.item_type === 'food'), [favorites]);
  
  const foodShopping = useMemo(() => 
    shoppingItems.filter(i => i.item_type === 'food'), [shoppingItems]);
  
  const foodScans = useMemo(() => 
    scanHistory.filter(s => s.product_type === 'food'), [scanHistory]);

  // Helper to find product in local DB or scan history
  const resolveProduct = useMemo(() => {
    return (itemId: string, quantity?: number) => {
      // Try local food database first
      const localFood = foodDatabase.find(f => f.id === itemId);
      if (localFood) return { type: 'local-food' as const, item: localFood, quantity };
      
      // Check ALL scan history for external products (by barcode or product_id)
      // Filter for food products when found
      const scannedProduct = scanHistory.find(s => 
        (s.barcode === itemId || s.product_id === itemId) && s.product_type === 'food'
      );
      if (scannedProduct) {
        return {
          type: 'scanned' as const,
          item: {
            id: scannedProduct.product_id,
            product_name: scannedProduct.product_name,
            product_type: 'food' as const,
            image_url: scannedProduct.image_url,
            barcode: scannedProduct.barcode,
            isScanned: true as const,
            quantity,
            scanHistoryItem: scannedProduct
          } satisfies ScannedProduct
        };
      }
      
      return null;
    };
  }, [scanHistory]);

  if (!user) {
    return null;
  }

  // Get favorite foods
  const resolvedFavorites = foodFavorites.map(f => resolveProduct(f.item_id)).filter(Boolean);
  const favoriteLocalFoodsAll = resolvedFavorites.filter(r => r?.type === 'local-food').map(r => r!.item as FoodItem);
  const favoriteScannedProductsAll = resolvedFavorites.filter(r => r?.type === 'scanned').map(r => r!.item as ScannedProduct);

  // Get shopping list items - To Buy
  const toBuyItems = foodShopping.filter(i => !i.is_purchased);
  const resolvedShopping = toBuyItems.map(i => resolveProduct(i.item_id, i.quantity)).filter(Boolean);
  const shoppingLocalFoodsAll = resolvedShopping.filter(r => r?.type === 'local-food').map(r => ({ ...r!.item as FoodItem, quantity: r!.quantity || 1 }));
  const shoppingScannedProductsAll = resolvedShopping.filter(r => r?.type === 'scanned').map(r => r!.item as ScannedProduct);

  // Get purchased items
  const purchasedItems = foodShopping.filter(i => i.is_purchased);
  const resolvedPurchased = purchasedItems.map(i => resolveProduct(i.item_id, i.quantity)).filter(Boolean);
  const purchasedLocalFoodsAll = resolvedPurchased.filter(r => r?.type === 'local-food').map(r => ({ ...r!.item as FoodItem, quantity: r!.quantity || 1 }));
  const purchasedScannedProductsAll = resolvedPurchased.filter(r => r?.type === 'scanned').map(r => r!.item as ScannedProduct);

  const recentScans = foodScans.slice(0, 12);
  const recentScansWithLocal = recentScans.map(scan => {
    const food = foodDatabase.find(f => f.id === scan.product_id);
    return { item: food, scannedAt: scan.scanned_at, scanData: scan };
  });

  // Dietary filter helpers
  const matchesDietaryForFood = useMemo(() => {
    return (food: FoodItem) => {
      const matchesHalal = dietaryFilters.halal === 'all' || food.dietaryClassification?.halal === dietaryFilters.halal;
      const matchesKosher = dietaryFilters.kosher === 'all' || food.dietaryClassification?.kosher === dietaryFilters.kosher;
      const matchesOrganic = dietaryFilters.organic === 'all' || food.dietaryClassification?.organic === dietaryFilters.organic;
      let matchesPhyto = true;
      if (dietaryFilters.phytoestrogen === 'hide') {
        matchesPhyto = !food.estrogenicProfile?.hasPhytoestrogens;
      } else if (dietaryFilters.phytoestrogen === 'low_only') {
        matchesPhyto = !food.estrogenicProfile?.hasPhytoestrogens || (food.estrogenicProfile?.estrogenicPotency === 'low' || food.estrogenicProfile?.estrogenicPotency === 'none');
      } else if (dietaryFilters.phytoestrogen === 'show_estrogenic') {
        matchesPhyto = food.estrogenicProfile?.hasPhytoestrogens === true;
      }
      return matchesHalal && matchesKosher && matchesOrganic && matchesPhyto;
    };
  }, [dietaryFilters]);

  const matchesDietaryForScan = useMemo(() => {
    return (item: ScanHistoryItem) => {
      const details = item.product_details;
      const halal = details?.dietaryClassification?.halal ?? details?.aiAnalysis?.halal;
      const kosher = details?.dietaryClassification?.kosher ?? details?.aiAnalysis?.kosher;
      const organic = details?.dietaryClassification?.organic ?? details?.aiAnalysis?.organic;
      const matchesHalal = dietaryFilters.halal === 'all' || halal === dietaryFilters.halal;
      const matchesKosher = dietaryFilters.kosher === 'all' || kosher === dietaryFilters.kosher;
      const matchesOrganic = dietaryFilters.organic === 'all' || organic === dietaryFilters.organic;
      return matchesHalal && matchesKosher && matchesOrganic;
    };
  }, [dietaryFilters]);

  // Apply dietary filters to displayed lists
  const favoriteLocalFoods = useMemo(() => favoriteLocalFoodsAll.filter(matchesDietaryForFood), [favoriteLocalFoodsAll, matchesDietaryForFood]);
  const favoriteScannedProducts = useMemo(() => favoriteScannedProductsAll.filter(p => p.scanHistoryItem && matchesDietaryForScan(p.scanHistoryItem)), [favoriteScannedProductsAll, matchesDietaryForScan]);
  const shoppingLocalFoods = useMemo(() => shoppingLocalFoodsAll.filter(f => matchesDietaryForFood(f)), [shoppingLocalFoodsAll, matchesDietaryForFood]);
  const shoppingScannedProducts = useMemo(() => shoppingScannedProductsAll.filter(p => p.scanHistoryItem && matchesDietaryForScan(p.scanHistoryItem)), [shoppingScannedProductsAll, matchesDietaryForScan]);
  const purchasedLocalFoods = useMemo(() => purchasedLocalFoodsAll.filter(f => matchesDietaryForFood(f)), [purchasedLocalFoodsAll, matchesDietaryForFood]);
  const purchasedScannedProducts = useMemo(() => purchasedScannedProductsAll.filter(p => p.scanHistoryItem && matchesDietaryForScan(p.scanHistoryItem)), [purchasedScannedProductsAll, matchesDietaryForScan]);

  const recentScansFiltered = useMemo(() => {
    return recentScansWithLocal.filter(s => {
      if (s.item) return matchesDietaryForFood(s.item);
      return matchesDietaryForScan(s.scanData);
    });
  }, [recentScansWithLocal, matchesDietaryForFood, matchesDietaryForScan]);
  const recentScansLocal = recentScansFiltered.filter(s => s.item != null);
  const recentScansExternal = recentScansFiltered.filter(s => s.item == null);

  const totalItems = foodFavorites.length + toBuyItems.length + foodScans.length;

  const handleScannedProductClick = (item: ScanHistoryItem) => {
    setSelectedScannedProduct(item);
  };

  const isLoading = favoritesLoading || shoppingLoading || scanLoading;

  if (isLoading) {
    return (
      <Card className="animate-pulse">
        <CardHeader>
          <div className="h-6 w-48 bg-muted rounded" />
          <div className="h-4 w-32 bg-muted rounded mt-2" />
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            {[1, 2, 3, 4].map(i => (
              <div key={i} className="h-32 bg-muted rounded-lg" />
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }

  if (totalItems === 0) {
    return (
      <Card className="border-dashed border-2 border-primary/20 bg-gradient-to-br from-primary/5 to-transparent">
        <CardContent className="flex flex-col items-center justify-center py-12">
          <div className="rounded-full bg-primary/10 p-4 mb-4">
            <Sparkles className="h-8 w-8 text-primary" />
          </div>
          <h3 className="text-xl font-semibold mb-2">Start Building Your Food Collection</h3>
          <p className="text-muted-foreground text-center mb-6 max-w-md">
            Scan foods, save favorites, and create shopping lists. Your personalized food dashboard will appear here!
          </p>
          <div className="flex flex-wrap gap-3 justify-center">
            <Link to="/barcode-scanner">
              <Button className="gap-2">
                <ScanLine className="h-4 w-4" />
                Scan a Food
              </Button>
            </Link>
            <Link to="/food-comparison">
              <Button variant="outline" className="gap-2">
                <Heart className="h-4 w-4" />
                Compare Foods
              </Button>
            </Link>
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <>
      <Card className="overflow-hidden">
        <CardHeader className="bg-gradient-to-r from-primary/5 via-transparent to-transparent">
          <div className="flex items-center justify-between flex-wrap gap-2">
            <div>
              <CardTitle className="flex items-center gap-2">
                <Star className="h-5 w-5 text-primary" />
                Your Food Collection
              </CardTitle>
              <CardDescription>
                {totalItems} items tracked • Favorites, scans & shopping list
              </CardDescription>
            </div>
            <div className="flex gap-2">
              <Badge variant="secondary" className="gap-1">
                <Heart className="h-3 w-3" /> {foodFavorites.length}
              </Badge>
              <Badge variant="secondary" className="gap-1">
                <ShoppingCart className="h-3 w-3" /> {toBuyItems.length}
              </Badge>
              <Badge variant="secondary" className="gap-1">
                <History className="h-3 w-3" /> {foodScans.length}
              </Badge>
            </div>
          </div>
        </CardHeader>
        <CardContent className="pt-6">
          <DietaryFilter filters={dietaryFilters} onChange={setDietaryFilters} />
          <Tabs defaultValue="favorites" className="w-full mt-6">
            <TabsList className="grid w-full grid-cols-3 mb-6">
              <TabsTrigger value="favorites" className="gap-2">
                <Heart className="h-4 w-4" />
                <span className="hidden sm:inline">Favorites</span>
                <Badge variant="secondary" className="ml-1">{foodFavorites.length}</Badge>
              </TabsTrigger>
              <TabsTrigger value="shopping" className="gap-2">
                <ShoppingCart className="h-4 w-4" />
                <span className="hidden sm:inline">Shopping</span>
                <Badge variant="secondary" className="ml-1">{toBuyItems.length}</Badge>
              </TabsTrigger>
              <TabsTrigger value="scans" className="gap-2">
                <History className="h-4 w-4" />
                <span className="hidden sm:inline">Recent Scans</span>
                <Badge variant="secondary" className="ml-1">{foodScans.length}</Badge>
              </TabsTrigger>
            </TabsList>

            <TabsContent value="favorites" className="space-y-4">
              {foodFavorites.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <Heart className="h-12 w-12 mx-auto mb-3 opacity-30" />
                  <p>No favorite foods yet. Browse products and tap the heart icon to save them!</p>
                </div>
              ) : favoriteLocalFoods.length === 0 && favoriteScannedProducts.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <Heart className="h-12 w-12 mx-auto mb-3 opacity-30" />
                  <p>No favorites match your current dietary filters. Try changing Halal, Kosher, or Organic above.</p>
                </div>
              ) : (
                <>
                  {favoriteLocalFoods.length > 0 && (
                    <div>
                      <h4 className="text-sm font-medium text-muted-foreground mb-3">Reference Foods</h4>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                        {favoriteLocalFoods.slice(0, 4).map(food => (
                          <FoodCard 
                            key={food.id} 
                            food={food} 
                            onClick={() => setSelectedFood(food)}
                          />
                        ))}
                      </div>
                      {favoriteLocalFoods.length > 4 && (
                        <Button variant="ghost" className="mt-3 gap-2">
                          View all {favoriteLocalFoods.length} foods <ArrowRight className="h-4 w-4" />
                        </Button>
                      )}
                    </div>
                  )}
                  {favoriteScannedProducts.length > 0 && (
                    <div className="mt-6">
                      <h4 className="text-sm font-medium text-muted-foreground mb-3">Scanned Foods</h4>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                        {favoriteScannedProducts.slice(0, 4).map(product => (
                          <ScannedProductMiniCard 
                            key={getScannedProductKey(product)} 
                            product={product} 
                            onProductClick={handleScannedProductClick}
                          />
                        ))}
                      </div>
                      {favoriteScannedProducts.length > 4 && (
                        <Button variant="ghost" className="mt-3 gap-2">
                          View all {favoriteScannedProducts.length} scanned foods <ArrowRight className="h-4 w-4" />
                        </Button>
                      )}
                    </div>
                  )}
                </>
              )}
            </TabsContent>

            <TabsContent value="shopping" className="space-y-4">
              {toBuyItems.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <ShoppingCart className="h-12 w-12 mx-auto mb-3 opacity-30" />
                  <p>Your food shopping list is empty. Add items from product pages!</p>
                  <Link to="/shopping-list">
                    <Button variant="outline" className="mt-4">View Shopping List</Button>
                  </Link>
                </div>
              ) : shoppingLocalFoods.length === 0 && shoppingScannedProducts.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <ShoppingCart className="h-12 w-12 mx-auto mb-3 opacity-30" />
                  <p>No shopping list items match your current dietary filters. Try changing the filters above.</p>
                  <Link to="/shopping-list">
                    <Button variant="outline" className="mt-4">View Shopping List</Button>
                  </Link>
                </div>
              ) : (
                <>
                  {shoppingLocalFoods.length > 0 && (
                    <div>
                      <h4 className="text-sm font-medium text-muted-foreground mb-3">Reference Foods to Buy</h4>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                        {shoppingLocalFoods.slice(0, 4).map(food => (
                          <div key={food.id} className="relative">
                            <FoodCard 
                              food={food} 
                              onClick={() => setSelectedFood(food)}
                            />
                            {food.quantity > 1 && (
                              <Badge className="absolute top-2 right-2 bg-primary">
                                ×{food.quantity}
                              </Badge>
                            )}
                          </div>
                        ))}
                      </div>
                    </div>
                  )}
                  {shoppingScannedProducts.length > 0 && (
                    <div className="mt-6">
                      <h4 className="text-sm font-medium text-muted-foreground mb-3">Scanned Foods to Buy</h4>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                        {shoppingScannedProducts.slice(0, 4).map(product => (
                          <ScannedProductMiniCard 
                            key={getScannedProductKey(product)} 
                            product={product} 
                            showQuantity 
                            onProductClick={handleScannedProductClick}
                          />
                        ))}
                      </div>
                      {shoppingScannedProducts.length > 4 && (
                        <Link to="/shopping-list">
                          <Button variant="ghost" className="mt-3 gap-2">
                            View all {shoppingScannedProducts.length} scanned foods <ArrowRight className="h-4 w-4" />
                          </Button>
                        </Link>
                      )}
                    </div>
                  )}
                  <div className="flex justify-center mt-4">
                    <Link to="/shopping-list">
                      <Button variant="outline" className="gap-2">
                        <ShoppingCart className="h-4 w-4" />
                        View Full Shopping List
                      </Button>
                    </Link>
                  </div>
                </>
              )}
              
              {/* Purchased Section */}
              {purchasedItems.length > 0 && (
                <div className="mt-8 pt-6 border-t">
                  <div className="flex items-center gap-2 mb-4">
                    <CheckCircle2 className="h-5 w-5 text-category-safe" />
                    <h3 className="font-medium">Recently Purchased</h3>
                    <Badge variant="secondary" className="ml-auto">{purchasedItems.length}</Badge>
                  </div>
                  
                  <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                    {purchasedLocalFoods.slice(0, 2).map(food => (
                      <div key={food.id} className="relative opacity-75">
                        <FoodCard 
                          food={food} 
                          onClick={() => setSelectedFood(food)}
                        />
                        <Badge className="absolute top-2 right-2 bg-category-safe gap-1">
                          <CheckCircle2 className="h-3 w-3" />
                          Purchased
                        </Badge>
                      </div>
                    ))}
                    {purchasedScannedProducts.slice(0, 2).map(product => (
                      <div key={getScannedProductKey(product)} className="relative opacity-75">
                        <ScannedProductMiniCard 
                          product={product} 
                          onProductClick={handleScannedProductClick}
                        />
                      </div>
                    ))}
                  </div>
                  
                  {purchasedItems.length > 4 && (
                    <Link to="/shopping-list">
                      <Button variant="ghost" className="mt-3 gap-2 text-muted-foreground">
                        View all {purchasedItems.length} purchased items <ArrowRight className="h-4 w-4" />
                      </Button>
                    </Link>
                  )}
                </div>
              )}
            </TabsContent>

            <TabsContent value="scans" className="space-y-4">
              {foodScans.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <ScanLine className="h-12 w-12 mx-auto mb-3 opacity-30" />
                  <p>No food scans yet. Use the barcode scanner to analyze products!</p>
                  <Link to="/barcode-scanner">
                    <Button variant="outline" className="mt-4 gap-2">
                      <ScanLine className="h-4 w-4" />
                      Scan a Food
                    </Button>
                  </Link>
                </div>
              ) : recentScansLocal.length === 0 && recentScansExternal.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <ScanLine className="h-12 w-12 mx-auto mb-3 opacity-30" />
                  <p>No recent scans match your current dietary filters. Try changing the filters above.</p>
                  <Link to="/barcode-scanner">
                    <Button variant="outline" className="mt-4 gap-2">
                      <ScanLine className="h-4 w-4" />
                      Scan a Food
                    </Button>
                  </Link>
                </div>
              ) : (
                <>
                  {recentScansLocal.length > 0 && (
                    <div>
                      <h4 className="text-sm font-medium text-muted-foreground mb-3">Reference Foods (from scans)</h4>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                        {recentScansLocal.map((scan) => (
                          <FoodCard
                            key={`food-${scan.scanData.barcode}`}
                            food={scan.item!}
                            onClick={() => setSelectedFood(scan.item!)}
                          />
                        ))}
                      </div>
                    </div>
                  )}
                  {recentScansExternal.length > 0 && (
                    <div className={recentScansLocal.length > 0 ? 'mt-6' : ''}>
                      <h4 className="text-sm font-medium text-muted-foreground mb-3">Scanned Products</h4>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3">
                        {recentScansExternal.map((scan) => (
                          <ScannedProductMiniCard
                            key={scan.scanData.barcode}
                            product={{
                              id: scan.scanData.product_id,
                              product_name: scan.scanData.product_name,
                              product_type: scan.scanData.product_type,
                              image_url: scan.scanData.image_url,
                              barcode: scan.scanData.barcode,
                              isScanned: true,
                              scanHistoryItem: scan.scanData
                            }}
                            onProductClick={handleScannedProductClick}
                          />
                        ))}
                      </div>
                    </div>
                  )}
                  {foodScans.length > 12 && (
                    <div className="flex justify-center mt-4">
                      <Link to="/barcode-scanner">
                        <Button variant="ghost" className="gap-2">
                          View all {foodScans.length} scans <ArrowRight className="h-4 w-4" />
                        </Button>
                      </Link>
                    </div>
                  )}
                </>
              )}
            </TabsContent>
          </Tabs>
        </CardContent>
      </Card>

      {/* Modals */}
      <FoodDetailModal
        food={selectedFood}
        open={!!selectedFood}
        onClose={() => setSelectedFood(null)}
      />
      <ScannedProductDetailModal
        item={selectedScannedProduct}
        open={!!selectedScannedProduct}
        onClose={() => setSelectedScannedProduct(null)}
      />
    </>
  );
}
