import { Link } from 'react-router-dom';
import { useAuth } from '@/hooks/useAuth';
import { useSubscription } from '@/hooks/useSubscription';
import { Button } from '@/components/ui/button';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { User, Heart, Settings, LogOut, LogIn, ShoppingCart, Trophy, Activity, Users, Crown, Sparkles } from 'lucide-react';
import { NotificationBell } from './NotificationBell';
import { MessagesDrawer } from './MessagesDrawer';

export const UserMenu = () => {
  const { user, loading, signOut } = useAuth();
  const { subscribed, is_trial } = useSubscription();

  if (loading) {
    return (
      <div className="flex items-center gap-2">
        <Button variant="ghost" size="icon" disabled>
          <User className="h-5 w-5" />
        </Button>
      </div>
    );
  }

  if (!user) {
    return (
      <Link to="/auth">
        <Button variant="outline" size="sm" className="gap-2">
          <LogIn className="h-4 w-4" />
          Sign In
        </Button>
      </Link>
    );
  }

  return (
    <div className="flex items-center gap-1">
      {/* Subscription Button */}
      {!subscribed ? (
        <Link to="/subscription">
          <Button size="sm" className="gap-1.5 hidden sm:flex">
            <Sparkles className="h-4 w-4" />
            <span>Upgrade</span>
          </Button>
          <Button size="icon" variant="default" className="sm:hidden">
            <Sparkles className="h-4 w-4" />
          </Button>
        </Link>
      ) : (
        <Link to="/subscription">
          <Button size="sm" variant="outline" className="gap-1.5 border-primary/50 text-primary hidden sm:flex">
            <Crown className="h-4 w-4" />
            <span>{is_trial ? 'Trial' : 'Premium'}</span>
          </Button>
          <Button size="icon" variant="outline" className="border-primary/50 text-primary sm:hidden">
            <Crown className="h-4 w-4" />
          </Button>
        </Link>
      )}
      
      <MessagesDrawer />
      <NotificationBell />
      
      <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button variant="ghost" size="icon" className="relative">
            <User className="h-5 w-5" />
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent align="end" className="w-48">
          <div className="px-2 py-1.5 text-sm text-muted-foreground truncate">
            {user.email}
          </div>
          <DropdownMenuSeparator />
          <DropdownMenuItem asChild>
            <Link to="/shopping-list" className="cursor-pointer">
              <ShoppingCart className="mr-2 h-4 w-4" />
              Shopping List
            </Link>
          </DropdownMenuItem>
          <DropdownMenuItem asChild>
            <Link to="/tobin-tracker" className="cursor-pointer">
              <Activity className="mr-2 h-4 w-4" />
              Exposure Tracker
            </Link>
          </DropdownMenuItem>
          <DropdownMenuItem asChild>
            <Link to="/achievements" className="cursor-pointer">
              <Trophy className="mr-2 h-4 w-4" />
              Achievements
            </Link>
          </DropdownMenuItem>
          <DropdownMenuItem asChild>
            <Link to="/community" className="cursor-pointer">
              <Users className="mr-2 h-4 w-4" />
              Community
            </Link>
          </DropdownMenuItem>
          <DropdownMenuItem asChild>
            <Link to="/profile" className="cursor-pointer">
              <Heart className="mr-2 h-4 w-4" />
              My Favorites
            </Link>
          </DropdownMenuItem>
          <DropdownMenuItem asChild>
            <Link to="/profile" className="cursor-pointer">
              <Settings className="mr-2 h-4 w-4" />
              Preferences
            </Link>
          </DropdownMenuItem>
          <DropdownMenuSeparator />
          <DropdownMenuItem onClick={() => signOut()} className="cursor-pointer text-red-600">
            <LogOut className="mr-2 h-4 w-4" />
            Sign Out
          </DropdownMenuItem>
        </DropdownMenuContent>
      </DropdownMenu>
    </div>
  );
};
