import { useEffect, useRef } from 'react';
import { useWebBarcodeScanner } from '@/hooks/useWebBarcodeScanner';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Camera, X, AlertTriangle } from 'lucide-react';

interface WebBarcodeScannerProps {
  onScan: (code: string) => void;
  onClose?: () => void;
}

export function WebBarcodeScanner({ onScan, onClose }: WebBarcodeScannerProps) {
  const {
    isScanning,
    startScan,
    stopScan,
    scannerElementId,
    apiSupported,
  } = useWebBarcodeScanner();

  const hasStartedRef = useRef(false);

  const handleStartScan = () => {
    startScan(onScan);
  };

  const handleStopScan = () => {
    stopScan();
    onClose?.();
  };

  // Auto-start scanning when component mounts
  useEffect(() => {
    if (!hasStartedRef.current && !isScanning && apiSupported !== false) {
      hasStartedRef.current = true;
      // Small delay to ensure DOM is ready
      setTimeout(() => {
        handleStartScan();
      }, 200);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [apiSupported]);

  // Show unsupported message
  if (apiSupported === false) {
    return (
      <Card className="border-destructive">
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between">
            <div>
              <CardTitle className="flex items-center gap-2">
                <AlertTriangle className="h-5 w-5 text-destructive" />
                Browser Not Supported
              </CardTitle>
              <CardDescription>
                Barcode scanning requires a modern browser
              </CardDescription>
            </div>
            {onClose && (
              <Button variant="ghost" size="icon" onClick={onClose}>
                <X className="h-4 w-4" />
              </Button>
            )}
          </div>
        </CardHeader>
        <CardContent className="space-y-4">
          <p className="text-sm text-muted-foreground">
            Your browser doesn't support the native BarcodeDetector API. 
            Please try using <strong>Google Chrome</strong>, <strong>Microsoft Edge</strong>, 
            or <strong>Samsung Internet</strong> on Android for the best experience.
          </p>
          <p className="text-sm text-muted-foreground">
            Alternatively, you can enter barcodes manually using the form below.
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card className="border-primary">
      <CardHeader className="pb-3">
        <div className="flex items-center justify-between">
          <div>
            <CardTitle className="flex items-center gap-2">
              <Camera className="h-5 w-5" />
              Camera Scanner
            </CardTitle>
            <CardDescription>Point your camera at a barcode</CardDescription>
          </div>
          {onClose && (
            <Button variant="ghost" size="icon" onClick={handleStopScan}>
              <X className="h-4 w-4" />
            </Button>
          )}
        </div>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Scanner viewport */}
        <div className="relative overflow-hidden rounded-lg bg-black">
          <div
            id={scannerElementId}
            className="aspect-video w-full"
            style={{ minHeight: '250px' }}
          />
          {isScanning && (
            <div className="pointer-events-none absolute inset-0 flex items-center justify-center">
              <div className="h-20 w-48 rounded border-2 border-primary bg-primary/5">
                <div className="h-full w-full animate-pulse border-2 border-dashed border-primary/50 rounded" />
              </div>
            </div>
          )}
        </div>

        {/* Controls */}
        <div className="flex gap-2">
          {isScanning ? (
            <Button variant="outline" className="w-full" onClick={handleStopScan}>
              Cancel Scan
            </Button>
          ) : (
            <Button className="w-full gap-2" onClick={handleStartScan}>
              <Camera className="h-4 w-4" />
              Start Scanning
            </Button>
          )}
        </div>

        <p className="text-center text-xs text-muted-foreground">
          Hold the barcode steady in good lighting. Using native browser detection for best results.
        </p>
      </CardContent>
    </Card>
  );
}
